"use strict";
// SPDX-FileCopyrightText: 2025 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from matrix-protection-suite
// https://github.com/Gnuxie/matrix-protection-suite
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProjectionOutputHelper = void 0;
const stream_1 = require("stream");
// Technically an orchestration engine needs to do the job of running reducers
// and applying input. Because in order for input to make sense, all of the dependencies
// need to process updates first before we do.
// I'd like to avoid orchestrating via a central engine and instead propagate ULIDs
// recognised revisions for each input. This information needs to live on projections
// for each input projection. And it can be accessed by downstream projections
// and also each downstream projection with a common dependency will want to
// be informed of when so they can do this dance themselves.
// idk progress markers can cause storms, it only matters when there is a
// fork in the tree and a convergence. It can probably be detected and handled
// automatically by PRR. hmm i don't know about that. Are we sure there isn't
// something unsafe about forks in the first place? there is.
// Dependencies like this can only be viewed through other projections.
class ProjectionOutputHelper {
    constructor(currentNode) {
        this.currentNode = currentNode;
        this.outputs = new Set();
        this.emitter = new stream_1.EventEmitter();
        // nothing to do.
    }
    applyInput(input) {
        const previousNode = this.currentNode;
        const delta = previousNode.reduceInput(input);
        this.currentNode = previousNode.reduceDelta(delta);
        for (const output of this.outputs) {
            output.applyInput(delta);
        }
        this.emitter.emit('projection', this.currentNode, delta, previousNode);
    }
    addOutput(projection) {
        this.outputs.add(projection);
        return this;
    }
    removeOutput(projection) {
        this.outputs.delete(projection);
        return this;
    }
    addNodeListener(listener) {
        this.emitter.addListener('projection', listener);
        return this;
    }
    removeNodeListener(listener) {
        this.emitter.removeListener('projection', listener);
        return this;
    }
    [Symbol.dispose]() {
        this.emitter.removeAllListeners();
        this.outputs.clear();
    }
}
exports.ProjectionOutputHelper = ProjectionOutputHelper;
//# sourceMappingURL=Projection.js.map