"use strict";
// Copyright 2022 - 2023 Gnuxie <Gnuxie@protonmail.com>
// Copyright 2019 - 2022 The Matrix.org Foundation C.I.C.
//
// SPDX-License-Identifier: AFL-3.0 AND Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from mjolnir
// https://github.com/matrix-org/mjolnir
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
exports.AccessControl = exports.Access = void 0;
const matrix_basic_types_1 = require("@the-draupnir-project/matrix-basic-types");
const PolicyEvents_1 = require("../MatrixTypes/PolicyEvents");
const PolicyRule_1 = require("../PolicyList/PolicyRule");
var Access;
(function (Access) {
    /// The entity was explicitly banned by a policy list.
    Access[Access["Banned"] = 0] = "Banned";
    /// The entity did not match any allow rule.
    Access[Access["NotAllowed"] = 1] = "NotAllowed";
    /// The user was allowed and didn't match any ban.
    Access[Access["Allowed"] = 2] = "Allowed";
})(Access || (exports.Access = Access = {}));
/**
 * This allows us to work out the access an entity has to some thing based on a set of watched/unwatched lists.
 */
class AccessControl {
    /**
     * Test whether the server is allowed by the ACL unit.
     * @param domain The server name to test.
     * @returns A description of the access that the server has.
     */
    static getAccessForServer(revision, domain) {
        return AccessControl.getAccessForEntity(revision, domain, PolicyEvents_1.PolicyRuleType.Server);
    }
    /**
     * Get the level of access the user has for the ACL unit.
     * @param mxid The user id to test.
     * @param policy Whether to check the server part of the user id against server rules.
     * @returns A description of the access that the user has.
     */
    static getAccessForUser(revision, userID, policy) {
        const userAccess = AccessControl.getAccessForEntity(revision, userID, PolicyEvents_1.PolicyRuleType.User);
        if (policy === 'IGNORE_SERVER' || userAccess.outcome === Access.Banned) {
            return userAccess;
        }
        else {
            const serverAccess = AccessControl.getAccessForEntity(revision, (0, matrix_basic_types_1.userServerName)(userID), PolicyEvents_1.PolicyRuleType.Server);
            if (userAccess.outcome === Access.Allowed &&
                serverAccess.outcome === Access.NotAllowed) {
                return userAccess;
            }
            else {
                return serverAccess;
            }
        }
    }
    static getAccessForEntity(revision, entity, entityType) {
        // Check if the entity is explicitly allowed.
        // We have to infer that a rule exists for '*' if the allowCache is empty, otherwise you brick the ACL.
        const allowRule = revision.findRuleMatchingEntity(entity, {
            type: entityType,
            recommendation: PolicyRule_1.Recommendation.Allow,
            searchHashedRules: false,
        });
        if (allowRule === undefined &&
            // this is gonna be a pita resource wise.
            !(revision.allRulesOfType(entityType, PolicyRule_1.Recommendation.Allow).length === 0)) {
            return { outcome: Access.NotAllowed };
        }
        // Now check if the entity is banned.
        const banRule = revision.findRuleMatchingEntity(entity, {
            type: entityType,
            recommendation: PolicyRule_1.Recommendation.Ban,
            searchHashedRules: true,
        });
        if (banRule !== undefined) {
            return { outcome: Access.Banned, rule: banRule };
        }
        const takedownRule = revision.findRuleMatchingEntity(entity, {
            type: entityType,
            recommendation: PolicyRule_1.Recommendation.Takedown,
            searchHashedRules: true,
        });
        if (takedownRule !== undefined) {
            return { outcome: Access.Banned, rule: takedownRule };
        }
        // If they got to this point, they're allowed!!
        return { outcome: Access.Allowed };
    }
}
exports.AccessControl = AccessControl;
//# sourceMappingURL=AccessControl.js.map