"use strict";
// SPDX-FileCopyrightText: 2025 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: Apache-2.0
var __addDisposableResource = (this && this.__addDisposableResource) || function (env, value, async) {
    if (value !== null && value !== void 0) {
        if (typeof value !== "object" && typeof value !== "function") throw new TypeError("Object expected.");
        var dispose, inner;
        if (async) {
            if (!Symbol.asyncDispose) throw new TypeError("Symbol.asyncDispose is not defined.");
            dispose = value[Symbol.asyncDispose];
        }
        if (dispose === void 0) {
            if (!Symbol.dispose) throw new TypeError("Symbol.dispose is not defined.");
            dispose = value[Symbol.dispose];
            if (async) inner = dispose;
        }
        if (typeof dispose !== "function") throw new TypeError("Object not disposable.");
        if (inner) dispose = function() { try { inner.call(this); } catch (e) { return Promise.reject(e); } };
        env.stack.push({ value: value, dispose: dispose, async: async });
    }
    else if (async) {
        env.stack.push({ async: true });
    }
    return value;
};
var __disposeResources = (this && this.__disposeResources) || (function (SuppressedError) {
    return function (env) {
        function fail(e) {
            env.error = env.hasError ? new SuppressedError(e, env.error, "An error was suppressed during disposal.") : e;
            env.hasError = true;
        }
        function next() {
            while (env.stack.length) {
                var rec = env.stack.pop();
                try {
                    var result = rec.dispose && rec.dispose.call(rec.value);
                    if (rec.async) return Promise.resolve(result).then(next, function(e) { fail(e); return next(); });
                }
                catch (e) {
                    fail(e);
                }
            }
            if (env.hasError) throw env.error;
        }
        return next();
    };
})(typeof SuppressedError === "function" ? SuppressedError : function (error, suppressed, message) {
    var e = new Error(message);
    return e.name = "SuppressedError", e.error = error, e.suppressed = suppressed, e;
});
Object.defineProperty(exports, "__esModule", { value: true });
exports.HandleRegistryDescriptionSemantics = void 0;
const HandleDescription_1 = require("./HandleDescription");
const SemanticType_1 = require("../../Interface/SemanticType");
const typescript_result_1 = require("@gnuxie/typescript-result");
const Lifetime_1 = require("../../Interface/Lifetime");
exports.HandleRegistryDescriptionSemantics = (0, SemanticType_1.SemanticType)('HandleRegistryDescription').Law({
    descriptionBuilding: {
        what: 'When a handle is registered and a new registry is returned, the original registry is not modified.',
        why: 'Keeps the builder pattern clean and prevents bugs in downstream consumers',
        law: "For empty RegistryDescription R and HandleDescription H, calling R.registerHandleDescription(H) => R' results in R.handleDescriptions = [] and R'.handleDescriptions = [H]",
        async check(makeSubject) {
            const registry = (await makeSubject()).expect('Should be able to make the subject');
            const handle = {
                handleName: 'handle',
                dataSourceType: HandleDescription_1.HandleDataSourceType.Context,
                establish: () => (0, typescript_result_1.Ok)(undefined),
            };
            const newRegistry = registry.registerHandleDescription(handle);
            expect(registry.handleDescriptions).toHaveLength(0);
            expect(newRegistry.handleDescriptions).toHaveLength(1);
            expect(newRegistry.handleDescriptions[0]).toBe(handle);
        },
    },
    registryFactory: {
        what: 'A HandleRegistryDescription can produce HandleRegistry instances for a given context.',
        why: 'Keeps the HandleRegistry abstraction clean by binding the context in construction rather than later on',
        law: 'For RegistryDescription R with handle union H, calling R.registryForContext(L, C) produces a HandleRegistry<R,C>',
        async check(makeSubject) {
            const env_1 = { stack: [], error: void 0, hasError: false };
            try {
                let establishCount = 0;
                const contextHandle = {
                    handleName: 'handle',
                    dataSourceType: HandleDescription_1.HandleDataSourceType.Context,
                    establish: () => {
                        establishCount += 1;
                        return (0, typescript_result_1.Ok)(undefined);
                    },
                };
                const registryDescription = (await makeSubject()).expect('Should be able to make the subject');
                const descriptionWithHandle = registryDescription.registerHandleDescription(contextHandle);
                const context = {};
                const lifetime = __addDisposableResource(env_1, new Lifetime_1.StandardLifetime(), true);
                // eslint-disable-next-line @typescript-eslint/no-unused-vars
                const _registry = __addDisposableResource(env_1, descriptionWithHandle
                    .registryForContext(lifetime, context)
                    .expect('Should be able to construct a registry for the context'), true);
                expect(establishCount).toBe(1);
            }
            catch (e_1) {
                env_1.error = e_1;
                env_1.hasError = true;
            }
            finally {
                const result_1 = __disposeResources(env_1);
                if (result_1)
                    await result_1;
            }
        },
    },
});
//# sourceMappingURL=HandleRegistryDescription.js.map