"use strict";
// Copyright 2019 2022 The Matrix.org Foundation C.I.C.
// Copyright 2022 - 2023 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0 AND Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from mjolnir
// https://github.com/matrix-org/mjolnir
// </text>
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.MjolnirProtectedRoomsConfig = void 0;
const Action_1 = require("../../Interface/Action");
const await_lock_1 = __importDefault(require("await-lock"));
const Logger_1 = require("../../Logging/Logger");
const PersistentConfigData_1 = require("../../Config/PersistentConfigData");
const MjolnirProtectedRoomsDescription_1 = require("./MjolnirProtectedRoomsDescription");
const log = new Logger_1.Logger('MjolnirProtectedroomsCofnig');
class MjolnirProtectedRoomsConfig {
    constructor(config, protectedRooms, 
    /**
     * We use this so that we can keep track of the raw data for logging purposes.
     */
    rawData, loggableConfigTracker) {
        this.config = config;
        this.protectedRooms = protectedRooms;
        this.rawData = rawData;
        this.writeLock = new await_lock_1.default();
        loggableConfigTracker.addLoggableConfig(this);
    }
    static async createFromStore(store, resolver, loggableConfigTracker) {
        var _a;
        const config = new PersistentConfigData_1.StandardPersistentConfigData(MjolnirProtectedRoomsDescription_1.MjolnirProtectedRoomsDescription, store);
        const dataResult = await config.requestParsedConfig();
        if ((0, Action_1.isError)(dataResult)) {
            return dataResult.elaborate(`Failed to load ProtectedRoomsConfig when creating ProtectedRoomsConfig`);
        }
        const data = (_a = dataResult.ok) !== null && _a !== void 0 ? _a : config.description.getDefaultConfig();
        const protectedRooms = new Map();
        for (const [i, ref] of data.rooms.entries()) {
            const resolvedRef = await resolver.resolveRoom(ref);
            if ((0, Action_1.isError)(resolvedRef)) {
                log.info(`Current config`, data);
                return await config.reportUseError('Unable to resolve room reference', {
                    path: `/rooms/${i}`,
                    value: ref,
                    cause: resolvedRef.error,
                });
            }
            protectedRooms.set(resolvedRef.ok.toRoomIDOrAlias(), resolvedRef.ok);
        }
        return (0, Action_1.Ok)(new MjolnirProtectedRoomsConfig(config, protectedRooms, data, loggableConfigTracker));
    }
    getProtectedRooms() {
        return [...this.protectedRooms.values()];
    }
    logCurrentConfig() {
        log.info('Current config', this.rawData);
    }
    async addRoom(room) {
        await this.writeLock.acquireAsync();
        try {
            // We would still like to persist the rooms even if the one being added is already there.
            // This is just to make sure the account data is consistent with what's represented in the model.
            // No, I don't know whether this is justified or not.
            const data = {
                rooms: [
                    ...this.protectedRooms.keys(),
                    ...(this.protectedRooms.has(room.toRoomIDOrAlias())
                        ? []
                        : [room.toRoomIDOrAlias()]),
                ],
            };
            const result = await this.config.saveConfig(data);
            if ((0, Action_1.isError)(result)) {
                return result.elaborate(`Failed to add ${room.toPermalink()} to protected rooms set.`);
            }
            this.protectedRooms.set(room.toRoomIDOrAlias(), room);
            this.rawData = data;
            return (0, Action_1.Ok)(undefined);
        }
        finally {
            this.writeLock.release();
        }
    }
    async removeRoom(room) {
        await this.writeLock.acquireAsync();
        try {
            const data = {
                rooms: this.getProtectedRooms()
                    .map((ref) => ref.toRoomIDOrAlias())
                    .filter((roomID) => roomID !== room.toRoomIDOrAlias()),
            };
            const result = await this.config.saveConfig(data);
            if ((0, Action_1.isError)(result)) {
                return result.elaborate(`Failed to remove ${room.toPermalink()} to protected rooms set.`);
            }
            this.protectedRooms.delete(room.toRoomIDOrAlias());
            this.rawData = data;
            return (0, Action_1.Ok)(undefined);
        }
        finally {
            this.writeLock.release();
        }
    }
    async reportUseError(message, room, error) {
        return await this.config.reportUseError(message, {
            path: `/rooms/${this.getProtectedRooms().indexOf(room)}`,
            value: room,
            cause: error,
        });
    }
}
exports.MjolnirProtectedRoomsConfig = MjolnirProtectedRoomsConfig;
//# sourceMappingURL=ProtectedRoomsConfig.js.map