import { ActionResult } from '../Interface/Action';
import { RoomEvent } from '../MatrixTypes/Events';
import { PolicyListRevision } from '../PolicyList/PolicyListRevision';
import { PolicyRuleChange } from '../PolicyList/PolicyRuleChange';
import { EventReport } from '../Reporting/EventReport';
import { MembershipChange } from '../Membership/MembershipChange';
import { RoomMembershipRevision } from '../Membership/MembershipRevision';
import { RoomStateRevision, StateChange } from '../StateTracking/StateRevisionIssuer';
import { ProtectedRoomsSet } from './ProtectedRoomsSet';
import { CapabilityInterfaceSet, CapabilityProviderSet, CapabilitySet, GenericCapabilityDescription } from './Capability/CapabilitySet';
import { PowerLevelPermission } from '../Client/PowerLevelsMirror';
import { MembershipEvent } from '../MatrixTypes/MembershipEvent';
import { MatrixRoomID, StringRoomID } from '@the-draupnir-project/matrix-basic-types';
import { ConfigDescription, UnknownConfig } from '../Config/ConfigDescription';
import { TObject } from '@sinclair/typebox';
import { EDStatic } from '../Interface/Static';
import { SetMembershipDelta, SetMembershipRevision } from '../Membership/SetMembershipRevision';
import { MembershipPolicyRevisionDelta, SetMembershipPolicyRevision } from '../MembershipPolicies/MembershipPolicyRevision';
import { EventWithMixins } from '../SafeMatrixEvents/EventMixinExtraction/EventMixinExtraction';
import { AllocatableLifetime, OwnLifetime } from '../Interface/Lifetime';
import { Result } from '@gnuxie/typescript-result';
import { ExtractProjectionNode, Projection } from '../Projection/Projection';
import { AnyProjectionNode, ExtractDeltaShape } from '../Projection/ProjectionNode';
/**
 * @param description The description for the protection being constructed.
 * @param consequenceProvider The consequence provider that should be used for this protection.
 * @param protectedRoomsSet The protected rooms that the constructed protection will reside within
 * and be informed of events within.
 * @param context This is a client specified argument, for example the draupnir project
 * will use the Draupnir instance that the `ProtectedRoomsSet` resides within here.
 * Not necessary and use should be avoided.
 * @param settings The settings for this protection as fetched from a persistent store or
 * the description's default settings.
 * @param ownerLifetime A lifetime from the owner of the protection.
 */
export type ProtectionFactoryMethod<Context = unknown, TConfigSchema extends TObject = UnknownConfig, TCapabilitySet extends CapabilitySet = CapabilitySet> = (description: ProtectionDescription<Context, TConfigSchema, TCapabilitySet>, lifetime: OwnLifetime<Protection<ProtectionDescription<Context, TConfigSchema, TCapabilitySet>>>, protectedRoomsSet: ProtectedRoomsSet, context: Context, capabilities: TCapabilitySet, settings: EDStatic<TConfigSchema>) => Promise<ActionResult<Protection<ProtectionDescription<Context, TConfigSchema, TCapabilitySet>>>>;
/**
 * This is a description of a protection, which is used
 * to create protections in a factory method dynamically.
 */
export interface ProtectionDescription<Context = unknown, TConfigSchema extends TObject = UnknownConfig, TCapabilitySet extends CapabilitySet = CapabilitySet> {
    readonly name: string;
    readonly description: string;
    readonly capabilities: CapabilityInterfaceSet<TCapabilitySet>;
    readonly factory: ProtectionFactoryMethod<Context, TConfigSchema, TCapabilitySet>;
    readonly protectionSettings: ConfigDescription<TConfigSchema>;
    readonly defaultCapabilities: CapabilityProviderSet<TCapabilitySet>;
}
/**
 * Represents a protection mechanism of sorts. Protections are intended to be
 * event-based (ie: X messages in a period of time, or posting X events).
 *
 * Protections are guaranteed to be run before redaction handlers.
 */
export type Protection<TProtectionDescription, TIntentProjection extends Projection | undefined = undefined> = {
    readonly description: TProtectionDescription;
    readonly requiredEventPermissions: string[];
    readonly requiredStatePermissions: string[];
    readonly requiredPermissions: PowerLevelPermission[];
    handleTimelineEvent?(room: MatrixRoomID, event: RoomEvent): Promise<ActionResult<void>>;
    /**
     * Handle a single timeline event from a protected room.
     * Rather than providing protections with the raw event,
     * media mixins (including text) are extracted from the event
     * for protections. This prevents parsing errors that would
     * enable evasion of protections.
     */
    handleTimelineEventMixins?(room: MatrixRoomID, event: EventWithMixins): void;
    handlePolicyChange?(revision: PolicyListRevision, changes: PolicyRuleChange[]): Promise<ActionResult<void>>;
    handleMembershipChange?(revision: RoomMembershipRevision, changes: MembershipChange[]): Promise<ActionResult<void>>;
    /**
     * Handle a change to the room state within a protected room.
     * @param revision A RoomStateRevision with the current revision of all the room's state.
     * @param changes Any changes since the previous revision.
     */
    handleStateChange?(revision: RoomStateRevision, changes: StateChange[]): Promise<ActionResult<void>>;
    handleEventReport?(report: EventReport): Promise<ActionResult<void>>;
    /**
     * Called when the permission requirements of the protection have been met
     * within a protected room. This includes if the room is a newly protected room.
     */
    handlePermissionRequirementsMet?(room: MatrixRoomID): void;
    /**
     * Handle an invitation, room join, kick, leave etc to a room that is external to the protected rooms set.
     * @param roomID The room the invitation is for.
     * @param event The invitation event itself.
     */
    handleExternalMembership?(roomID: StringRoomID, event: MembershipEvent): void;
    /**
     * This can be used to determine who are new to the overall protected rooms
     * set, and who has left every protected room.
     */
    handleSetMembershipChange?(revision: SetMembershipRevision, delta: SetMembershipDelta): void;
    /**
     * Handle a change to the set of members who have matched policies.
     * So if a user joins the room with a matching policy, this will result in
     * the handle being called. Or a policy being created that matches existing users.
     * Or a policy being rescinded that previously matched users.
     */
    handleSetMembershipPolicyMatchesChange?(revision: SetMembershipPolicyRevision, delta: MembershipPolicyRevisionDelta): void;
    [Symbol.asyncDispose](): Promise<void>;
} & (TIntentProjection extends undefined ? {
    readonly intentProjection?: Projection;
} : {
    /**
     * Used to view the state that the protection intends to make effectual via
     * capabilities.
     */
    readonly intentProjection: TIntentProjection;
}) & (TIntentProjection extends undefined ? {
    handleIntentProjectionNode?(node: AnyProjectionNode, delta: unknown): void;
} : {
    handleIntentProjectionNode?(node: ExtractProjectionNode<TIntentProjection>, delta: ExtractDeltaShape<ExtractProjectionNode<TIntentProjection>>): void;
});
export declare class AbstractProtection<TProtectionDescription> implements Protection<TProtectionDescription> {
    readonly description: TProtectionDescription;
    protected readonly lifetime: OwnLifetime<Protection<TProtectionDescription>>;
    protected readonly capabilitySet: CapabilitySet;
    protected readonly protectedRoomsSet: ProtectedRoomsSet;
    private readonly clientEventPermissions;
    private readonly clientPermissions;
    private readonly clientStatePermissions;
    protected constructor(description: TProtectionDescription, lifetime: OwnLifetime<Protection<TProtectionDescription>>, capabilitySet: CapabilitySet, protectedRoomsSet: ProtectedRoomsSet, permissions: {
        requiredEventPermissions?: string[];
        requiredPermissions?: PowerLevelPermission[];
        requiredStatePermissions?: string[];
    });
    get requiredEventPermissions(): string[];
    get requiredPermissions(): PowerLevelPermission[];
    get requiredStatePermissions(): string[];
    [Symbol.asyncDispose](): Promise<void>;
}
export declare function registerProtection<Context = unknown, TConfigSchema extends TObject = UnknownConfig, TCapabilitySet extends CapabilitySet = CapabilitySet>(description: ProtectionDescription<Context, TConfigSchema, TCapabilitySet>): ProtectionDescription<Context, TConfigSchema, TCapabilitySet>;
export declare function findProtection(name: string): ProtectionDescription | undefined;
export declare function describeProtection<TCapabilitySet extends CapabilitySet = CapabilitySet, Context = unknown, TConfigSchema extends TObject = UnknownConfig>({ name, description, capabilityInterfaces, defaultCapabilities, factory, configSchema, }: {
    name: string;
    description: string;
    factory: ProtectionDescription<Context, TConfigSchema, TCapabilitySet>['factory'];
    capabilityInterfaces: GenericCapabilityDescription<TCapabilitySet>;
    defaultCapabilities: GenericCapabilityDescription<TCapabilitySet>;
    configSchema?: TConfigSchema;
}): ProtectionDescription<Context, TConfigSchema, TCapabilitySet>;
export declare function getAllProtections(): IterableIterator<ProtectionDescription>;
/**
 * Simple compatibility helper to help the migration to lifetimes.
 * @deprecated Use lifetimes directly.
 */
export declare function allocateProtection<T>(lifetime: AllocatableLifetime, protection: T & {
    handleProtectionDisable(): void;
}): Result<T>;
//# sourceMappingURL=Protection.d.ts.map