"use strict";
// SPDX-FileCopyrightText: 2024 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.StandardProtectionCapabilityProviderSetConfig = exports.CapabilityProviderConfig = void 0;
const typescript_result_1 = require("@gnuxie/typescript-result");
const PersistentConfigData_1 = require("../../../Config/PersistentConfigData");
const describeConfig_1 = require("../../../Config/describeConfig");
const typebox_1 = require("@sinclair/typebox");
const CapabilityProvider_1 = require("../../Capability/CapabilityProvider");
const Logger_1 = require("../../../Logging/Logger");
const SchemedMatrixData_1 = require("../../../Interface/SchemedMatrixData");
const log = new Logger_1.Logger('StandardProtectionCapabilityProviderSetConfig');
exports.CapabilityProviderConfig = typebox_1.Type.Object({ [SchemedMatrixData_1.DRAUPNIR_SCHEMA_VERSION_KEY]: typebox_1.Type.Optional(typebox_1.Type.Number()) }, {
    additionalProperties: typebox_1.Type.Object({
        capability_provider_name: typebox_1.Type.String(),
    }),
});
const CapabilityProviderSetConfigDescription = (0, describeConfig_1.describeConfig)({
    schema: exports.CapabilityProviderConfig,
});
class StandardProtectionCapabilityProviderSetConfig {
    constructor(makePersistentConfigBackend, migrationHandler) {
        this.makePersistentConfigBackend = makePersistentConfigBackend;
        this.migrationHandler = migrationHandler;
        // nothing to do mare.
    }
    async storeActivateCapabilityProviderSet(protectionDescription, capabilityproviderSet) {
        const persistentConfigBackend = this.makePersistentConfigBackend(protectionDescription);
        if ((0, typescript_result_1.isError)(persistentConfigBackend)) {
            return persistentConfigBackend;
        }
        const persistentConfigData = new PersistentConfigData_1.StandardPersistentConfigData(CapabilityProviderSetConfigDescription, persistentConfigBackend.ok);
        let config = {};
        for (const [capabilityName, capabilityProvider] of Object.entries(capabilityproviderSet)) {
            config = {
                ...config,
                [capabilityName]: { capability_provider_name: capabilityProvider.name },
            };
        }
        return await persistentConfigData.saveConfig({
            ...config,
            ...(this.migrationHandler === undefined
                ? {}
                : {
                    [SchemedMatrixData_1.DRAUPNIR_SCHEMA_VERSION_KEY]: this.migrationHandler.latestVersion,
                }),
        });
    }
    async getCapabilityProviderSet(protectionDescription) {
        const persistentConfigData = this.makePersistentConfigBackend(protectionDescription);
        if ((0, typescript_result_1.isError)(persistentConfigData)) {
            return persistentConfigData;
        }
        const result = await persistentConfigData.ok.requestUnparsedConfig();
        if ((0, typescript_result_1.isError)(result)) {
            return result;
        }
        if (result.ok === undefined) {
            return (0, typescript_result_1.Ok)(protectionDescription.defaultCapabilities);
        }
        const migrateData = async () => {
            if (this.migrationHandler === undefined) {
                return (0, typescript_result_1.Ok)(result.ok);
            }
            return await this.migrationHandler.migrateData(result.ok);
        };
        const migratedResult = await migrateData();
        if ((0, typescript_result_1.isError)(migratedResult)) {
            return migratedResult;
        }
        const capabilityProviderSet = {
            ...protectionDescription.defaultCapabilities,
        };
        const versionKey = SchemedMatrixData_1.DRAUPNIR_SCHEMA_VERSION_KEY;
        // eslint-disable-next-line @typescript-eslint/no-unused-vars
        const { [versionKey]: _version, ...capabilityConfigs } = migratedResult.ok;
        for (const [capabilityName, providerConfig] of Object.entries(capabilityConfigs)) {
            const providerDescription = (0, CapabilityProvider_1.findCapabilityProvider)(providerConfig.capability_provider_name);
            // drats, this should really be a config use error but it's a bitch because
            // we don't eagerly load all the capability configs to create this config,
            // so it is failing late and bad if we use it here.
            if (providerDescription === undefined) {
                log.error(`Unable to find a capability provider for ${providerConfig.capability_provider_name} in the protection ${protectionDescription.name}, so using the default for the ${capabilityName}`);
                continue;
            }
            capabilityProviderSet[capabilityName] = providerDescription;
        }
        return (0, typescript_result_1.Ok)(capabilityProviderSet);
    }
}
exports.StandardProtectionCapabilityProviderSetConfig = StandardProtectionCapabilityProviderSetConfig;
//# sourceMappingURL=StandardProtectionCapabilityProviderSetConfig.js.map