"use strict";
// Copyright 2022-2023 Gnuxie <Gnuxie@protonmail.com>
// Copyright 2019 2022 The Matrix.org Foundation C.I.C.
//
// SPDX-License-Identifier: AFL-3.0 AND Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from mjolnir
// https://github.com/matrix-org/mjolnir
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
exports.MemberBanSynchronisationProtection = void 0;
const Action_1 = require("../../../Interface/Action");
const Protection_1 = require("../../Protection");
const MembershipChange_1 = require("../../../Membership/MembershipChange");
const PolicyEvents_1 = require("../../../MatrixTypes/PolicyEvents");
const PolicyRule_1 = require("../../../PolicyList/PolicyRule");
const MultipleErrors_1 = require("../../../Interface/MultipleErrors");
require("../../Capability/StandardCapability/UserConsequences"); // need this to load the interface.
require("../../Capability/StandardCapability/StandardUserConsequences"); // need this to load the providers.
const Task_1 = require("../../../Interface/Task");
const MemberBanIntentProjection_1 = require("./MemberBanIntentProjection");
function revisionMatchesWithUserRules(revision) {
    return revision
        .allMembersWithRules()
        .filter((match) => match.policies.some((policy) => policy.kind === PolicyEvents_1.PolicyRuleType.User));
}
class MemberBanSynchronisationProtection extends Protection_1.AbstractProtection {
    constructor(description, lifetime, capabilities, protectedRoomsSet, intentProjection) {
        super(description, lifetime, capabilities, protectedRoomsSet, {});
        this.intentProjection = intentProjection;
        this.userConsequences = capabilities.userConsequences;
    }
    handleSetMembershipPolicyMatchesChange(revision) {
        void (0, Task_1.Task)(this.synchroniseWithRevision(revision));
    }
    // FIXME:
    // This is a legacy handle that needs to remain while we figure out how to
    // put room membership into the the protection's intent projection.
    // Which is a lot of work.
    async handleMembershipChange(revision, changes) {
        var _a;
        // we need access to the policy list issuer from the protected rooms set.
        const directIssuer = this.protectedRoomsSet.watchedPolicyRooms.revisionIssuer;
        // then check the changes against the policies
        const errors = [];
        for (const change of changes) {
            switch (change.membershipChangeType) {
                case MembershipChange_1.MembershipChangeType.Banned:
                case MembershipChange_1.MembershipChangeType.Kicked:
                case MembershipChange_1.MembershipChangeType.Left:
                case MembershipChange_1.MembershipChangeType.NoChange:
                case MembershipChange_1.MembershipChangeType.Unbanned:
                    continue;
            }
            const applicableRules = directIssuer.currentRevision
                .allRulesMatchingEntity(change.userID, {
                type: PolicyEvents_1.PolicyRuleType.User,
                searchHashedRules: false,
            })
                .filter((rule) => rule.recommendation === PolicyRule_1.Recommendation.Ban ||
                rule.recommendation === PolicyRule_1.Recommendation.Takedown);
            const firstRule = applicableRules[0];
            if (firstRule) {
                const result = await this.userConsequences.consequenceForUserInRoom(revision.room.toRoomIDOrAlias(), change.userID, firstRule.recommendation === PolicyRule_1.Recommendation.Takedown
                    ? 'takedown'
                    : (_a = firstRule.reason) !== null && _a !== void 0 ? _a : '<no reason provided>');
                if ((0, Action_1.isError)(result)) {
                    errors.push(result.error);
                }
            }
        }
        if (errors.length > 1) {
            return MultipleErrors_1.MultipleErrors.Result(`There were errors when banning members in ${revision.room.toPermalink()}`, { errors });
        }
        else {
            return (0, Action_1.Ok)(undefined);
        }
    }
    async synchroniseWithRevision(revision) {
        const result = await this.userConsequences.consequenceForUsersInRoomSet(revisionMatchesWithUserRules(revision));
        if ((0, Action_1.isError)(result)) {
            return result;
        }
        else {
            return (0, Action_1.Ok)(undefined);
        }
    }
    handlePermissionRequirementsMet(room) {
        void (0, Task_1.Task)((async () => {
            await this.userConsequences.consequenceForUsersInRoom(room.toRoomIDOrAlias(), revisionMatchesWithUserRules(this.protectedRoomsSet.setPoliciesMatchingMembership.currentRevision));
        })());
    }
}
exports.MemberBanSynchronisationProtection = MemberBanSynchronisationProtection;
(0, Protection_1.describeProtection)({
    name: 'MemberBanSynchronisationProtection',
    description: 'Synchronises `m.ban` events from watch policy lists with room level bans.',
    capabilityInterfaces: {
        userConsequences: 'UserConsequences',
    },
    defaultCapabilities: {
        userConsequences: 'StandardUserConsequences',
    },
    factory: async (decription, lifetime, protectedRoomsSet, _settings, capabilitySet) => {
        const intentProjection = lifetime.allocateDisposable(() => (0, Action_1.Ok)(new MemberBanIntentProjection_1.StandardMemberBanIntentProjection(protectedRoomsSet.setPoliciesMatchingMembership)));
        if ((0, Action_1.isError)(intentProjection)) {
            return intentProjection.elaborate('Unable to allocate intent projection');
        }
        return (0, Action_1.Ok)(new MemberBanSynchronisationProtection(decription, lifetime, capabilitySet, protectedRoomsSet, intentProjection.ok));
    },
});
//# sourceMappingURL=MemberBanSynchronisation.js.map