"use strict";
// SPDX-FileCopyrightText: 2023-2024 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const Protection_1 = require("../../Protection");
require("./MemberBanSynchronisation");
const Action_1 = require("../../../Interface/Action");
const EventGeneration_1 = require("../../../TestUtilities/EventGeneration");
const DeclareRoomState_1 = require("../../../StateTracking/DeclareRoomState");
const MembershipChange_1 = require("../../../Membership/MembershipChange");
const wait_for_expect_1 = __importDefault(require("wait-for-expect"));
const PolicyEvents_1 = require("../../../MatrixTypes/PolicyEvents");
const Lifetime_1 = require("../../../Interface/Lifetime");
const SimulatedUserConsequences_1 = require("../../Capability/StandardCapability/SimulatedUserConsequences");
async function createMemberBanSynchronisationProtection(capabilities, protectedRoomsSet) {
    const description = (0, Protection_1.findProtection)('MemberBanSynchronisationProtection');
    if (description === undefined) {
        throw new TypeError('Should be able to find the member ban synchronisation protection');
    }
    const protectionResult = await description.factory(description, 
    // TODO: replace this when protected rooms set gets a lifetime.
    new Lifetime_1.StandardLifetime(), protectedRoomsSet, undefined, capabilities, {});
    if ((0, Action_1.isError)(protectionResult)) {
        throw new TypeError('Should be able to construct the protection');
    }
    // cba to add a generic to the protection description so the factory can have
    // a generic return type. Shouldn't be this level of disconect between types
    // and the real bullshit getting annoyed now whenver i have to do any meta
    // programming.
    return protectionResult.ok;
}
// handleMembershipChange
test('Membership changes that should result in a ban when matching an existing policy', async function () {
    const policyRoom = (0, EventGeneration_1.randomRoomID)([]);
    const protectedRoom = (0, EventGeneration_1.randomRoomID)([]);
    const changesToTest = [
        MembershipChange_1.MembershipChangeType.Invited,
        MembershipChange_1.MembershipChangeType.Joined,
        MembershipChange_1.MembershipChangeType.Knocked,
        MembershipChange_1.MembershipChangeType.Rejoined,
    ];
    const usersToTest = changesToTest.map((_change) => (0, EventGeneration_1.randomUserID)());
    const rejoiningUser = usersToTest[3];
    if (rejoiningUser === undefined) {
        throw new TypeError(`Test setup incorrectly`);
    }
    const { protectedRoomsSet, roomStateManager, roomMembershipManager } = await (0, DeclareRoomState_1.describeProtectedRoomsSet)({
        rooms: [
            {
                room: protectedRoom,
                membershipDescriptions: [
                    {
                        // we need this for the user who will rejoin the room.
                        sender: rejoiningUser,
                        membership: MembershipChange_1.Membership.Leave,
                    },
                ],
            },
        ],
        lists: [
            {
                room: policyRoom,
                policyDescriptions: usersToTest.map((userID) => ({
                    entity: userID,
                    type: PolicyEvents_1.PolicyRuleType.User,
                })),
            },
        ],
    });
    const userConsequences = new SimulatedUserConsequences_1.SimulatedUserConsequences(protectedRoomsSet.setRoomMembership);
    const consequenceSpy = jest.spyOn(userConsequences, 'consequenceForUserInRoom');
    const protection = await createMemberBanSynchronisationProtection({ userConsequences }, protectedRoomsSet);
    const membershipRevisionIssuer = roomMembershipManager.getFakeRoomMembershpRevisionIssuer(protectedRoom);
    roomStateManager.appendState({
        room: protectedRoom,
        membershipDescriptions: changesToTest.map((changeType, index) => {
            const membership = (() => {
                switch (changeType) {
                    case MembershipChange_1.MembershipChangeType.Invited:
                        return MembershipChange_1.Membership.Invite;
                    case MembershipChange_1.MembershipChangeType.Joined:
                    case MembershipChange_1.MembershipChangeType.Rejoined:
                        return MembershipChange_1.Membership.Join;
                    case MembershipChange_1.MembershipChangeType.Knocked:
                        return MembershipChange_1.Membership.Knock;
                    default:
                        throw new TypeError(`Unexpected membership change type in test`);
                }
            })();
            const userToTest = usersToTest[index];
            if (userToTest === undefined) {
                throw new TypeError(`There aren't enough test users for the changes to test`);
            }
            return {
                state_key: userToTest,
                sender: userToTest,
                membership: membership,
            };
        }),
    });
    await (0, wait_for_expect_1.default)(() => {
        expect(membershipRevisionIssuer.getNumberOfRevisions()).toBe(1);
    });
    const revisionEntry = membershipRevisionIssuer.getLastRevision();
    const protectionHandlerResult = await protection.handleMembershipChange.call(protection, revisionEntry[0], revisionEntry[1]);
    expect((0, Action_1.isOk)(protectionHandlerResult)).toBeTruthy();
    expect(consequenceSpy).toHaveBeenCalledTimes(usersToTest.length);
});
// handlePolicyRevision
// We need to test the consequence method itself in another test?
test('A policy change banning a user on a directly watched list will call the consequence to update for the revision', async function () {
    const spammerToBanUserID = `@spam:example.com`;
    const policyRoom = (0, EventGeneration_1.randomRoomID)([]);
    const { protectedRoomsSet, roomStateManager, policyRoomManager } = await (0, DeclareRoomState_1.describeProtectedRoomsSet)({
        rooms: [
            {
                membershipDescriptions: [
                    {
                        sender: spammerToBanUserID,
                        membership: MembershipChange_1.Membership.Join,
                    },
                ],
            },
        ],
        lists: [
            {
                room: policyRoom,
            },
        ],
    });
    const userConsequences = new SimulatedUserConsequences_1.SimulatedUserConsequences(protectedRoomsSet.setRoomMembership);
    const consequenceSpy = jest.spyOn(userConsequences, 'consequenceForUsersInRoomSet');
    const protection = await createMemberBanSynchronisationProtection({ userConsequences }, protectedRoomsSet);
    const policyRoomRevisionIssuer = policyRoomManager.getFakePolicyRoomRevisionIssuer(policyRoom);
    roomStateManager.appendState({
        room: policyRoom,
        policyDescriptions: [
            { entity: spammerToBanUserID, type: PolicyEvents_1.PolicyRuleType.User },
        ],
    });
    await (0, wait_for_expect_1.default)(() => {
        expect(policyRoomRevisionIssuer.getNumberOfRevisions()).toBe(1);
    });
    const revision = protectedRoomsSet.setPoliciesMatchingMembership.currentRevision;
    const protectionHandlerResult = await protection.synchroniseWithRevision(revision);
    expect((0, Action_1.isOk)(protectionHandlerResult)).toBeTruthy();
    expect(consequenceSpy).toHaveBeenCalled();
    expect(revision.allRulesMatchingMember(spammerToBanUserID, {})).toHaveLength(1);
});
//# sourceMappingURL=MemberBanSynchronisation.test.js.map