"use strict";
// SPDX-FileCopyrightText: 2024 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServerACLSynchronisationCapability = void 0;
exports.compileServerACL = compileServerACL;
const matrix_basic_types_1 = require("@the-draupnir-project/matrix-basic-types");
require("./ServerBanSynchronisationCapability"); // we need this so the interface is loaded AND yes we are going to move to description objects instead at some point FML.
const typescript_result_1 = require("@gnuxie/typescript-result");
const ActionException_1 = require("../../../Interface/ActionException");
const CapabilityProvider_1 = require("../../Capability/CapabilityProvider");
const RoomSetResult_1 = require("../../Capability/StandardCapability/RoomSetResult");
const ServerACLBuilder_1 = require("../../../MatrixTypes/ServerACLBuilder");
class ServerACLQueue {
    constructor(stateEventSender, serverName, protectedRoomsSet) {
        this.stateEventSender = stateEventSender;
        this.serverName = serverName;
        this.protectedRoomsSet = protectedRoomsSet;
        this.pendingRoomChecks = new Map();
        this.activeRoomChecks = new Map();
        // nothing to do.
    }
    async applyPolicyRevisionToRoom(roomID, projection) {
        const ACL = compileServerACL(this.serverName, projection.currentNode);
        const stateRevision = this.protectedRoomsSet.setRoomState.getRevision(roomID);
        if (stateRevision === undefined) {
            throw new TypeError(`Somehowe we can't get the state for this room ${roomID}`);
        }
        const existingStateEvent = stateRevision.getStateEvent('m.room.server_acl', '');
        if (existingStateEvent !== undefined &&
            ACL.matches(existingStateEvent.content)) {
            return (0, typescript_result_1.Ok)(false);
        }
        const result = await this.stateEventSender.sendStateEvent(roomID, 'm.room.server_acl', '', ACL.safeAclContent());
        // Give some time between ACL updates to not spam rooms.
        await new Promise((resolve) => setTimeout(resolve, 15000));
        if ((0, typescript_result_1.isError)(result)) {
            return result;
        }
        else {
            return (0, typescript_result_1.Ok)(true);
        }
    }
    async doActiveCheck(roomID, projection) {
        try {
            const activeCheck = this.applyPolicyRevisionToRoom(roomID, projection);
            this.activeRoomChecks.set(roomID, activeCheck);
            return await activeCheck;
        }
        finally {
            this.activeRoomChecks.delete(roomID);
        }
    }
    async enqueueCheck(roomID, projection, activeCheck) {
        try {
            await activeCheck;
        }
        finally {
            this.pendingRoomChecks.delete(roomID);
        }
        return await this.doActiveCheck(roomID, projection);
    }
    async enqueueRoomCheck(roomID, projection) {
        const pendingCheck = this.pendingRoomChecks.get(roomID);
        if (pendingCheck) {
            return pendingCheck;
        }
        const activeCheck = this.activeRoomChecks.get(roomID);
        if (activeCheck) {
            const pendingCheck = this.enqueueCheck(roomID, projection, activeCheck);
            this.pendingRoomChecks.set(roomID, pendingCheck);
            return await pendingCheck;
        }
        else {
            return await this.doActiveCheck(roomID, projection);
        }
    }
}
function compileServerACL(ourServerName, projectionNode) {
    const builder = new ServerACLBuilder_1.ServerACLBuilder(ourServerName).denyIpAddresses();
    builder.allowServer('*');
    for (const serverName of projectionNode.deny) {
        builder.denyServer(serverName);
    }
    return builder;
}
class ServerACLSynchronisationCapability {
    constructor(stateEventSender, protectedRoomsSet) {
        this.protectedRoomsSet = protectedRoomsSet;
        this.requiredPermissions = [];
        this.requiredEventPermissions = [];
        this.requiredStatePermissions = ['m.room.server_acl'];
        this.queue = new ServerACLQueue(stateEventSender, (0, matrix_basic_types_1.userServerName)(this.protectedRoomsSet.userID), protectedRoomsSet);
    }
    async applyIntentToSet(intentProjection) {
        const resultBuilder = new RoomSetResult_1.RoomSetResultBuilder();
        try {
            await Promise.all(this.protectedRoomsSet.allProtectedRooms.map((room) => this.queue
                .enqueueRoomCheck(room.toRoomIDOrAlias(), intentProjection)
                .then((result) => {
                resultBuilder.addResult(room.toRoomIDOrAlias(), result);
            })));
        }
        catch (e) {
            if (e instanceof Error) {
                return ActionException_1.ActionException.Result(`Uncaught error while applying server ACLS`, {
                    exception: e,
                    exceptionKind: ActionException_1.ActionExceptionKind.Unknown,
                });
            }
        }
        return (0, typescript_result_1.Ok)(resultBuilder.getResult());
    }
    async outcomeFromIntentInRoom(roomID, projection) {
        return await this.queue.enqueueRoomCheck(roomID, projection);
    }
    async outcomeFromIntentInRoomSet(projection) {
        return await this.applyIntentToSet(projection);
    }
}
exports.ServerACLSynchronisationCapability = ServerACLSynchronisationCapability;
(0, CapabilityProvider_1.describeCapabilityProvider)({
    name: 'ServerACLSynchronisationCapability',
    description: 'An implementation of ServerConsequences that uses m.room.server_acl to change access to rooms for servers.',
    interface: 'ServerBanSynchronisationCapability',
    factory(_protectionDescription, context) {
        return new ServerACLSynchronisationCapability(context.stateEventSender, context.protectedRoomsSet);
    },
});
//# sourceMappingURL=ServerACLSynchronisationCapability.js.map