/**
 * The purpose of the hash reverser is to implement the PolicyListRevisionIssuer interface by taking another PolicyListRevision,
 * and matching the HashedLiteral policies against known entities, and issuing new policies that
 * have a plain text literal instead of a hash.
 */
import { StringRoomID, StringServerName, StringUserID } from '@the-draupnir-project/matrix-basic-types';
import { HashedLiteralPolicyRule, LiteralPolicyRule } from '../../../PolicyList/PolicyRule';
import { PolicyRoomRevisionIssuer } from '../../../PolicyList/PolicyListRevisionIssuer';
import EventEmitter from 'events';
import { Result } from '@gnuxie/typescript-result';
import { StandardDirectPropagationPolicyListRevisionIssuer } from '../../DirectPropagationPolicyListRevisionIssuer';
export type RoomBasicDetails = {
    creator?: StringUserID | undefined;
    room_id: StringRoomID;
    name?: string | undefined;
    topic?: string | undefined;
    avatar?: string | undefined;
    joined_members?: number | undefined;
};
export type SHA256RerversedHashListener = (roomHashes: RoomHashRecord[], userHashes: UserHashRecord[], serverHashes: ServerHashRecord[]) => void;
export type RoomHashRecord = {
    room_id: StringRoomID;
    sha256: string;
};
export type UserHashRecord = {
    user_id: StringUserID;
    sha256: string;
    server_name: StringServerName;
};
export type ServerHashRecord = {
    server_name: string;
    sha256: string;
};
export type HashedRoomDetails = {
    roomID: StringRoomID;
    creator: StringUserID;
    server: string;
};
/**
 * In the future, when the time comes that we need to reverse all known users,
 * we will probably need a component that just forwards "discovered" users
 * to a store. This would then be a dependency included in the SetMembershipRevision.
 * It will probably want to accept the entire SetMembershipRevision itself
 * for checking at startup. This is so that it can break up the amount of work
 * on the db by breaking up the query size. And then the revision will just forward updates
 * as it discovers users as normal.
 */
export type SHA256Base64FromEntity = (entity: string) => string;
export interface SHA256HashStore {
    on(event: 'ReversedHashes', listener: SHA256RerversedHashListener): this;
    off(event: 'ReversedHashes', listener: SHA256RerversedHashListener): this;
    emit(event: 'ReversedHashes', ...args: Parameters<SHA256RerversedHashListener>): void;
    findUserHash(hash: string): Promise<Result<StringUserID | undefined>>;
    findRoomHash(hash: string): Promise<Result<StringRoomID | undefined>>;
    findServerHash(hash: string): Promise<Result<string | undefined>>;
    reverseHashedPolicies(policies: HashedLiteralPolicyRule[]): Promise<Result<LiteralPolicyRule[]>>;
    storeUndiscoveredRooms(roomIDs: StringRoomID[]): Promise<Result<RoomHashRecord[]>>;
    storeUndiscoveredUsers(userIDs: StringUserID[]): Promise<Result<UserHashRecord[]>>;
    storeRoomIdentification(details: HashedRoomDetails): Promise<Result<void>>;
    findRoomsByServer(server: StringServerName): Promise<Result<StringRoomID[]>>;
    findRoomsByCreator(creator: StringUserID): Promise<Result<StringRoomID[]>>;
    destroy(): void;
}
/**
 * The hash reverser works against a hash store to reveal policies.
 * The way it does this is by taking the hashed literal rules from a policy list revision,
 * and comparing them against the store. When policies are revealed, the reverser
 * goes to the original policy revision sourcing the policy, and tells the store
 * that the policy is now revealed.
 *
 * The policy room revision then updates its cache to show a literal rule with the
 * plain text entity rather than the hashed literal rule.
 *
 * The reverser is used by the RoomStateManagerFactory. This means that all
 * Draupnir in draupnir appservice mode share the same reverser. The reason
 * why we see this as acceptable is because if we end up with huge centralized
 * d4all instances where this becomes a problem, then we have failed at
 * decentralization.
 */
interface SHA256HashReverser {
    addPolicyRoomRevisionIssuer(issuer: PolicyRoomRevisionIssuer): void;
    unregisterListeners(): void;
}
export declare class StandardSHA256HashReverser extends EventEmitter implements SHA256HashReverser {
    private readonly store;
    readonly hashedPoliciesRevisionIssuer: StandardDirectPropagationPolicyListRevisionIssuer;
    private readonly issuers;
    constructor(store: SHA256HashStore);
    addPolicyRoomRevisionIssuer(issuer: PolicyRoomRevisionIssuer): void;
    private updateUpstreamWithRevealedPolicies;
    private readonly handleDiscoveredHashes;
    private readonly handlePolicyRoomRevision;
    private checkPoliciesAgainstStore;
    unregisterListeners(): void;
}
export {};
//# sourceMappingURL=SHA256HashReverser.d.ts.map