"use strict";
// SPDX-FileCopyrightText: 2025 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.StandardWatchedPolicyRooms = void 0;
const typescript_result_1 = require("@gnuxie/typescript-result");
const PolicyListConfig_1 = require("../PolicyListConfig/PolicyListConfig");
const DirectPropagationPolicyListRevisionIssuer_1 = require("../DirectPropagationPolicyListRevisionIssuer");
class StandardWatchedPolicyRooms {
    constructor(policyListConfig, policyRoomRevisionIssuers, revisionIssuer, policyRoomManager, roomJoiner) {
        this.policyListConfig = policyListConfig;
        this.policyRoomRevisionIssuers = policyRoomRevisionIssuers;
        this.revisionIssuer = revisionIssuer;
        this.policyRoomManager = policyRoomManager;
        this.roomJoiner = roomJoiner;
        // nothing to do.
    }
    static async create(policyListConfig, policyRoomManager, roomJoiner) {
        const issuers = new Map();
        for (const profile of policyListConfig.allWatchedLists) {
            const revisionIssuer = (await policyRoomManager.getPolicyRoomRevisionIssuer(profile.room)).expect("Something is badly wrong if we can't get a policy room revision issuer at this point");
            issuers.set(profile.room.toRoomIDOrAlias(), revisionIssuer);
        }
        const revisionIssuer = new DirectPropagationPolicyListRevisionIssuer_1.StandardDirectPropagationPolicyListRevisionIssuer([
            ...issuers.values(),
        ]);
        return (0, typescript_result_1.Ok)(new StandardWatchedPolicyRooms(policyListConfig, issuers, revisionIssuer, policyRoomManager, roomJoiner));
    }
    async watchPolicyRoomDirectly(room) {
        const joinResult = await this.roomJoiner.joinRoom(room);
        if ((0, typescript_result_1.isError)(joinResult)) {
            return joinResult.elaborate('Unable to join a policy room to be able to watch it');
        }
        const issuerResult = await this.policyRoomManager.getPolicyRoomRevisionIssuer(room);
        if ((0, typescript_result_1.isError)(issuerResult)) {
            return issuerResult.elaborate('Unable to get the policy room revision issuer to watch the policy room');
        }
        const storeResult = await this.policyListConfig.watchList(PolicyListConfig_1.PropagationType.Direct, room, {});
        if ((0, typescript_result_1.isError)(storeResult)) {
            return storeResult.elaborate('Unable to persist the new list subscription');
        }
        this.revisionIssuer.addIssuer(issuerResult.ok);
        this.policyRoomRevisionIssuers.set(room.toRoomIDOrAlias(), issuerResult.ok);
        return (0, typescript_result_1.Ok)(undefined);
    }
    async unwatchPolicyRoom(room) {
        const issuer = this.policyRoomRevisionIssuers.get(room.toRoomIDOrAlias());
        if (issuer === undefined) {
            return typescript_result_1.ResultError.Result('Unable to unwatch the list because it is not currently being watched');
        }
        const storeResult = await this.policyListConfig.unwatchList(PolicyListConfig_1.PropagationType.Direct, room);
        if ((0, typescript_result_1.isError)(storeResult)) {
            return storeResult.elaborate('Unable to persist removing the list subscription');
        }
        this.policyRoomRevisionIssuers.delete(room.toRoomIDOrAlias());
        this.revisionIssuer.removeIssuer(issuer);
        return (0, typescript_result_1.Ok)(undefined);
    }
    get allRooms() {
        return [...this.policyRoomRevisionIssuers.values()].map((issuer) => ({
            revision: issuer.currentRevision,
            room: issuer.room,
            propagation: PolicyListConfig_1.PropagationType.Direct,
        }));
    }
    get currentRevision() {
        return this.revisionIssuer.currentRevision;
    }
    unregisterListeners() {
        this.revisionIssuer.unregisterListeners();
    }
    findPolicyRoomFromShortcode(shortcode) {
        for (const { currentRevision } of this.policyRoomRevisionIssuers.values()) {
            if (currentRevision.shortcode === shortcode) {
                return {
                    revision: currentRevision,
                    room: currentRevision.room,
                    propagation: PolicyListConfig_1.PropagationType.Direct,
                };
            }
        }
        return undefined;
    }
}
exports.StandardWatchedPolicyRooms = StandardWatchedPolicyRooms;
//# sourceMappingURL=StandardWatchedPolicyRooms.js.map