"use strict";
// Copyright (C) 2023 - 2024 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.describeProtectedRoomsSet = describeProtectedRoomsSet;
exports.describeRoomStateEvents = describeRoomStateEvents;
exports.describeRoom = describeRoom;
exports.describeRoomMember = describeRoomMember;
exports.describePolicyRule = describePolicyRule;
exports.describeStateEvent = describeStateEvent;
const crypto_1 = require("crypto");
const EventGeneration_1 = require("../TestUtilities/EventGeneration");
const MembershipChange_1 = require("../Membership/MembershipChange");
const StandardRoomStateRevision_1 = require("./StandardRoomStateRevision");
const Action_1 = require("../Interface/Action");
const PolicyRule_1 = require("../PolicyList/PolicyRule");
const StandardRoomMembershipRevision_1 = require("../Membership/StandardRoomMembershipRevision");
const StandardPolicyRoomRevision_1 = require("../PolicyList/StandardPolicyRoomRevision");
const ProtectedRoomsSet_1 = require("../Protection/ProtectedRoomsSet");
const FakeProtectedRoomsConfig_1 = require("../Protection/ProtectedRoomsConfig/FakeProtectedRoomsConfig");
const FakeRoomStateRevisionIssuer_1 = require("./FakeRoomStateRevisionIssuer");
const FakeRoomStateManager_1 = require("./FakeRoomStateManager");
const StandardSetRoomMembership_1 = require("../Membership/StandardSetRoomMembership");
const FakeRoomMembershipManager_1 = require("../Membership/FakeRoomMembershipManager");
const FakePolicyRoomManager_1 = require("./FakePolicyRoomManager");
const StandardSetRoomState_1 = require("./StandardSetRoomState");
const FakePolicyRoomRevisionIssuer_1 = require("../PolicyList/FakePolicyRoomRevisionIssuer");
const FakeRoomMembershipRevisionIssuer_1 = require("../Membership/FakeRoomMembershipRevisionIssuer");
const PolicyRuleEventBuilder_1 = require("../PolicyList/PolicyRuleEventBuilder");
const FakeProtectionsManager_1 = require("../Protection/ProtectionsManager/FakeProtectionsManager");
const StandardProtectedRoomsManager_1 = require("../Protection/ProtectedRoomsManager/StandardProtectedRoomsManager");
const DummyClientPlatform_1 = require("../Client/DummyClientPlatform");
const Logger_1 = require("../Logging/Logger");
const FakePersistentMatrixData_1 = require("../Interface/FakePersistentMatrixData");
const MjolnirPolicyRoomsConfig_1 = require("../Protection/PolicyListConfig/MjolnirPolicyRoomsConfig");
const StandardWatchedPolicyRooms_1 = require("../Protection/WatchedPolicyRooms/StandardWatchedPolicyRooms");
const DefaultEventDecoder_1 = require("../MatrixTypes/DefaultEventDecoder");
const DefaultMixinExtractor_1 = require("../SafeMatrixEvents/MatrixEventMixinDescriptions/DefaultMixinExtractor");
const log = new Logger_1.Logger('DeclareRoomState');
async function describeProtectedRoomsSet({ rooms = [], lists = [], clientUserID = (0, EventGeneration_1.randomUserID)(), }) {
    const listDescriptions = lists.map(describeRoom);
    const roomDescriptions = [...listDescriptions, ...rooms.map(describeRoom)];
    const roomStateManager = new FakeRoomStateManager_1.FakeRoomStateManager(roomDescriptions.map((description) => description.stateRevisionIssuer));
    const roomMembershipManager = new FakeRoomMembershipManager_1.FakeRoomMembershipManager(roomDescriptions.map((description) => description.membershipRevisionIssuer));
    const policyRoomManager = new FakePolicyRoomManager_1.FakePolicyRoomManager(roomDescriptions.map((description) => description.policyRevisionIssuer));
    const protectedRoomsConfig = new FakeProtectedRoomsConfig_1.FakeProtectedRoomsConfig(roomDescriptions.map((room) => room.stateRevisionIssuer.room));
    const setMembership = await StandardSetRoomMembership_1.StandardSetRoomMembership.create(roomMembershipManager, protectedRoomsConfig.getProtectedRooms());
    if ((0, Action_1.isError)(setMembership)) {
        log.error(`Unable to create set membership`, setMembership.error);
        throw new TypeError(`Unable to create set membership`);
    }
    const setRoomState = await StandardSetRoomState_1.StandardSetRoomState.create(roomStateManager, protectedRoomsConfig.getProtectedRooms());
    if ((0, Action_1.isError)(setRoomState)) {
        log.error(`Unable to create set room state`, setRoomState.error);
        throw new TypeError(`Unable to create set room state`);
    }
    const protectedRoomsManager = await StandardProtectedRoomsManager_1.StandardProtectedRoomsManager.create(protectedRoomsConfig, roomStateManager, roomMembershipManager, DummyClientPlatform_1.DummyRoomJoiner, setMembership.ok, setRoomState.ok);
    if ((0, Action_1.isError)(protectedRoomsManager)) {
        log.error(`Unable to create protected rooms manager`, protectedRoomsManager.error);
        throw new TypeError(`Unable to create protected rooms manager`);
    }
    const policyListConfigAccountData = new FakePersistentMatrixData_1.FakePersistentConfigBackend({
        references: listDescriptions.map((description) => description.policyRevisionIssuer.currentRevision.room.toPermalink()),
    });
    const policyRoomsConfig = (await MjolnirPolicyRoomsConfig_1.MjolnirPolicyRoomsConfig.createFromStore(policyListConfigAccountData, DummyClientPlatform_1.DummyRoomJoiner)).expect('Unable to create policy rooms config backend');
    const watchedPolicyRooms = (await StandardWatchedPolicyRooms_1.StandardWatchedPolicyRooms.create(policyRoomsConfig, policyRoomManager, DummyClientPlatform_1.DummyRoomJoiner)).expect('unable to create watched policy rooms');
    const protectedRoomsSet = new ProtectedRoomsSet_1.StandardProtectedRoomsSet(watchedPolicyRooms, protectedRoomsManager.ok, new FakeProtectionsManager_1.FakeProtectionsManager(), clientUserID, DefaultMixinExtractor_1.DefaultMixinExtractor);
    return {
        protectedRoomsSet,
        roomStateManager,
        policyRoomManager,
        roomMembershipManager,
    };
}
function describeRoomStateEvents({ room, stateDescriptions = [], membershipDescriptions = [], policyDescriptions = [], }) {
    const membershipEvents = membershipDescriptions.map((description) => describeRoomMember({
        ...description,
        room_id: room.toRoomIDOrAlias(),
    }));
    const policyEvents = policyDescriptions.map((description) => describePolicyRule({
        ...description,
        room_id: room.toRoomIDOrAlias(),
    }));
    const stateEvents = [
        ...stateDescriptions.map((description) => describeStateEvent({
            ...description,
            room_id: room.toRoomIDOrAlias(),
        })),
        ...membershipEvents,
        ...policyEvents,
    ];
    return {
        room,
        stateEvents,
        membershipEvents,
        policyEvents,
    };
}
function describeRoom({ room = (0, EventGeneration_1.randomRoomID)([]), stateDescriptions = [], membershipDescriptions = [], policyDescriptions = [], }) {
    const { policyEvents, stateEvents, membershipEvents } = describeRoomStateEvents({
        room,
        stateDescriptions,
        membershipDescriptions,
        policyDescriptions,
    });
    const stateRevision = StandardRoomStateRevision_1.StandardRoomStateRevision.blankRevision(room).reviseFromState(stateEvents);
    const membershipRevision = StandardRoomMembershipRevision_1.StandardRoomMembershipRevision.blankRevision(room).reviseFromMembership(membershipEvents);
    const policyRevision = StandardPolicyRoomRevision_1.StandardPolicyRoomRevision.blankRevision(room).reviseFromState(policyEvents);
    const stateRevisionIssuer = new FakeRoomStateRevisionIssuer_1.FakeRoomStateRevisionIssuer(stateRevision, room);
    const membershipRevisionIssuer = new FakeRoomMembershipRevisionIssuer_1.FakeRoomMembershipRevisionIssuer(room, membershipRevision, stateRevisionIssuer);
    const policyRevisionIssuer = new FakePolicyRoomRevisionIssuer_1.FakePolicyRoomRevisionIssuer(room, policyRevision, stateRevisionIssuer);
    return {
        stateRevisionIssuer,
        membershipRevisionIssuer,
        policyRevisionIssuer,
    };
}
function describeRoomMember({ sender, target = sender, membership = MembershipChange_1.Membership.Join, room_id = (0, EventGeneration_1.randomRoomID)([]).toRoomIDOrAlias(), avatar_url, displayname, reason, }) {
    return describeStateEvent({
        sender,
        state_key: target,
        content: {
            membership,
            avatar_url,
            displayname,
            reason,
        },
        type: 'm.room.member',
        room_id,
    });
}
function describePolicyRule({ sender = (0, EventGeneration_1.randomUserID)(), room_id, type, entity, reason = '<no reason supplied>', recommendation = PolicyRule_1.Recommendation.Ban, copyFrom, remove, hashes, }) {
    const content = (() => {
        if (remove !== undefined) {
            return undefined;
        }
        else if (copyFrom !== undefined) {
            return undefined;
        }
        else if (entity !== undefined) {
            return {
                entity,
                reason,
                recommendation,
            };
        }
        else if (hashes !== undefined) {
            return {
                hashes,
                reason,
                recommendation,
            };
        }
        else {
            throw new TypeError(`Content fields should be defined when copyFrom and remove aren't being used`);
        }
    })();
    const description = (0, PolicyRuleEventBuilder_1.buildPolicyEvent)({
        type,
        content,
        copyFrom,
        remove,
    });
    return describeStateEvent({
        sender,
        state_key: description.state_key,
        room_id,
        type: description.type,
        content: description.content,
    });
}
function describeStateEvent({ sender, state_key = '', type, content = {}, room_id = `!${(0, crypto_1.randomUUID)()}:example.com`, }) {
    const rawEventJSON = {
        room_id,
        event_id: `$${(0, crypto_1.randomUUID)()}:example.com`,
        origin_server_ts: Date.now(),
        state_key: state_key,
        type,
        sender,
        content,
    };
    const decodeResult = DefaultEventDecoder_1.DefaultEventDecoder.decodeStateEvent(rawEventJSON);
    if ((0, Action_1.isError)(decodeResult)) {
        throw new TypeError(`Something is wrong with the event generator [${decodeResult.error.errors.toString()}]`);
    }
    else {
        return decodeResult.ok;
    }
}
//# sourceMappingURL=DeclareRoomState.js.map