"use strict";
// Copyright (C) 2022 - 2023 Gnuxie <Gnuxie@protonmail.com>
// Copyright 2019 - 2021 The Matrix.org Foundation C.I.C.
//
// SPDX-License-Identifier: AFL-3.0 AND Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from mjolnir
// https://github.com/matrix-org/mjolnir
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
exports.StandardRoomStateRevision = void 0;
const StateChangeType_1 = require("./StateChangeType");
const Revision_1 = require("../PolicyList/Revision");
const immutable_1 = require("immutable");
/**
 * A standard implementation of a `PolicyListRevision` using immutable's persistent maps.
 */
class StandardRoomStateRevision {
    /**
     * Use {@link StandardRoomStateRevision.blankRevision} to get started.
     * Only use this constructor if you are implementing a variant of PolicyListRevision.
     * @param revisionID A revision ID to represent this revision.
     * @param policyRules A map containing the rules for this revision by state type and then state key.
     * @param policyRuleByEventId A map containing the rules ofr this revision by event id.
     */
    constructor(room, revisionID, 
    /**
     * A map of state events indexed first by state type and then state keys.
     */
    stateEvents, 
    /**
     * Allow us to detect whether we have updated the state for this event.
     */
    stateEventsByEventID) {
        this.room = room;
        this.revisionID = revisionID;
        this.stateEvents = stateEvents;
        this.stateEventsByEventID = stateEventsByEventID;
    }
    /**
     * @returns An empty revision.
     */
    static blankRevision(room) {
        return new StandardRoomStateRevision(room, new Revision_1.Revision(), (0, immutable_1.Map)(), (0, immutable_1.Map)());
    }
    isBlankRevision() {
        return this.stateEventsByEventID.isEmpty();
    }
    get allState() {
        return [...this.stateEventsByEventID.values()];
    }
    getStateEvent(type, key) {
        var _a;
        return (_a = this.stateEvents.get(type)) === null || _a === void 0 ? void 0 : _a.get(key);
    }
    getStateEventsOfType(type) {
        const typeTable = this.stateEvents.get(type);
        if (typeTable) {
            return [...typeTable.values()];
        }
        else {
            return [];
        }
    }
    getStateEventsOfTypes(types) {
        return types.map((type) => this.getStateEventsOfType(type)).flat();
    }
    reviseFromChanges(changes) {
        let nextStateEvents = this.stateEvents;
        let nextStateEventsByEventID = this.stateEventsByEventID;
        const setStateEvent = (change) => {
            const event = change.state;
            nextStateEvents = nextStateEvents.setIn([event.type, event.state_key], event);
            if (change.previousState !== undefined) {
                nextStateEventsByEventID = nextStateEventsByEventID.delete(change.previousState.event_id);
            }
            nextStateEventsByEventID = nextStateEventsByEventID.set(event.event_id, event);
        };
        for (const change of changes) {
            if ((0, StateChangeType_1.isChanged)(change.changeType)) {
                setStateEvent(change);
            }
        }
        return new StandardRoomStateRevision(this.room, new Revision_1.Revision(), nextStateEvents, nextStateEventsByEventID);
    }
    hasEvent(eventId) {
        return this.stateEventsByEventID.has(eventId);
    }
    /**
     * Calculate the changes from this revision with a more recent set of state events.
     * Will only show the difference, if the set is the same then no changes will be returned.
     * @param state The state events that reflect a different revision of the list.
     * @returns Any changes between this revision and the new set of state events.
     */
    changesFromState(state) {
        const changes = [];
        for (const event of state) {
            const existingState = this.getStateEvent(event.type, event.state_key);
            const changeType = (0, StateChangeType_1.calculateStateChange)(event, existingState);
            if ((0, StateChangeType_1.isChanged)(changeType)) {
                changes.push({
                    eventType: event.type,
                    changeType,
                    state: event,
                    ...(existingState ? { previousState: existingState } : {}),
                });
            }
        }
        return changes;
    }
    reviseFromState(state) {
        const changes = this.changesFromState(state);
        return this.reviseFromChanges(changes);
    }
}
exports.StandardRoomStateRevision = StandardRoomStateRevision;
//# sourceMappingURL=StandardRoomStateRevision.js.map