"use strict";
// SPDX-FileCopyrightText: 2023-2024 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
Object.defineProperty(exports, "__esModule", { value: true });
// please note that the changes calculated from this test need to be tested
// against the standard policy list revision.
const PolicyEvents_1 = require("../MatrixTypes/PolicyEvents");
const DeclareRoomState_1 = require("./DeclareRoomState");
const StandardRoomStateRevision_1 = require("./StandardRoomStateRevision");
const EventGeneration_1 = require("../TestUtilities/EventGeneration");
const StateChangeType_1 = require("./StateChangeType");
const MembershipChange_1 = require("../Membership/MembershipChange");
// if events aren't normalized as they are indexed then we really need to make
// sure that the policy room editor removes them according to their source
// event type, not their normalised state type.
// So in case you haven't realised, we've started using the state revision here,
// which might change the meaning of the test.
// in the early days of MPS we allowed policy list revisions to be created
// without depending on a room state revision to be informed of changes.
// that's probably going to change, so that policyRoomRevision don't have
// a method to `reviseFromState`.
test('A new policy rule will be seen as an Introduced rule by the revision', function () {
    var _a;
    const blankRevision = StandardRoomStateRevision_1.StandardRoomStateRevision.blankRevision((0, EventGeneration_1.randomRoomID)([]));
    const changes = blankRevision.changesFromState([
        (0, DeclareRoomState_1.describePolicyRule)({
            type: PolicyEvents_1.PolicyRuleType.User,
            entity: (0, EventGeneration_1.randomUserID)(),
        }),
    ]);
    expect(changes.length).toBe(1);
    expect((_a = changes.at(0)) === null || _a === void 0 ? void 0 : _a.changeType).toBe(StateChangeType_1.StateChangeType.Introduced);
});
test('Sending a contentful state event over a blank state event with the same type-key pair will be seen as Reintroducing a rule', function () {
    var _a;
    const entity = (0, EventGeneration_1.randomUserID)();
    const policy = (0, DeclareRoomState_1.describePolicyRule)({
        type: PolicyEvents_1.PolicyRuleType.User,
        entity,
    });
    const revision = StandardRoomStateRevision_1.StandardRoomStateRevision.blankRevision((0, EventGeneration_1.randomRoomID)([]))
        .reviseFromState([policy])
        .reviseFromState([
        (0, DeclareRoomState_1.describePolicyRule)({
            remove: policy,
        }),
    ]);
    const changes = revision.changesFromState([policy]);
    expect(changes.length).toBe(1);
    expect((_a = changes.at(0)) === null || _a === void 0 ? void 0 : _a.changeType).toBe(StateChangeType_1.StateChangeType.Reintroduced);
});
test('A redacted event state event that is returned by `/state` on a blank revision should result in IntroducedAsEmpty', function () {
    var _a;
    const policy = (0, DeclareRoomState_1.describePolicyRule)({
        type: PolicyEvents_1.PolicyRuleType.User,
        entity: (0, EventGeneration_1.randomUserID)(),
    });
    const blankRevision = StandardRoomStateRevision_1.StandardRoomStateRevision.blankRevision((0, EventGeneration_1.randomRoomID)([]));
    const changes = blankRevision.changesFromState([
        {
            ...policy,
            content: {},
            unsigned: {
                redacted_because: {
                    reason: 'unbanning the user',
                },
            },
        },
    ]);
    expect(changes.length).toBe(1);
    expect((_a = changes.at(0)) === null || _a === void 0 ? void 0 : _a.changeType).toBe(StateChangeType_1.StateChangeType.IntroducedAsBlank);
});
test('Sending a blank state event to an already blank type-key pair will result in BlankingEmptyContent', function () {
    var _a;
    const entity = (0, EventGeneration_1.randomUserID)();
    const policy = (0, DeclareRoomState_1.describePolicyRule)({
        type: PolicyEvents_1.PolicyRuleType.User,
        entity,
    });
    const revision = StandardRoomStateRevision_1.StandardRoomStateRevision.blankRevision((0, EventGeneration_1.randomRoomID)([])).reviseFromState([
        (0, DeclareRoomState_1.describePolicyRule)({
            remove: policy,
        }),
    ]);
    const changes = revision.changesFromState([
        (0, DeclareRoomState_1.describePolicyRule)({
            remove: policy,
        }),
    ]);
    expect(changes.length).toBe(1);
    expect((_a = changes.at(0)) === null || _a === void 0 ? void 0 : _a.changeType).toBe(StateChangeType_1.StateChangeType.BlankedEmptyContent);
});
test('Sending a blank state event with the same type-key pair will be seen as making the rule have BlankedContent', function () {
    var _a;
    const entity = (0, EventGeneration_1.randomUserID)();
    const policy = (0, DeclareRoomState_1.describePolicyRule)({
        type: PolicyEvents_1.PolicyRuleType.User,
        entity,
    });
    const revision = StandardRoomStateRevision_1.StandardRoomStateRevision.blankRevision((0, EventGeneration_1.randomRoomID)([])).reviseFromState([policy]);
    const changes = revision.changesFromState([
        (0, DeclareRoomState_1.describePolicyRule)({
            remove: policy,
        }),
    ]);
    expect(changes.length).toBe(1);
    expect((_a = changes.at(0)) === null || _a === void 0 ? void 0 : _a.changeType).toBe(StateChangeType_1.StateChangeType.BlankedContent);
});
test('Redacting a rule will be seen as CompletelyRedacting a rule (without checking redacted_because)', function () {
    var _a;
    const entity = (0, EventGeneration_1.randomUserID)();
    const event = (0, DeclareRoomState_1.describePolicyRule)({
        type: PolicyEvents_1.PolicyRuleType.User,
        entity,
    });
    const revision = StandardRoomStateRevision_1.StandardRoomStateRevision.blankRevision((0, EventGeneration_1.randomRoomID)([])).reviseFromState([event]);
    const changes = revision.changesFromState([
        {
            ...event,
            content: {},
        },
    ]);
    expect(changes.length).toBe(1);
    expect((_a = changes.at(0)) === null || _a === void 0 ? void 0 : _a.changeType).toBe(StateChangeType_1.StateChangeType.CompletelyRedacted);
});
test('A redacted event for an existing state (ensures check for redacted_because)', function () {
    var _a;
    const policy = (0, DeclareRoomState_1.describePolicyRule)({
        type: PolicyEvents_1.PolicyRuleType.User,
        entity: (0, EventGeneration_1.randomUserID)(),
    });
    const revision = StandardRoomStateRevision_1.StandardRoomStateRevision.blankRevision((0, EventGeneration_1.randomRoomID)([])).reviseFromState([policy]);
    const changes = revision.changesFromState([
        {
            ...policy,
            content: {},
            unsigned: {
                redacted_because: {
                    reason: 'unbanning the user',
                },
            },
        },
    ]);
    expect(changes.length).toBe(1);
    expect((_a = changes.at(0)) === null || _a === void 0 ? void 0 : _a.changeType).toBe(StateChangeType_1.StateChangeType.CompletelyRedacted);
});
test('A redacted membership event is classified as PartiallyRedacted because it still has keys', function () {
    var _a;
    const roomID = (0, EventGeneration_1.randomRoomID)([]);
    const member = (0, DeclareRoomState_1.describeRoomMember)({
        sender: (0, EventGeneration_1.randomUserID)(),
        avatar_url: 'mxc://example.com/wiejfoiejf',
        displayname: 'Red Wine from Coloroy',
        membership: MembershipChange_1.Membership.Join,
    });
    const revision = StandardRoomStateRevision_1.StandardRoomStateRevision.blankRevision(roomID).reviseFromState([member]);
    const changes = revision.changesFromState([
        {
            ...member,
            content: {
                membership: MembershipChange_1.Membership.Join,
            },
            unsigned: {
                redacted_because: {
                    reason: 'unbanning the user',
                },
            },
        },
    ]);
    expect(changes.length).toBe(1);
    expect((_a = changes.at(0)) === null || _a === void 0 ? void 0 : _a.changeType).toBe(StateChangeType_1.StateChangeType.PartiallyRedacted);
});
test('A modified rule will be seen as a Superseding an existing rule', function () {
    var _a;
    const entity = (0, EventGeneration_1.randomUserID)();
    const revision = StandardRoomStateRevision_1.StandardRoomStateRevision.blankRevision((0, EventGeneration_1.randomRoomID)([])).reviseFromState([
        (0, DeclareRoomState_1.describePolicyRule)({
            type: PolicyEvents_1.PolicyRuleType.User,
            entity,
        }),
    ]);
    const changes = revision.changesFromState([
        (0, DeclareRoomState_1.describePolicyRule)({
            type: PolicyEvents_1.PolicyRuleType.User,
            entity,
            reason: 'A brand new reason, because the old one was out of date',
        }),
    ]);
    expect(changes.length).toBe(1);
    expect((_a = changes.at(0)) === null || _a === void 0 ? void 0 : _a.changeType).toBe(StateChangeType_1.StateChangeType.SupersededContent);
});
test('Recieving the same poliy rule will not count as a modification or addition', function () {
    const policy = (0, DeclareRoomState_1.describePolicyRule)({
        type: PolicyEvents_1.PolicyRuleType.User,
        entity: (0, EventGeneration_1.randomUserID)(),
    });
    const blankRevision = StandardRoomStateRevision_1.StandardRoomStateRevision.blankRevision((0, EventGeneration_1.randomRoomID)([])).reviseFromState([policy]);
    const changes = blankRevision.changesFromState([policy]);
    expect(changes.length).toBe(0);
});
//# sourceMappingURL=StateChangeType.test.js.map