"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BanPropagationProtection = void 0;
// Copyright 2022 - 2025 Gnuxie <Gnuxie@protonmail.com>
// Copyright 2019 - 2022 The Matrix.org Foundation C.I.C.
//
// SPDX-License-Identifier: AFL-3.0 AND Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from mjolnir
// https://github.com/matrix-org/mjolnir
// </text>
const interface_manager_1 = require("@the-draupnir-project/interface-manager");
const matrix_protection_suite_1 = require("matrix-protection-suite");
const matrix_protection_suite_for_matrix_bot_sdk_1 = require("matrix-protection-suite-for-matrix-bot-sdk");
const matrix_basic_types_1 = require("@the-draupnir-project/matrix-basic-types");
const Unban_1 = require("../commands/unban/Unban");
const UnbanUsers_1 = require("../commands/unban/UnbanUsers");
const mps_interface_adaptor_1 = require("@the-draupnir-project/mps-interface-adaptor");
const log = new matrix_protection_suite_1.Logger("BanPropagationProtection");
const BAN_PROPAGATION_PROMPT_LISTENER = "ge.applied-langua.ge.draupnir.ban_propagation";
function makePolicyRoomReactionReferenceMap(rooms) {
    return mps_interface_adaptor_1.MatrixReactionHandler.createItemizedReactionMap(rooms.map((room) => room.toPermalink()));
}
/**
 * Prompt the management room to propagate a user ban to a policy list of their choice.
 * @param mjolnir Draupnir.
 * @param event The ban event.
 * @param roomId The room that the ban happened in.
 * @returns An event id which can be used by the `PromptResponseListener`.
 */
async function promptBanPropagation(draupnir, change) {
    const editablePolicyRoomIDs = draupnir.policyRoomManager.getEditablePolicyRoomIDs(draupnir.clientUserID, matrix_protection_suite_1.PolicyRuleType.User);
    const reactionMap = makePolicyRoomReactionReferenceMap(editablePolicyRoomIDs);
    const promptSendResult = await (0, mps_interface_adaptor_1.sendMatrixEventsFromDeadDocument)(draupnir.clientPlatform.toRoomMessageSender(), draupnir.managementRoomID, interface_manager_1.DeadDocumentJSX.JSXFactory("root", null,
        "The user",
        " ",
        (0, mps_interface_adaptor_1.renderMentionPill)(change.userID, change.content.displayname ?? change.userID),
        " ",
        "was banned in",
        " ",
        interface_manager_1.DeadDocumentJSX.JSXFactory("a", { href: `https://matrix.to/#/${change.roomID}` }, change.roomID),
        " by",
        " ",
        (0, mps_interface_adaptor_1.renderMentionPill)(change.sender, change.sender),
        " for",
        " ",
        interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, change.content.reason ?? "<no reason supplied>"),
        ".",
        interface_manager_1.DeadDocumentJSX.JSXFactory("br", null),
        "Would you like to add the ban to a policy list?",
        interface_manager_1.DeadDocumentJSX.JSXFactory("ol", null, editablePolicyRoomIDs.map((room) => (interface_manager_1.DeadDocumentJSX.JSXFactory("li", null,
            interface_manager_1.DeadDocumentJSX.JSXFactory("a", { href: room.toPermalink() }, room.toRoomIDOrAlias())))))), {
        additionalContent: draupnir.reactionHandler.createAnnotation(BAN_PROPAGATION_PROMPT_LISTENER, reactionMap, {
            target: change.userID,
            reason: change.content.reason,
        }),
    });
    if ((0, matrix_protection_suite_1.isError)(promptSendResult)) {
        log.error(`Could not send the prompt to the management room for the ban in ${change.roomID} for the user ${change.userID}`, promptSendResult.error);
        return;
    }
    await draupnir.reactionHandler.addReactionsToEvent(draupnir.managementRoomID, promptSendResult.ok[0], reactionMap);
}
function renderUnbanPrompt(membershipChange, roomID, unbanPreview) {
    return (interface_manager_1.DeadDocumentJSX.JSXFactory("root", null,
        "The user",
        " ",
        (0, mps_interface_adaptor_1.renderMentionPill)(membershipChange.userID, membershipChange.content.displayname ?? membershipChange.userID),
        " ",
        "was unbanned from the room",
        " ",
        (0, mps_interface_adaptor_1.renderRoomPill)(matrix_basic_types_1.MatrixRoomReference.fromRoomID(roomID)),
        " by",
        " ",
        membershipChange.sender,
        " for",
        " ",
        interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, membershipChange.content.reason ?? "<no reason supplied>"),
        interface_manager_1.DeadDocumentJSX.JSXFactory("br", null),
        "Would you like to remove these rules and unban the user from all protected rooms?",
        (0, Unban_1.renderUnbanMembersPreview)(unbanPreview)));
}
async function promptUnbanPropagation(draupnir, membershipChange, roomID) {
    const unbanPreview = (0, UnbanUsers_1.findUnbanInformationForMember)(draupnir.protectedRoomsSet.setRoomMembership, new matrix_basic_types_1.MatrixUserID(membershipChange.userID), draupnir.protectedRoomsSet.watchedPolicyRooms, { inviteMembers: false });
    const promptSendResult = await (0, mps_interface_adaptor_1.sendConfirmationPrompt)(draupnir, {
        commandDesignator: ["draupnir", "unban"],
        readItems: [membershipChange.userID],
    }, renderUnbanPrompt(membershipChange, roomID, unbanPreview), { roomID: draupnir.managementRoomID });
    if ((0, matrix_protection_suite_1.isError)(promptSendResult)) {
        log.error(`Could not send the prompt to the management room for the unban in ${roomID} for the user ${membershipChange.userID}`, promptSendResult.error);
        return;
    }
}
class BanPropagationProtection extends matrix_protection_suite_1.AbstractProtection {
    constructor(description, lifetime, capabilities, protectedRoomsSet, draupnir) {
        super(description, lifetime, capabilities, protectedRoomsSet, {});
        this.draupnir = draupnir;
        this.banPropagationPromptListener = this.banReactionListener.bind(this);
        this.draupnir.reactionHandler.on(BAN_PROPAGATION_PROMPT_LISTENER, this.banPropagationPromptListener);
    }
    handleProtectionDisable() {
        this.draupnir.reactionHandler.off(BAN_PROPAGATION_PROMPT_LISTENER, this.banPropagationPromptListener);
    }
    async handleMembershipChange(revision, changes) {
        // use Membership and not MembershipChangeType so that we can detect edits to ban reasons.
        const bans = changes.filter((change) => change.membership === matrix_protection_suite_1.Membership.Ban &&
            change.sender !== this.protectedRoomsSet.userID);
        const unbans = changes.filter((change) => change.membershipChangeType === matrix_protection_suite_1.MembershipChangeType.Unbanned &&
            change.sender !== this.protectedRoomsSet.userID);
        for (const ban of bans) {
            this.handleBan(ban);
        }
        for (const unban of unbans) {
            void (0, matrix_protection_suite_1.Task)(this.handleUnban(unban));
        }
        return (0, matrix_protection_suite_1.Ok)(undefined);
    }
    handleBan(change) {
        const policyRevision = this.protectedRoomsSet.watchedPolicyRooms.currentRevision;
        const rulesMatchingUser = (0, UnbanUsers_1.revisionRulesMatchingUser)(change.userID, [matrix_protection_suite_1.Recommendation.Ban, matrix_protection_suite_1.Recommendation.Takedown], policyRevision);
        if (rulesMatchingUser.length > 0) {
            return; // user is already banned.
        }
        void (0, matrix_protection_suite_1.Task)(promptBanPropagation(this.draupnir, change));
    }
    async handleUnban(change) {
        await promptUnbanPropagation(this.draupnir, change, change.roomID);
    }
    async banReactionListener(key, item, context) {
        if (typeof item === "string") {
            const policyRoomRef = matrix_basic_types_1.MatrixRoomReference.fromPermalink(item);
            if ((0, matrix_protection_suite_1.isError)(policyRoomRef)) {
                log.error(`Could not parse the room reference for the policy list to ban a user within ${item}`, policyRoomRef.error, context);
                return;
            }
            const roomID = await (0, matrix_protection_suite_for_matrix_bot_sdk_1.resolveRoomReferenceSafe)(this.draupnir.client, policyRoomRef.ok);
            if ((0, matrix_protection_suite_1.isError)(roomID)) {
                log.error(`Could not resolve the room reference for the policy list to ban a user within ${policyRoomRef.ok.toPermalink()}`, roomID.error);
                return;
            }
            const listResult = await this.draupnir.policyRoomManager.getPolicyRoomEditor(roomID.ok);
            if ((0, matrix_protection_suite_1.isError)(listResult)) {
                log.error(`Could not find a policy list for the policy room ${policyRoomRef.ok.toPermalink()}`, listResult.error);
                return;
            }
            const banResult = await listResult.ok.banEntity(matrix_protection_suite_1.PolicyRuleType.User, context.target, context.reason);
            if ((0, matrix_protection_suite_1.isError)(banResult)) {
                log.error(`Could not ban a user ${context.target} from the list ${policyRoomRef.ok.toPermalink()}`, banResult.error);
            }
        }
        else {
            log.error(`The Ban Result map has been malformed somehow item:`, item);
        }
    }
}
exports.BanPropagationProtection = BanPropagationProtection;
(0, matrix_protection_suite_1.describeProtection)({
    name: "BanPropagationProtection",
    description: "When you ban a user in any protected room with a client, this protection\
    will turn the room level ban into a policy for a policy list of your choice.\
    This will then allow the bot to ban the user from all of your rooms.",
    capabilityInterfaces: {
        userConsequences: "UserConsequences",
    },
    defaultCapabilities: {
        userConsequences: "StandardUserConsequences",
    },
    factory: async (decription, lifetime, protectedRoomsSet, draupnir, capabilities, _settings) => (0, matrix_protection_suite_1.allocateProtection)(lifetime, new BanPropagationProtection(decription, lifetime, capabilities, protectedRoomsSet, draupnir)),
});
//# sourceMappingURL=BanPropagation.js.map