"use strict";
// SPDX-FileCopyrightText: 2025 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.NotificationRoomCreator = void 0;
const typescript_result_1 = require("@gnuxie/typescript-result");
const matrix_protection_suite_1 = require("matrix-protection-suite");
class NotificationRoomCreator {
    constructor(protectedRoomsManager, settingChangeCB, roomCreateCapability, roomInviter, roomStateEventCapability, roomName, draupnirUserID, draupnirManagementRoomID, draupnirModerators, log) {
        this.protectedRoomsManager = protectedRoomsManager;
        this.settingChangeCB = settingChangeCB;
        this.roomCreateCapability = roomCreateCapability;
        this.roomInviter = roomInviter;
        this.roomStateEventCapability = roomStateEventCapability;
        this.roomName = roomName;
        this.draupnirUserID = draupnirUserID;
        this.draupnirManagementRoomID = draupnirManagementRoomID;
        this.draupnirModerators = draupnirModerators;
        this.log = log;
        // nothing to do.
    }
    static async extractMembersFromManagementRoom(managementRoom, draupnirUserID, membershipManager) {
        const membershipRevisionIssuer = await membershipManager.getRoomMembershipRevisionIssuer(managementRoom);
        if ((0, matrix_protection_suite_1.isError)(membershipRevisionIssuer)) {
            return membershipRevisionIssuer;
        }
        const revision = membershipRevisionIssuer.ok.currentRevision;
        return (0, typescript_result_1.Ok)([...revision.members()]
            .filter((member) => member.membership === "join")
            .map((member) => member.userID)
            .filter((userID) => userID !== draupnirUserID));
    }
    static async createNotificationRoomFromDraupnir(draupnir, description, settings, notificationRoomSettingName, notificationRoomName, log) {
        const moderators = await NotificationRoomCreator.extractMembersFromManagementRoom(draupnir.managementRoom, draupnir.clientUserID, draupnir.roomMembershipManager);
        if ((0, matrix_protection_suite_1.isError)(moderators)) {
            return moderators.elaborate("Unable to find the draupnir moderators");
        }
        const notificationRoomCreator = new NotificationRoomCreator(draupnir.protectedRoomsSet.protectedRoomsManager, async function (roomID) {
            const newSettings = description.protectionSettings
                .toMirror()
                .setValue(settings, notificationRoomSettingName, roomID);
            if ((0, matrix_protection_suite_1.isError)(newSettings)) {
                return newSettings;
            }
            const result = await draupnir.protectedRoomsSet.protections.changeProtectionSettings(description, draupnir.protectedRoomsSet, draupnir, newSettings.ok);
            if ((0, matrix_protection_suite_1.isError)(result)) {
                return result.elaborate("Unable to add the notification room to the protection settings");
            }
            return result;
        }, draupnir.clientPlatform.toRoomCreator(), draupnir.clientPlatform.toRoomInviter(), draupnir.clientPlatform.toRoomStateEventSender(), notificationRoomName, draupnir.clientUserID, draupnir.managementRoomID, moderators.ok, log);
        return await notificationRoomCreator.createMissingNotificationRoom();
    }
    async createMissingNotificationRoom() {
        const roomTitle = `${this.draupnirUserID}'s ${this.roomName}`;
        const newRoom = await this.roomCreateCapability.createRoom({
            preset: "private_chat",
            name: roomTitle,
        });
        if ((0, matrix_protection_suite_1.isError)(newRoom)) {
            this.log.error(`Failed to create notification room for ${this.roomName}`, newRoom.error);
            return newRoom;
        }
        const protectRoomResult = await this.protectedRoomsManager.addRoom(newRoom.ok);
        if ((0, matrix_protection_suite_1.isError)(protectRoomResult)) {
            this.log.error(`Failed to protect notification room for ${this.roomName}`, protectRoomResult.error);
            return protectRoomResult;
        }
        const protectionEnableResult = await this.settingChangeCB(newRoom.ok.toRoomIDOrAlias());
        const restrictionResult = await this.roomStateEventCapability.sendStateEvent(newRoom.ok, "m.room.join_rules", "", {
            join_rule: "restricted",
            allow: [
                {
                    room_id: this.draupnirManagementRoomID,
                    type: "m.room_membership",
                },
            ],
        });
        if ((0, matrix_protection_suite_1.isError)(restrictionResult)) {
            this.log.error(`Failed to restrict notification room for ${this.roomName}`, restrictionResult.error);
            return restrictionResult;
        }
        if ((0, matrix_protection_suite_1.isError)(protectionEnableResult)) {
            this.log.error(`Failed to enable protection for notification room for ${this.roomName}`, protectionEnableResult.error);
            return protectionEnableResult;
        }
        // We invite seperate to the /createRoom call because otherwise the entire /createRoom
        // call will fail if just one user couldn't be invited. Which is really bad.
        // This happens when servers are no longer reachable.
        for (const invitee of this.draupnirModerators) {
            const inviteResult = await this.roomInviter.inviteUser(newRoom.ok, invitee);
            if ((0, matrix_protection_suite_1.isError)(inviteResult)) {
                this.log.error(`Failed to invite moderator ${invitee} to notification room for ${this.roomName}`, inviteResult.error);
            }
        }
        return protectionEnableResult;
    }
}
exports.NotificationRoomCreator = NotificationRoomCreator;
//# sourceMappingURL=NotificationRoom.js.map