"use strict";
// SPDX-FileCopyrightText: 2025 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.SynapseHTTPAntispamRoomExplorer = void 0;
const matrix_protection_suite_1 = require("matrix-protection-suite");
const log = new matrix_protection_suite_1.Logger("SynapseHTTPAntispamRoomDiscovery");
class SynapseHTTPAntispamRoomExplorer {
    constructor(synapseHTTPAntispam, discoveredRoomStore, batchWaitPeriodMS = 500) {
        this.synapseHTTPAntispam = synapseHTTPAntispam;
        this.discoveredRoomStore = discoveredRoomStore;
        this.batchWaitPeriodMS = batchWaitPeriodMS;
        this.forwardDiscoveredBatch = async function (rawEntries) {
            const discoveryResult = await this.discoveredRoomStore.checkRoomsDiscovered(rawEntries.map(([roomID]) => ({ roomID })));
            if ((0, matrix_protection_suite_1.isError)(discoveryResult)) {
                log.error("Unxpected error while trying to store undiscovered rooms", discoveryResult.error);
                return;
            }
        }.bind(this);
        this.handleCheckEventForSpam = function ({ event }) {
            if (!this.discoveredRoomStore.isRoomDiscovered(event.room_id)) {
                this.batcher.add(event.room_id);
            }
        }.bind(this);
        this.handleUserMayInvite = function ({ room_id }) {
            if (!this.discoveredRoomStore.isRoomDiscovered(room_id)) {
                this.batcher.add(room_id);
            }
        }.bind(this);
        this.handleUserMayJoin = function ({ room }) {
            if (!this.discoveredRoomStore.isRoomDiscovered(room)) {
                this.batcher.add(room);
            }
        }.bind(this);
        synapseHTTPAntispam.checkEventForSpamHandles.registerNonBlockingHandle(this.handleCheckEventForSpam);
        synapseHTTPAntispam.userMayInviteHandles.registerNonBlockingHandle(this.handleUserMayInvite);
        synapseHTTPAntispam.userMayJoinRoomHandles.registerNonBlockingHandle(this.handleUserMayJoin);
        this.batcher = new matrix_protection_suite_1.StandardBatcher(() => new matrix_protection_suite_1.ConstantPeriodItemBatch(this.forwardDiscoveredBatch, { waitPeriodMS: this.batchWaitPeriodMS }));
    }
    unregisterListeners() {
        this.synapseHTTPAntispam.checkEventForSpamHandles.unregisterHandle(this.handleCheckEventForSpam);
        this.synapseHTTPAntispam.userMayInviteHandles.unregisterHandle(this.handleUserMayInvite);
        this.synapseHTTPAntispam.userMayJoinRoomHandles.unregisterHandle(this.handleUserMayJoin);
        this.batcher.dispose();
    }
    handlePolicyChange(_change) {
        // nothing to do.
        // We don't need to refresh when rules are added or removed.
    }
}
exports.SynapseHTTPAntispamRoomExplorer = SynapseHTTPAntispamRoomExplorer;
//# sourceMappingURL=SynapseHTTPAntispamRoomExplorer.js.map