"use strict";
// Copyright 2019 - 2021 The Matrix.org Foundation C.I.C.
// Copyright 2022 - 2024 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0 AND Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from mjolnir
// https://github.com/matrix-org/mjolnir
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
exports.WatchRoomsOnInvite = void 0;
exports.isRevisionLikelyPolicyRoom = isRevisionLikelyPolicyRoom;
const matrix_protection_suite_1 = require("matrix-protection-suite");
const mps_interface_adaptor_1 = require("@the-draupnir-project/mps-interface-adaptor");
const typebox_1 = require("@sinclair/typebox");
const interface_manager_1 = require("@the-draupnir-project/interface-manager");
const log = new matrix_protection_suite_1.Logger("WatchRoomsOnInvite");
const WATCH_LISTS_ON_INVITE_PROMPT_LISTENER = "me.marewolf.draupnir.watch_rooms_on_invite";
// would be nice to be able to use presentation types here idk.
const WatchRoomsOnInvitePromptContext = typebox_1.Type.Object({
    invited_room: matrix_protection_suite_1.RoomIDPermalinkSchema,
});
function isRevisionContainingPolicies(revision) {
    return revision.getStateEventsOfTypes(matrix_protection_suite_1.ALL_RULE_TYPES).length > 0;
}
function isRevisionContainingShortcode(revision) {
    return revision.getStateEvent(matrix_protection_suite_1.MJOLNIR_SHORTCODE_EVENT_TYPE, "") !== undefined;
}
function isRevisionLikelyPolicyRoom(revision) {
    return (isRevisionContainingPolicies(revision) ||
        isRevisionContainingShortcode(revision));
}
class WatchRoomsOnInvite {
    constructor(draupnir, protectedRoomsSet) {
        this.draupnir = draupnir;
        this.protectedRoomsSet = protectedRoomsSet;
        this.watchPromptListener = this.watchListener.bind(this);
        this.draupnir.reactionHandler.on(WATCH_LISTS_ON_INVITE_PROMPT_LISTENER, this.watchPromptListener);
    }
    handleProtectionDisable() {
        this.draupnir.reactionHandler.off(WATCH_LISTS_ON_INVITE_PROMPT_LISTENER, this.watchPromptListener);
    }
    promptIfPossiblePolicyRoom(candidateRoom, invitation) {
        void (0, matrix_protection_suite_1.Task)((async () => {
            const stateRevisionIssuer = await this.draupnir.roomStateManager.getRoomStateRevisionIssuer(candidateRoom);
            if ((0, matrix_protection_suite_1.isError)(stateRevisionIssuer)) {
                return stateRevisionIssuer.elaborate(`Unable to fetch the room state revision issuer to check if newly joined room was a policy room.`);
            }
            if (!isRevisionLikelyPolicyRoom(stateRevisionIssuer.ok.currentRevision)) {
                return (0, matrix_protection_suite_1.Ok)(undefined);
            }
            const promptResult = await this.promptWatchPolicyRoom(candidateRoom, invitation);
            if ((0, matrix_protection_suite_1.isError)(promptResult)) {
                return promptResult.elaborate(`Unable to send prompt to ask if Draupnir should watch a policy room`);
            }
            return (0, matrix_protection_suite_1.Ok)(undefined);
        })());
    }
    async promptWatchPolicyRoom(candidateRoom, invitation) {
        const renderPromptWatch = () => (interface_manager_1.DeadDocumentJSX.JSXFactory("root", null,
            (0, mps_interface_adaptor_1.renderMentionPill)(invitation.sender, invitation.sender),
            " has invited me to a policy room  ",
            (0, mps_interface_adaptor_1.renderRoomPill)(candidateRoom),
            ", would you like Draupnir to watch this room as a policy list?"));
        const reactionMap = new Map(Object.entries({ OK: "OK", Cancel: "Cancel" }));
        const promptSendResult = await (0, mps_interface_adaptor_1.sendMatrixEventsFromDeadDocument)(this.draupnir.clientPlatform.toRoomMessageSender(), this.draupnir.managementRoomID, renderPromptWatch(), {
            additionalContent: this.draupnir.reactionHandler.createAnnotation(WATCH_LISTS_ON_INVITE_PROMPT_LISTENER, reactionMap, {
                invited_room: candidateRoom.toPermalink(),
            }),
        });
        if ((0, matrix_protection_suite_1.isError)(promptSendResult)) {
            log.error(`Could not send the prompt to watch the policy room: ${candidateRoom.toPermalink()}`, promptSendResult.error);
            return promptSendResult.elaborate("Could not send the prompt to watch the policy room");
        }
        const promptEventID = promptSendResult.ok.at(0);
        if (promptEventID === undefined) {
            throw new TypeError(`We should have an eventID for the event that we just sent...`);
        }
        await this.draupnir.reactionHandler.addReactionsToEvent(this.draupnir.managementRoomID, promptEventID, reactionMap);
        return (0, matrix_protection_suite_1.Ok)(undefined);
    }
    watchListener(key, _item, rawContext, _reactionMap, promptEvent) {
        if (key === "Cancel") {
            void (0, matrix_protection_suite_1.Task)(this.draupnir.reactionHandler.cancelPrompt(promptEvent));
            return;
        }
        if (key !== "OK") {
            return;
        }
        const context = matrix_protection_suite_1.Value.Decode(WatchRoomsOnInvitePromptContext, rawContext);
        if ((0, matrix_protection_suite_1.isError)(context)) {
            log.error(`Could not decode context from prompt event`, context.error);
            (0, mps_interface_adaptor_1.renderActionResultToEvent)(this.draupnir.clientPlatform.toRoomMessageSender(), this.draupnir.clientPlatform.toRoomReactionSender(), promptEvent, context);
            return;
        }
        void (0, matrix_protection_suite_1.Task)((async () => {
            const resolvedRoom = await this.draupnir.clientPlatform
                .toRoomResolver()
                .resolveRoom(context.ok.invited_room);
            if ((0, matrix_protection_suite_1.isError)(resolvedRoom)) {
                resolvedRoom.elaborate(`Could not resolve the policy room to watch from the MatrixRoomReference: ${context.ok.invited_room.toPermalink()}.`);
                (0, mps_interface_adaptor_1.renderActionResultToEvent)(this.draupnir.clientPlatform.toRoomMessageSender(), this.draupnir.clientPlatform.toRoomReactionSender(), promptEvent, resolvedRoom);
                return;
            }
            const addResult = await this.protectedRoomsSet.watchedPolicyRooms.watchPolicyRoomDirectly(resolvedRoom.ok);
            if ((0, matrix_protection_suite_1.isError)(addResult)) {
                addResult.elaborate(`Could not watch the policy room: ${resolvedRoom.ok.toPermalink()}`);
                (0, mps_interface_adaptor_1.renderActionResultToEvent)(this.draupnir.clientPlatform.toRoomMessageSender(), this.draupnir.clientPlatform.toRoomReactionSender(), promptEvent, addResult);
                return;
            }
            (0, mps_interface_adaptor_1.renderActionResultToEvent)(this.draupnir.clientPlatform.toRoomMessageSender(), this.draupnir.clientPlatform.toRoomReactionSender(), promptEvent, addResult);
            void (0, matrix_protection_suite_1.Task)(this.draupnir.reactionHandler.completePrompt(promptEvent.room_id, promptEvent.event_id));
        })());
    }
}
exports.WatchRoomsOnInvite = WatchRoomsOnInvite;
//# sourceMappingURL=WatchRoomsOnInvite.js.map