"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UnlistedUserRedactionQueue = void 0;
// Copyright 2022 Gnuxie <Gnuxie@protonmail.com>
// Copyright 2019 - 2021 The Matrix.org Foundation C.I.C.
//
// SPDX-License-Identifier: AFL-3.0 AND Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from mjolnir
// https://github.com/matrix-org/mjolnir
// </text>
const matrix_bot_sdk_1 = require("matrix-bot-sdk");
const matrix_basic_types_1 = require("@the-draupnir-project/matrix-basic-types");
/**
 * A queue of users who have been flagged for redaction typically by the flooding or image protection.
 * Specifically any new events sent by a queued user will be redacted.
 * This does not handle previously sent events, for that see the `EventRedactionQueue`.
 * These users are not listed as banned in any watch list and so may continue
 * to view a room until a moderator can investigate.
 */
class UnlistedUserRedactionQueue {
    constructor() {
        this.usersToRedact = new Set();
    }
    addUser(userID) {
        this.usersToRedact.add(userID);
    }
    removeUser(userID) {
        this.usersToRedact.delete(userID);
    }
    isUserQueued(userID) {
        return this.usersToRedact.has(userID);
    }
    async handleEvent(roomID, event, draupnir) {
        if (this.isUserQueued(event["sender"])) {
            const permalink = matrix_basic_types_1.Permalinks.forEvent(roomID, event["event_id"]);
            try {
                matrix_bot_sdk_1.LogService.info("AutomaticRedactionQueue", `Redacting event because the user is listed as bad: ${permalink}`);
                if (!draupnir.config.noop) {
                    await draupnir.client.redactEvent(roomID, event["event_id"]);
                }
                else {
                    await draupnir.managementRoomOutput.logMessage(matrix_bot_sdk_1.LogLevel.WARN, "AutomaticRedactionQueue", `Tried to redact ${permalink} but Draupnir is running in no-op mode`);
                }
            }
            catch (e) {
                await draupnir.managementRoomOutput.logMessage(matrix_bot_sdk_1.LogLevel.WARN, "AutomaticRedactionQueue", `Unable to redact message: ${permalink}`);
                matrix_bot_sdk_1.LogService.warn("AutomaticRedactionQueue", e);
            }
        }
    }
}
exports.UnlistedUserRedactionQueue = UnlistedUserRedactionQueue;
//# sourceMappingURL=UnlistedUserRedactionQueue.js.map