"use strict";
// SPDX-FileCopyrightText: 2024 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.StandardPersistentConfigEditor = void 0;
const typescript_result_1 = require("@gnuxie/typescript-result");
const matrix_protection_suite_1 = require("matrix-protection-suite");
const matrix_protection_suite_for_matrix_bot_sdk_1 = require("matrix-protection-suite-for-matrix-bot-sdk");
const SafeModeCause_1 = require("./SafeModeCause");
class StandardPersistentConfigEditor {
    constructor(client) {
        this.configAdaptors = [];
        // We do some sweepy sweepy casting here because the ConfigMirror has methods
        // that accept a specific shape, and obviously that means the type parameter
        // becomes contravariant. I think the only way to fix this is to make the mirrors
        // only work with the general shape rather than the specific one, in the way that
        // the `remove` methods do, but I'm not convinced that works either, as those
        // methods accept a Record that at least has the keys from the specific shape
        // of the config. OK that's not why, because I tried to remove the toMirror method.
        // I don't understand why it won't work then...
        this.configAdaptors = [
            new matrix_protection_suite_1.StandardPersistentConfigData(matrix_protection_suite_1.MjolnirPolicyRoomsDescription, new matrix_protection_suite_for_matrix_bot_sdk_1.BotSDKAccountDataConfigBackend(client, matrix_protection_suite_1.MJOLNIR_WATCHED_POLICY_ROOMS_EVENT_TYPE)),
            new matrix_protection_suite_1.StandardPersistentConfigData(matrix_protection_suite_1.MjolnirProtectedRoomsDescription, new matrix_protection_suite_for_matrix_bot_sdk_1.BotSDKAccountDataConfigBackend(client, matrix_protection_suite_1.MJOLNIR_PROTECTED_ROOMS_EVENT_TYPE)),
            new matrix_protection_suite_1.StandardPersistentConfigData(matrix_protection_suite_1.MjolnirEnabledProtectionsDescription, new matrix_protection_suite_for_matrix_bot_sdk_1.BotSDKAccountDataConfigBackend(client, matrix_protection_suite_1.MjolnirEnabledProtectionsEventType)),
        ];
    }
    getConfigAdaptors() {
        return this.configAdaptors;
    }
    async requestConfigStatus() {
        const info = [];
        for (const adaptor of this.configAdaptors) {
            const dataResult = await adaptor.requestParsedConfig();
            if ((0, typescript_result_1.isError)(dataResult)) {
                if (dataResult.error instanceof matrix_protection_suite_1.ConfigParseError) {
                    info.push({
                        description: adaptor.description,
                        data: dataResult.error.config,
                        error: dataResult.error,
                    });
                }
                else {
                    return dataResult;
                }
            }
            else {
                info.push({
                    description: adaptor.description,
                    data: dataResult.ok,
                    error: undefined,
                });
            }
        }
        return (0, typescript_result_1.Ok)(info);
    }
    async supplementStatusWithSafeModeCause(cause) {
        const info = await this.requestConfigStatus();
        if ((0, typescript_result_1.isError)(info)) {
            return info;
        }
        if (cause.reason === SafeModeCause_1.SafeModeReason.ByRequest) {
            return (0, typescript_result_1.Ok)(info.ok);
        }
        if (!(cause.error instanceof matrix_protection_suite_1.ConfigPropertyUseError)) {
            return (0, typescript_result_1.Ok)(info.ok);
        }
        const relevantStatus = info.ok.find((status) => status.description ===
            cause.error.configDescription);
        if (relevantStatus === undefined) {
            throw new TypeError("The cause of the safe mode error was not found in the configuration status.");
        }
        relevantStatus.error = new matrix_protection_suite_1.ConfigParseError("There was a problem when using a property in the configuration.", relevantStatus.description, [cause.error], relevantStatus.data);
        return (0, typescript_result_1.Ok)(info.ok);
    }
}
exports.StandardPersistentConfigEditor = StandardPersistentConfigEditor;
//# sourceMappingURL=PersistentConfigEditor.js.map