"use strict";
// SPDX-FileCopyrightText: 2024 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.SafeModeRecoverCommand = void 0;
exports.getRecoveryOptions = getRecoveryOptions;
exports.sendAndAnnotateWithRecoveryOptions = sendAndAnnotateWithRecoveryOptions;
const typescript_result_1 = require("@gnuxie/typescript-result");
const interface_manager_1 = require("@the-draupnir-project/interface-manager");
const SafeModeCause_1 = require("../SafeModeCause");
const matrix_protection_suite_1 = require("matrix-protection-suite");
const SafeModeAdaptor_1 = require("./SafeModeAdaptor");
const PersistentConfigEditor_1 = require("../PersistentConfigEditor");
const PersistentConfigRenderer_1 = require("../PersistentConfigRenderer");
const mps_interface_adaptor_1 = require("@the-draupnir-project/mps-interface-adaptor");
exports.SafeModeRecoverCommand = (0, interface_manager_1.describeCommand)({
    summary: "Select an availale recovery option to enact.",
    parameters: (0, interface_manager_1.tuple)({
        name: "recovery option",
        acceptor: interface_manager_1.StringPresentationType,
        description: "The recovery option to enact, e.g. 1.",
    }),
    keywords: {
        keywordDescriptions: {
            "no-confirm": {
                description: "Do not prompt for confirmation.",
                isFlag: true,
            },
        },
    },
    async executor(safeModeDraupnir, _info, keywords, _rest, optionDesignator) {
        const optionNumber = (() => {
            try {
                return (0, typescript_result_1.Ok)(parseInt(optionDesignator));
            }
            catch {
                return typescript_result_1.ResultError.Result(`Unable to parse the recovery option from ${optionDesignator}`);
            }
        })();
        if ((0, typescript_result_1.isError)(optionNumber)) {
            return optionNumber;
        }
        const options = getRecoveryOptions(safeModeDraupnir);
        const selectedOption = options[optionNumber.ok - 1];
        if (selectedOption === undefined) {
            return typescript_result_1.ResultError.Result(`No recovery option with the number ${optionNumber.ok} exists.`);
        }
        if (!keywords.getKeywordValue("no-confirm", false)) {
            return (0, typescript_result_1.Ok)({
                recoveryOption: selectedOption,
                configStatus: [],
            });
        }
        const recoveryResult = await selectedOption.recover();
        if ((0, typescript_result_1.isError)(recoveryResult)) {
            return recoveryResult;
        }
        // Now we're going to get the config again to show the user the outcome of recovery.
        const editor = new PersistentConfigEditor_1.StandardPersistentConfigEditor(safeModeDraupnir.client);
        const configStatus = await editor.requestConfigStatus();
        if ((0, typescript_result_1.isError)(configStatus)) {
            return configStatus.elaborate("The recovery option has been applied successfully. There was an error when trying to fetch the status of the persistent config.");
        }
        return (0, typescript_result_1.Ok)({
            configStatus: configStatus.ok,
            recoveryOption: selectedOption,
        });
    },
});
SafeModeAdaptor_1.SafeModeInterfaceAdaptor.describeRenderer(exports.SafeModeRecoverCommand, {
    confirmationPromptJSXRenderer(result) {
        if ((0, typescript_result_1.isError)(result)) {
            return (0, typescript_result_1.Ok)(undefined);
        }
        const { recoveryOption } = result.ok;
        return (0, typescript_result_1.Ok)(interface_manager_1.DeadDocumentJSX.JSXFactory("root", null,
            interface_manager_1.DeadDocumentJSX.JSXFactory("h4", null, "You are about to use the following recovery option:"),
            interface_manager_1.DeadDocumentJSX.JSXFactory("p", null, recoveryOption.description),
            interface_manager_1.DeadDocumentJSX.JSXFactory("p", null, "Please confirm that you wish to proceed.")));
    },
    JSXRenderer(result) {
        if ((0, typescript_result_1.isError)(result)) {
            return (0, typescript_result_1.Ok)(undefined);
        }
        const { configStatus, recoveryOption } = result.ok;
        return (0, typescript_result_1.Ok)(interface_manager_1.DeadDocumentJSX.JSXFactory("root", null,
            interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null,
                "The following recovery option has been applied successfully:",
                interface_manager_1.DeadDocumentJSX.JSXFactory("br", null),
                recoveryOption.description,
                interface_manager_1.DeadDocumentJSX.JSXFactory("br", null),
                PersistentConfigRenderer_1.StandardPersistentConfigRenderer.renderAdaptorStatus(configStatus),
                interface_manager_1.DeadDocumentJSX.JSXFactory("br", null),
                "You may now restart Draupnir with ",
                interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, "!draupnir restart"))));
    },
});
function getRecoveryOptions(safeModeDraupnir) {
    return safeModeDraupnir.cause.reason === SafeModeCause_1.SafeModeReason.ByRequest
        ? []
        : safeModeDraupnir.cause.error instanceof matrix_protection_suite_1.ConfigRecoverableError
            ? safeModeDraupnir.cause.error.recoveryOptions
            : [];
}
async function sendAndAnnotateWithRecoveryOptions(safeModeDraupnir, document, { replyToEvent }) {
    const reactionMap = mps_interface_adaptor_1.MatrixReactionHandler.createItemizedReactionMap(getRecoveryOptions(safeModeDraupnir).map((_option, index) => String(index + 1)));
    const sendResult = await (0, mps_interface_adaptor_1.sendMatrixEventsFromDeadDocument)(safeModeDraupnir.clientPlatform.toRoomMessageSender(), safeModeDraupnir.commandRoomID, document, {
        replyToEvent,
        additionalContent: safeModeDraupnir.reactionHandler.createAnnotation(mps_interface_adaptor_1.ARGUMENT_PROMPT_LISTENER, reactionMap, {
            command_designator: ["draupnir", "recover"],
            read_items: [],
        }),
    });
    if ((0, typescript_result_1.isError)(sendResult)) {
        return sendResult;
    }
    if (sendResult.ok[0] === undefined) {
        throw new TypeError(`Something is really wrong with the code`);
    }
    await safeModeDraupnir.reactionHandler.addReactionsToEvent(safeModeDraupnir.commandRoomID, sendResult.ok[0], reactionMap);
    return (0, typescript_result_1.Ok)(undefined);
}
//# sourceMappingURL=RecoverCommand.js.map