const tslib = require('tslib');
const electron = require('electron');
const os = require('os');
const electronNormalize = require('../electron-normalize.js');
const merge = require('../../common/merge.js');

const DEFAULT_OPTIONS = {
    cpu: true,
    screen: true,
    memory: true,
    language: true,
};
/** Adds Electron context to events and normalises paths. */
class AdditionalContext {
    constructor(options = {}) {
        this._lazyDeviceContext = {};
        this.name = AdditionalContext.id;
        this._options = Object.assign(Object.assign({}, DEFAULT_OPTIONS), options);
    }
    /** @inheritDoc */
    setupOnce(addGlobalEventProcessor) {
        addGlobalEventProcessor((event) => tslib.__awaiter(this, void 0, void 0, function* () { return this._addAdditionalContext(event); }));
        // Some metrics are only available after app ready so we lazily load them
        void electronNormalize.whenAppReady.then(() => {
            const { language, screen } = this._options;
            if (language) {
                this._lazyDeviceContext.language = electron.app.getLocale();
            }
            if (screen) {
                this._setPrimaryDisplayInfo();
                electron.screen.on('display-metrics-changed', () => {
                    this._setPrimaryDisplayInfo();
                });
            }
        });
    }
    /** Adds additional context to event */
    _addAdditionalContext(event) {
        const device = this._lazyDeviceContext;
        const { memory, cpu } = this._options;
        if (memory) {
            const { total, free } = process.getSystemMemoryInfo();
            device.memory_size = total * 1024;
            device.free_memory = free * 1024;
        }
        if (cpu) {
            const cpuInfo = os.cpus();
            if (cpuInfo && cpuInfo.length) {
                const firstCpu = cpuInfo[0];
                device.processor_count = cpuInfo.length;
                device.cpu_description = firstCpu.model;
                device.processor_frequency = firstCpu.speed;
                if (electron.app.runningUnderARM64Translation) {
                    device.machine_arch = 'arm64';
                }
            }
        }
        return merge.mergeEvents(event, { contexts: { device } });
    }
    /** Sets the display info */
    _setPrimaryDisplayInfo() {
        const display = electron.screen.getPrimaryDisplay();
        const width = Math.floor(display.size.width * display.scaleFactor);
        const height = Math.floor(display.size.height * display.scaleFactor);
        this._lazyDeviceContext.screen_density = display.scaleFactor;
        this._lazyDeviceContext.screen_resolution = `${width}x${height}`;
    }
}
/** @inheritDoc */
AdditionalContext.id = 'AdditionalContext';

exports.AdditionalContext = AdditionalContext;
//# sourceMappingURL=additional-context.js.map
