import { defaultIntegrations as defaultIntegrations$1, init as init$1 } from '@sentry/browser';
import { logger } from '@sentry/utils';
import { enableAnrRendererMessages } from './anr.js';
import { electronRendererStackParser } from './stack-parse.js';
import { makeRendererTransport } from './transport.js';
import { ScopeToMain } from './integrations/scope-to-main.js';
import { ensureProcess } from '../common/process.js';

/* eslint-disable no-restricted-globals */
const defaultIntegrations = [...defaultIntegrations$1, new ScopeToMain()];
/**
 * Initialize Sentry in the Electron renderer process
 * @param options SDK options
 * @param originalInit Optional init function for a specific framework SDK
 * @returns
 */
function init(options = {}, 
// This parameter name ensures that TypeScript error messages contain a hint for fixing SDK version mismatches
originalInit = init$1) {
    ensureProcess('renderer');
    // Ensure the browser SDK is only init'ed once.
    if (window === null || window === void 0 ? void 0 : window.__SENTRY__RENDERER_INIT__) {
        logger.warn(`The browser SDK has already been initialized.
If init has been called in the preload and contextIsolation is disabled, is not required to call init in the renderer`);
        return;
    }
    window.__SENTRY__RENDERER_INIT__ = true;
    // We don't want browser session tracking enabled by default because we already have Electron
    // specific session tracking from the main process.
    if (options.autoSessionTracking === undefined) {
        options.autoSessionTracking = false;
    }
    // Disable client reports for renderer as the sdk should only send
    // events using the main process.
    options.sendClientReports = false;
    if (options.defaultIntegrations === undefined) {
        options.defaultIntegrations = defaultIntegrations;
    }
    if (options.stackParser === undefined) {
        options.stackParser = electronRendererStackParser;
    }
    if (options.dsn === undefined) {
        // Events are sent via the main process but browser SDK wont start without dsn
        options.dsn = 'https://12345@dummy.dsn/12345';
    }
    if (options.transport === undefined) {
        options.transport = makeRendererTransport;
    }
    if (options.anrDetection) {
        enableAnrRendererMessages(options.anrDetection === true ? {} : options.anrDetection);
    }
    // We only handle initialScope in the main process otherwise it can cause race conditions over IPC
    delete options.initialScope;
    originalInit(options);
}

export { defaultIntegrations, init };
//# sourceMappingURL=sdk.js.map
