#include <GLES3/gl32.h>
#include <hyprgraphics/color/Color.hpp>
#include <hyprutils/string/String.hpp>
#include <hyprutils/path/Path.hpp>
#include <numbers>
#include <random>
#include <pango/pangocairo.h>
#include "OpenGL.hpp"
#include "Renderer.hpp"
#include "../Compositor.hpp"
#include "../helpers/MiscFunctions.hpp"
#include "../helpers/CursorShapes.hpp"
#include "../config/ConfigValue.hpp"
#include "../config/ConfigManager.hpp"
#include "../managers/PointerManager.hpp"
#include "../desktop/view/LayerSurface.hpp"
#include "../desktop/state/FocusState.hpp"
#include "../protocols/LayerShell.hpp"
#include "../protocols/core/Compositor.hpp"
#include "../protocols/ColorManagement.hpp"
#include "../protocols/types/ColorManagement.hpp"
#include "../managers/HookSystemManager.hpp"
#include "../managers/input/InputManager.hpp"
#include "../managers/eventLoop/EventLoopManager.hpp"
#include "../managers/CursorManager.hpp"
#include "../helpers/fs/FsUtils.hpp"
#include "../helpers/env/Env.hpp"
#include "../helpers/MainLoopExecutor.hpp"
#include "../i18n/Engine.hpp"
#include "debug/HyprNotificationOverlay.hpp"
#include "hyprerror/HyprError.hpp"
#include "pass/TexPassElement.hpp"
#include "pass/RectPassElement.hpp"
#include "pass/PreBlurElement.hpp"
#include "pass/ClearPassElement.hpp"
#include "render/Shader.hpp"
#include "AsyncResourceGatherer.hpp"
#include <ranges>
#include <algorithm>
#include <fstream>
#include <string>
#include <xf86drm.h>
#include <fcntl.h>
#include <gbm.h>
#include <filesystem>
#include <cstring>
#include "./shaders/Shaders.hpp"

using namespace Hyprutils::OS;
using namespace NColorManagement;

const std::vector<const char*> ASSET_PATHS = {
#ifdef DATAROOTDIR
    DATAROOTDIR,
#endif
    "/usr/share",
    "/usr/local/share",
};

static inline void loadGLProc(void* pProc, const char* name) {
    void* proc = rc<void*>(eglGetProcAddress(name));
    if (proc == nullptr) {
        Log::logger->log(Log::CRIT, "[Tracy GPU Profiling] eglGetProcAddress({}) failed", name);
        abort();
    }
    *sc<void**>(pProc) = proc;
}

static enum Hyprutils::CLI::eLogLevel eglLogToLevel(EGLint type) {
    switch (type) {
        case EGL_DEBUG_MSG_CRITICAL_KHR: return Log::CRIT;
        case EGL_DEBUG_MSG_ERROR_KHR: return Log::ERR;
        case EGL_DEBUG_MSG_WARN_KHR: return Log::WARN;
        case EGL_DEBUG_MSG_INFO_KHR: return Log::DEBUG;
        default: return Log::DEBUG;
    }
}

static const char* eglErrorToString(EGLint error) {
    switch (error) {
        case EGL_SUCCESS: return "EGL_SUCCESS";
        case EGL_NOT_INITIALIZED: return "EGL_NOT_INITIALIZED";
        case EGL_BAD_ACCESS: return "EGL_BAD_ACCESS";
        case EGL_BAD_ALLOC: return "EGL_BAD_ALLOC";
        case EGL_BAD_ATTRIBUTE: return "EGL_BAD_ATTRIBUTE";
        case EGL_BAD_CONTEXT: return "EGL_BAD_CONTEXT";
        case EGL_BAD_CONFIG: return "EGL_BAD_CONFIG";
        case EGL_BAD_CURRENT_SURFACE: return "EGL_BAD_CURRENT_SURFACE";
        case EGL_BAD_DISPLAY: return "EGL_BAD_DISPLAY";
        case EGL_BAD_DEVICE_EXT: return "EGL_BAD_DEVICE_EXT";
        case EGL_BAD_SURFACE: return "EGL_BAD_SURFACE";
        case EGL_BAD_MATCH: return "EGL_BAD_MATCH";
        case EGL_BAD_PARAMETER: return "EGL_BAD_PARAMETER";
        case EGL_BAD_NATIVE_PIXMAP: return "EGL_BAD_NATIVE_PIXMAP";
        case EGL_BAD_NATIVE_WINDOW: return "EGL_BAD_NATIVE_WINDOW";
        case EGL_CONTEXT_LOST: return "EGL_CONTEXT_LOST";
    }
    return "Unknown";
}

static void eglLog(EGLenum error, const char* command, EGLint type, EGLLabelKHR thread, EGLLabelKHR obj, const char* msg) {
    Log::logger->log(eglLogToLevel(type), "[EGL] Command {} errored out with {} (0x{}): {}", command, eglErrorToString(error), error, msg);
}

static int openRenderNode(int drmFd) {
    auto renderName = drmGetRenderDeviceNameFromFd(drmFd);
    if (!renderName) {
        // This can happen on split render/display platforms, fallback to
        // primary node
        renderName = drmGetPrimaryDeviceNameFromFd(drmFd);
        if (!renderName) {
            Log::logger->log(Log::ERR, "drmGetPrimaryDeviceNameFromFd failed");
            return -1;
        }
        Log::logger->log(Log::DEBUG, "DRM dev {} has no render node, falling back to primary", renderName);

        drmVersion* render_version = drmGetVersion(drmFd);
        if (render_version && render_version->name) {
            Log::logger->log(Log::DEBUG, "DRM dev versionName", render_version->name);
            if (strcmp(render_version->name, "evdi") == 0) {
                free(renderName); // NOLINT(cppcoreguidelines-no-malloc)
                renderName = strdup("/dev/dri/card0");
            }
            drmFreeVersion(render_version);
        }
    }

    Log::logger->log(Log::DEBUG, "openRenderNode got drm device {}", renderName);

    int renderFD = open(renderName, O_RDWR | O_CLOEXEC);
    if (renderFD < 0)
        Log::logger->log(Log::ERR, "openRenderNode failed to open drm device {}", renderName);

    free(renderName); // NOLINT(cppcoreguidelines-no-malloc)
    return renderFD;
}

void CHyprOpenGLImpl::initEGL(bool gbm) {
    std::vector<EGLint> attrs;
    if (m_exts.KHR_display_reference) {
        attrs.push_back(EGL_TRACK_REFERENCES_KHR);
        attrs.push_back(EGL_TRUE);
    }

    attrs.push_back(EGL_NONE);

    m_eglDisplay = m_proc.eglGetPlatformDisplayEXT(gbm ? EGL_PLATFORM_GBM_KHR : EGL_PLATFORM_DEVICE_EXT, gbm ? m_gbmDevice : m_eglDevice, attrs.data());
    if (m_eglDisplay == EGL_NO_DISPLAY)
        RASSERT(false, "EGL: failed to create a platform display");

    attrs.clear();

    EGLint major, minor;
    if (eglInitialize(m_eglDisplay, &major, &minor) == EGL_FALSE)
        RASSERT(false, "EGL: failed to initialize a platform display");

    const std::string EGLEXTENSIONS = eglQueryString(m_eglDisplay, EGL_EXTENSIONS);

    m_exts.IMG_context_priority               = EGLEXTENSIONS.contains("IMG_context_priority");
    m_exts.EXT_create_context_robustness      = EGLEXTENSIONS.contains("EXT_create_context_robustness");
    m_exts.EXT_image_dma_buf_import           = EGLEXTENSIONS.contains("EXT_image_dma_buf_import");
    m_exts.EXT_image_dma_buf_import_modifiers = EGLEXTENSIONS.contains("EXT_image_dma_buf_import_modifiers");
    m_exts.KHR_context_flush_control          = EGLEXTENSIONS.contains("EGL_KHR_context_flush_control");

    if (m_exts.IMG_context_priority) {
        Log::logger->log(Log::DEBUG, "EGL: IMG_context_priority supported, requesting high");
        attrs.push_back(EGL_CONTEXT_PRIORITY_LEVEL_IMG);
        attrs.push_back(EGL_CONTEXT_PRIORITY_HIGH_IMG);
    }

    if (m_exts.EXT_create_context_robustness) {
        Log::logger->log(Log::DEBUG, "EGL: EXT_create_context_robustness supported, requesting lose on reset");
        attrs.push_back(EGL_CONTEXT_OPENGL_RESET_NOTIFICATION_STRATEGY_EXT);
        attrs.push_back(EGL_LOSE_CONTEXT_ON_RESET_EXT);
    }

    if (m_exts.KHR_context_flush_control) {
        Log::logger->log(Log::DEBUG, "EGL: Using KHR_context_flush_control");
        attrs.push_back(EGL_CONTEXT_RELEASE_BEHAVIOR_KHR);
        attrs.push_back(EGL_CONTEXT_RELEASE_BEHAVIOR_NONE_KHR); // or _FLUSH_KHR
    }

    auto attrsNoVer = attrs;

    attrs.push_back(EGL_CONTEXT_MAJOR_VERSION);
    attrs.push_back(3);
    attrs.push_back(EGL_CONTEXT_MINOR_VERSION);
    attrs.push_back(2);
    attrs.push_back(EGL_NONE);

    m_eglContext = eglCreateContext(m_eglDisplay, EGL_NO_CONFIG_KHR, EGL_NO_CONTEXT, attrs.data());
    if (m_eglContext == EGL_NO_CONTEXT) {
        Log::logger->log(Log::WARN, "EGL: Failed to create a context with GLES3.2, retrying 3.0");

        attrs = attrsNoVer;
        attrs.push_back(EGL_CONTEXT_MAJOR_VERSION);
        attrs.push_back(3);
        attrs.push_back(EGL_CONTEXT_MINOR_VERSION);
        attrs.push_back(0);
        attrs.push_back(EGL_NONE);

        m_eglContext        = eglCreateContext(m_eglDisplay, EGL_NO_CONFIG_KHR, EGL_NO_CONTEXT, attrs.data());
        m_eglContextVersion = EGL_CONTEXT_GLES_3_0;

        if (m_eglContext == EGL_NO_CONTEXT)
            RASSERT(false, "EGL: failed to create a context with either GLES3.2 or 3.0");
    }

    if (m_exts.IMG_context_priority) {
        EGLint priority = EGL_CONTEXT_PRIORITY_MEDIUM_IMG;
        eglQueryContext(m_eglDisplay, m_eglContext, EGL_CONTEXT_PRIORITY_LEVEL_IMG, &priority);
        if (priority != EGL_CONTEXT_PRIORITY_HIGH_IMG)
            Log::logger->log(Log::ERR, "EGL: Failed to obtain a high priority context");
        else
            Log::logger->log(Log::DEBUG, "EGL: Got a high priority context");
    }

    eglMakeCurrent(m_eglDisplay, EGL_NO_SURFACE, EGL_NO_SURFACE, m_eglContext);
}

static bool drmDeviceHasName(const drmDevice* device, const std::string& name) {
    for (size_t i = 0; i < DRM_NODE_MAX; i++) {
        if (!(device->available_nodes & (1 << i)))
            continue;

        if (device->nodes[i] == name)
            return true;
    }
    return false;
}

EGLDeviceEXT CHyprOpenGLImpl::eglDeviceFromDRMFD(int drmFD) {
    EGLint nDevices = 0;
    if (!m_proc.eglQueryDevicesEXT(0, nullptr, &nDevices)) {
        Log::logger->log(Log::ERR, "eglDeviceFromDRMFD: eglQueryDevicesEXT failed");
        return EGL_NO_DEVICE_EXT;
    }

    if (nDevices <= 0) {
        Log::logger->log(Log::ERR, "eglDeviceFromDRMFD: no devices");
        return EGL_NO_DEVICE_EXT;
    }

    std::vector<EGLDeviceEXT> devices;
    devices.resize(nDevices);

    if (!m_proc.eglQueryDevicesEXT(nDevices, devices.data(), &nDevices)) {
        Log::logger->log(Log::ERR, "eglDeviceFromDRMFD: eglQueryDevicesEXT failed (2)");
        return EGL_NO_DEVICE_EXT;
    }

    drmDevice* drmDev = nullptr;
    if (int ret = drmGetDevice(drmFD, &drmDev); ret < 0) {
        Log::logger->log(Log::ERR, "eglDeviceFromDRMFD: drmGetDevice failed");
        return EGL_NO_DEVICE_EXT;
    }

    for (auto const& d : devices) {
        auto devName = m_proc.eglQueryDeviceStringEXT(d, EGL_DRM_DEVICE_FILE_EXT);
        if (!devName)
            continue;

        if (drmDeviceHasName(drmDev, devName)) {
            Log::logger->log(Log::DEBUG, "eglDeviceFromDRMFD: Using device {}", devName);
            drmFreeDevice(&drmDev);
            return d;
        }
    }

    drmFreeDevice(&drmDev);
    Log::logger->log(Log::DEBUG, "eglDeviceFromDRMFD: No drm devices found");
    return EGL_NO_DEVICE_EXT;
}

CHyprOpenGLImpl::CHyprOpenGLImpl() : m_drmFD(g_pCompositor->m_drmRenderNode.fd >= 0 ? g_pCompositor->m_drmRenderNode.fd : g_pCompositor->m_drm.fd) {
    const std::string EGLEXTENSIONS = eglQueryString(EGL_NO_DISPLAY, EGL_EXTENSIONS);

    Log::logger->log(Log::DEBUG, "Supported EGL global extensions: ({}) {}", std::ranges::count(EGLEXTENSIONS, ' '), EGLEXTENSIONS);

    m_exts.KHR_display_reference = EGLEXTENSIONS.contains("KHR_display_reference");

    loadGLProc(&m_proc.glEGLImageTargetRenderbufferStorageOES, "glEGLImageTargetRenderbufferStorageOES");
    loadGLProc(&m_proc.eglCreateImageKHR, "eglCreateImageKHR");
    loadGLProc(&m_proc.eglDestroyImageKHR, "eglDestroyImageKHR");
    loadGLProc(&m_proc.eglQueryDmaBufFormatsEXT, "eglQueryDmaBufFormatsEXT");
    loadGLProc(&m_proc.eglQueryDmaBufModifiersEXT, "eglQueryDmaBufModifiersEXT");
    loadGLProc(&m_proc.glEGLImageTargetTexture2DOES, "glEGLImageTargetTexture2DOES");
    loadGLProc(&m_proc.eglDebugMessageControlKHR, "eglDebugMessageControlKHR");
    loadGLProc(&m_proc.eglGetPlatformDisplayEXT, "eglGetPlatformDisplayEXT");
    loadGLProc(&m_proc.eglCreateSyncKHR, "eglCreateSyncKHR");
    loadGLProc(&m_proc.eglDestroySyncKHR, "eglDestroySyncKHR");
    loadGLProc(&m_proc.eglDupNativeFenceFDANDROID, "eglDupNativeFenceFDANDROID");
    loadGLProc(&m_proc.eglWaitSyncKHR, "eglWaitSyncKHR");

    RASSERT(m_proc.eglCreateSyncKHR, "Display driver doesn't support eglCreateSyncKHR");
    RASSERT(m_proc.eglDupNativeFenceFDANDROID, "Display driver doesn't support eglDupNativeFenceFDANDROID");
    RASSERT(m_proc.eglWaitSyncKHR, "Display driver doesn't support eglWaitSyncKHR");

    if (EGLEXTENSIONS.contains("EGL_EXT_device_base") || EGLEXTENSIONS.contains("EGL_EXT_device_enumeration"))
        loadGLProc(&m_proc.eglQueryDevicesEXT, "eglQueryDevicesEXT");

    if (EGLEXTENSIONS.contains("EGL_EXT_device_base") || EGLEXTENSIONS.contains("EGL_EXT_device_query")) {
        loadGLProc(&m_proc.eglQueryDeviceStringEXT, "eglQueryDeviceStringEXT");
        loadGLProc(&m_proc.eglQueryDisplayAttribEXT, "eglQueryDisplayAttribEXT");
    }

    if (EGLEXTENSIONS.contains("EGL_KHR_debug")) {
        loadGLProc(&m_proc.eglDebugMessageControlKHR, "eglDebugMessageControlKHR");
        static const EGLAttrib debugAttrs[] = {
            EGL_DEBUG_MSG_CRITICAL_KHR, EGL_TRUE, EGL_DEBUG_MSG_ERROR_KHR, EGL_TRUE, EGL_DEBUG_MSG_WARN_KHR, EGL_TRUE, EGL_DEBUG_MSG_INFO_KHR, EGL_TRUE, EGL_NONE,
        };
        m_proc.eglDebugMessageControlKHR(::eglLog, debugAttrs);
    }

    RASSERT(eglBindAPI(EGL_OPENGL_ES_API) != EGL_FALSE, "Couldn't bind to EGL's opengl ES API. This means your gpu driver f'd up. This is not a hyprland issue.");

    bool success = false;
    if (EGLEXTENSIONS.contains("EXT_platform_device") || !m_proc.eglQueryDevicesEXT || !m_proc.eglQueryDeviceStringEXT) {
        m_eglDevice = eglDeviceFromDRMFD(m_drmFD);

        if (m_eglDevice != EGL_NO_DEVICE_EXT) {
            success = true;
            initEGL(false);
        }
    }

    if (!success) {
        Log::logger->log(Log::WARN, "EGL: EXT_platform_device or EGL_EXT_device_query not supported, using gbm");
        if (EGLEXTENSIONS.contains("KHR_platform_gbm")) {
            success = true;
            m_gbmFD = CFileDescriptor{openRenderNode(m_drmFD)};
            if (!m_gbmFD.isValid())
                RASSERT(false, "Couldn't open a gbm fd");

            m_gbmDevice = gbm_create_device(m_gbmFD.get());
            if (!m_gbmDevice)
                RASSERT(false, "Couldn't open a gbm device");

            initEGL(true);
        }
    }

    RASSERT(success, "EGL does not support KHR_platform_gbm or EXT_platform_device, this is an issue with your gpu driver.");

    auto* const EXTENSIONS = rc<const char*>(glGetString(GL_EXTENSIONS));
    RASSERT(EXTENSIONS, "Couldn't retrieve openGL extensions!");

    m_extensions = EXTENSIONS;

    Log::logger->log(Log::DEBUG, "Creating the Hypr OpenGL Renderer!");
    Log::logger->log(Log::DEBUG, "Using: {}", rc<const char*>(glGetString(GL_VERSION)));
    Log::logger->log(Log::DEBUG, "Vendor: {}", rc<const char*>(glGetString(GL_VENDOR)));
    Log::logger->log(Log::DEBUG, "Renderer: {}", rc<const char*>(glGetString(GL_RENDERER)));
    Log::logger->log(Log::DEBUG, "Supported extensions: ({}) {}", std::ranges::count(m_extensions, ' '), m_extensions);

    m_exts.EXT_read_format_bgra = m_extensions.contains("GL_EXT_read_format_bgra");

    RASSERT(m_extensions.contains("GL_EXT_texture_format_BGRA8888"), "GL_EXT_texture_format_BGRA8888 support by the GPU driver is required");

    if (!m_exts.EXT_read_format_bgra)
        Log::logger->log(Log::WARN, "Your GPU does not support GL_EXT_read_format_bgra, this may cause issues with texture importing");
    if (!m_exts.EXT_image_dma_buf_import || !m_exts.EXT_image_dma_buf_import_modifiers)
        Log::logger->log(Log::WARN, "Your GPU does not support DMABUFs, this will possibly cause issues and will take a hit on the performance.");

    const std::string EGLEXTENSIONS_DISPLAY = eglQueryString(m_eglDisplay, EGL_EXTENSIONS);

    Log::logger->log(Log::DEBUG, "Supported EGL display extensions: ({}) {}", std::ranges::count(EGLEXTENSIONS_DISPLAY, ' '), EGLEXTENSIONS_DISPLAY);

#if defined(__linux__)
    m_exts.EGL_ANDROID_native_fence_sync_ext = EGLEXTENSIONS_DISPLAY.contains("EGL_ANDROID_native_fence_sync");

    if (!m_exts.EGL_ANDROID_native_fence_sync_ext)
        Log::logger->log(Log::WARN, "Your GPU does not support explicit sync via the EGL_ANDROID_native_fence_sync extension.");
#else
    m_exts.EGL_ANDROID_native_fence_sync_ext = false;
    Log::logger->log(Log::WARN, "Forcefully disabling explicit sync: BSD is missing support for proper timeline export");
#endif

#ifdef USE_TRACY_GPU

    loadGLProc(&glQueryCounter, "glQueryCounterEXT");
    loadGLProc(&glGetQueryObjectiv, "glGetQueryObjectivEXT");
    loadGLProc(&glGetQueryObjectui64v, "glGetQueryObjectui64vEXT");

#endif

    TRACY_GPU_CONTEXT;

    initDRMFormats();

    initAssets();

    static auto P = g_pHookSystem->hookDynamic("preRender", [&](void* self, SCallbackInfo& info, std::any data) { preRender(std::any_cast<PHLMONITOR>(data)); });

    RASSERT(eglMakeCurrent(m_eglDisplay, EGL_NO_SURFACE, EGL_NO_SURFACE, EGL_NO_CONTEXT), "Couldn't unset current EGL!");

    m_globalTimer.reset();

    pushMonitorTransformEnabled(false);

    static auto addLastPressToHistory = [this](const Vector2D& pos, bool killing, bool touch) {
        // shift the new pos and time in
        std::ranges::rotate(m_pressedHistoryPositions, m_pressedHistoryPositions.end() - 1);
        m_pressedHistoryPositions[0] = pos;

        std::ranges::rotate(m_pressedHistoryTimers, m_pressedHistoryTimers.end() - 1);
        m_pressedHistoryTimers[0].reset();

        // shift killed flag in
        m_pressedHistoryKilled <<= 1;
        m_pressedHistoryKilled |= killing ? 1 : 0;
#if POINTER_PRESSED_HISTORY_LENGTH < 32
        m_pressedHistoryKilled &= (1 >> POINTER_PRESSED_HISTORY_LENGTH) - 1;
#endif

        // shift touch flag in
        m_pressedHistoryTouched <<= 1;
        m_pressedHistoryTouched |= touch ? 1 : 0;
#if POINTER_PRESSED_HISTORY_LENGTH < 32
        m_pressedHistoryTouched &= (1 >> POINTER_PRESSED_HISTORY_LENGTH) - 1;
#endif
    };

    static auto P2 = g_pHookSystem->hookDynamic("mouseButton", [](void* self, SCallbackInfo& info, std::any e) {
        auto E = std::any_cast<IPointer::SButtonEvent>(e);

        if (E.state != WL_POINTER_BUTTON_STATE_PRESSED)
            return;

        addLastPressToHistory(g_pInputManager->getMouseCoordsInternal(), g_pInputManager->getClickMode() == CLICKMODE_KILL, false);
    });

    static auto P3 = g_pHookSystem->hookDynamic("touchDown", [](void* self, SCallbackInfo& info, std::any e) {
        auto E = std::any_cast<ITouch::SDownEvent>(e);

        auto PMONITOR = g_pCompositor->getMonitorFromName(!E.device->m_boundOutput.empty() ? E.device->m_boundOutput : "");

        PMONITOR = PMONITOR ? PMONITOR : Desktop::focusState()->monitor();

        const auto TOUCH_COORDS = PMONITOR->m_position + (E.pos * PMONITOR->m_size);

        addLastPressToHistory(TOUCH_COORDS, g_pInputManager->getClickMode() == CLICKMODE_KILL, true);
    });

    m_finalScreenShader = makeShared<CShader>();
}

CHyprOpenGLImpl::~CHyprOpenGLImpl() {
    if (m_eglDisplay && m_eglContext != EGL_NO_CONTEXT)
        eglDestroyContext(m_eglDisplay, m_eglContext);

    if (m_eglDisplay)
        eglTerminate(m_eglDisplay);

    eglReleaseThread();

    if (m_gbmDevice)
        gbm_device_destroy(m_gbmDevice);
}

std::optional<std::vector<uint64_t>> CHyprOpenGLImpl::getModsForFormat(EGLint format) {
    // TODO: return std::expected when clang supports it

    if (!m_exts.EXT_image_dma_buf_import_modifiers)
        return std::nullopt;

    EGLint len = 0;
    if (!m_proc.eglQueryDmaBufModifiersEXT(m_eglDisplay, format, 0, nullptr, nullptr, &len)) {
        Log::logger->log(Log::ERR, "EGL: Failed to query mods");
        return std::nullopt;
    }

    if (len <= 0)
        return std::vector<uint64_t>{};

    std::vector<uint64_t>   mods;
    std::vector<EGLBoolean> external;

    mods.resize(len);
    external.resize(len);

    m_proc.eglQueryDmaBufModifiersEXT(m_eglDisplay, format, len, mods.data(), external.data(), &len);

    std::vector<uint64_t> result;
    // reserve number of elements to avoid reallocations
    result.reserve(mods.size());

    bool linearIsExternal = false;
    for (size_t i = 0; i < std::min(mods.size(), external.size()); ++i) {
        if (external[i]) {
            if (mods[i] == DRM_FORMAT_MOD_LINEAR)
                linearIsExternal = true;
            continue;
        }

        result.push_back(mods[i]);
    }

    // if the driver doesn't mark linear as external, add it. It's allowed unless the driver says otherwise. (e.g. nvidia)
    if (!linearIsExternal && std::ranges::find(mods, DRM_FORMAT_MOD_LINEAR) == mods.end())
        result.push_back(DRM_FORMAT_MOD_LINEAR);

    return result;
}

void CHyprOpenGLImpl::initDRMFormats() {
    const auto DISABLE_MODS = Env::envEnabled("HYPRLAND_EGL_NO_MODIFIERS");
    if (DISABLE_MODS)
        Log::logger->log(Log::WARN, "HYPRLAND_EGL_NO_MODIFIERS set, disabling modifiers");

    if (!m_exts.EXT_image_dma_buf_import) {
        Log::logger->log(Log::ERR, "EGL: No dmabuf import, DMABufs will not work.");
        return;
    }

    std::vector<EGLint> formats;

    if (!m_exts.EXT_image_dma_buf_import_modifiers || !m_proc.eglQueryDmaBufFormatsEXT) {
        formats.push_back(DRM_FORMAT_ARGB8888);
        formats.push_back(DRM_FORMAT_XRGB8888);
        Log::logger->log(Log::WARN, "EGL: No mod support");
    } else {
        EGLint len = 0;
        m_proc.eglQueryDmaBufFormatsEXT(m_eglDisplay, 0, nullptr, &len);
        formats.resize(len);
        m_proc.eglQueryDmaBufFormatsEXT(m_eglDisplay, len, formats.data(), &len);
    }

    if (formats.empty()) {
        Log::logger->log(Log::ERR, "EGL: Failed to get formats, DMABufs will not work.");
        return;
    }

    Log::logger->log(Log::DEBUG, "Supported DMA-BUF formats:");

    std::vector<SDRMFormat> dmaFormats;
    // reserve number of elements to avoid reallocations
    dmaFormats.reserve(formats.size());

    for (auto const& fmt : formats) {
        std::vector<uint64_t> mods;
        if (!DISABLE_MODS) {
            auto ret = getModsForFormat(fmt);
            if (!ret.has_value())
                continue;

            mods = *ret;
        } else
            mods = {DRM_FORMAT_MOD_LINEAR};

        m_hasModifiers = m_hasModifiers || !mods.empty();

        // EGL can always do implicit modifiers.
        mods.push_back(DRM_FORMAT_MOD_INVALID);

        dmaFormats.push_back(SDRMFormat{
            .drmFormat = fmt,
            .modifiers = mods,
        });

        std::vector<std::pair<uint64_t, std::string>> modifierData;
        // reserve number of elements to avoid reallocations
        modifierData.reserve(mods.size());

        auto fmtName = drmGetFormatName(fmt);
        Log::logger->log(Log::DEBUG, "EGL: GPU Supports Format {} (0x{:x})", fmtName ? fmtName : "?unknown?", fmt);
        for (auto const& mod : mods) {
            auto modName = drmGetFormatModifierName(mod);
            modifierData.emplace_back(std::make_pair<>(mod, modName ? modName : "?unknown?"));
            free(modName); // NOLINT(cppcoreguidelines-no-malloc)
        }
        free(fmtName); // NOLINT(cppcoreguidelines-no-malloc)

        mods.clear();
        std::ranges::sort(modifierData, [](const auto& a, const auto& b) {
            if (a.first == 0)
                return false;
            if (a.second.contains("DCC"))
                return false;
            return true;
        });

        for (auto const& [m, name] : modifierData) {
            Log::logger->log(Log::DEBUG, "EGL: | with modifier {} (0x{:x})", name, m);
            mods.emplace_back(m);
        }
    }

    Log::logger->log(Log::DEBUG, "EGL: {} formats found in total. Some modifiers may be omitted as they are external-only.", dmaFormats.size());

    if (dmaFormats.empty())
        Log::logger->log(
            Log::WARN, "EGL: WARNING: No dmabuf formats were found, dmabuf will be disabled. This will degrade performance, but is most likely a driver issue or a very old GPU.");

    m_drmFormats = dmaFormats;
}

EGLImageKHR CHyprOpenGLImpl::createEGLImage(const Aquamarine::SDMABUFAttrs& attrs) {
    std::array<EGLint, 50> attribs;
    size_t                 idx = 0;

    attribs[idx++] = EGL_WIDTH;
    attribs[idx++] = attrs.size.x;
    attribs[idx++] = EGL_HEIGHT;
    attribs[idx++] = attrs.size.y;
    attribs[idx++] = EGL_LINUX_DRM_FOURCC_EXT;
    attribs[idx++] = attrs.format;

    struct {
        EGLint fd;
        EGLint offset;
        EGLint pitch;
        EGLint modlo;
        EGLint modhi;
    } attrNames[4] = {
        {EGL_DMA_BUF_PLANE0_FD_EXT, EGL_DMA_BUF_PLANE0_OFFSET_EXT, EGL_DMA_BUF_PLANE0_PITCH_EXT, EGL_DMA_BUF_PLANE0_MODIFIER_LO_EXT, EGL_DMA_BUF_PLANE0_MODIFIER_HI_EXT},
        {EGL_DMA_BUF_PLANE1_FD_EXT, EGL_DMA_BUF_PLANE1_OFFSET_EXT, EGL_DMA_BUF_PLANE1_PITCH_EXT, EGL_DMA_BUF_PLANE1_MODIFIER_LO_EXT, EGL_DMA_BUF_PLANE1_MODIFIER_HI_EXT},
        {EGL_DMA_BUF_PLANE2_FD_EXT, EGL_DMA_BUF_PLANE2_OFFSET_EXT, EGL_DMA_BUF_PLANE2_PITCH_EXT, EGL_DMA_BUF_PLANE2_MODIFIER_LO_EXT, EGL_DMA_BUF_PLANE2_MODIFIER_HI_EXT},
        {EGL_DMA_BUF_PLANE3_FD_EXT, EGL_DMA_BUF_PLANE3_OFFSET_EXT, EGL_DMA_BUF_PLANE3_PITCH_EXT, EGL_DMA_BUF_PLANE3_MODIFIER_LO_EXT, EGL_DMA_BUF_PLANE3_MODIFIER_HI_EXT}};

    for (int i = 0; i < attrs.planes; ++i) {
        attribs[idx++] = attrNames[i].fd;
        attribs[idx++] = attrs.fds[i];
        attribs[idx++] = attrNames[i].offset;
        attribs[idx++] = attrs.offsets[i];
        attribs[idx++] = attrNames[i].pitch;
        attribs[idx++] = attrs.strides[i];

        if (m_hasModifiers && attrs.modifier != DRM_FORMAT_MOD_INVALID) {
            attribs[idx++] = attrNames[i].modlo;
            attribs[idx++] = sc<uint32_t>(attrs.modifier & 0xFFFFFFFF);
            attribs[idx++] = attrNames[i].modhi;
            attribs[idx++] = sc<uint32_t>(attrs.modifier >> 32);
        }
    }

    attribs[idx++] = EGL_IMAGE_PRESERVED_KHR;
    attribs[idx++] = EGL_TRUE;
    attribs[idx++] = EGL_NONE;

    RASSERT(idx <= attribs.size(), "createEglImage: attribs array out of bounds.");

    EGLImageKHR image = m_proc.eglCreateImageKHR(m_eglDisplay, EGL_NO_CONTEXT, EGL_LINUX_DMA_BUF_EXT, nullptr, attribs.data());
    if (image == EGL_NO_IMAGE_KHR) {
        Log::logger->log(Log::ERR, "EGL: EGLCreateImageKHR failed: {}", eglGetError());
        return EGL_NO_IMAGE_KHR;
    }

    return image;
}

void CHyprOpenGLImpl::beginSimple(PHLMONITOR pMonitor, const CRegion& damage, SP<CRenderbuffer> rb, CFramebuffer* fb) {
    m_renderData.pMonitor = pMonitor;

    const GLenum RESETSTATUS = glGetGraphicsResetStatus();
    if (RESETSTATUS != GL_NO_ERROR) {
        std::string errStr = "";
        switch (RESETSTATUS) {
            case GL_GUILTY_CONTEXT_RESET: errStr = "GL_GUILTY_CONTEXT_RESET"; break;
            case GL_INNOCENT_CONTEXT_RESET: errStr = "GL_INNOCENT_CONTEXT_RESET"; break;
            case GL_UNKNOWN_CONTEXT_RESET: errStr = "GL_UNKNOWN_CONTEXT_RESET"; break;
            default: errStr = "UNKNOWN??"; break;
        }
        RASSERT(false, "Aborting, glGetGraphicsResetStatus returned {}. Cannot continue until proper GPU reset handling is implemented.", errStr);
        return;
    }

    TRACY_GPU_ZONE("RenderBeginSimple");

    const auto FBO = rb ? rb->getFB() : fb;

    setViewport(0, 0, pMonitor->m_pixelSize.x, pMonitor->m_pixelSize.y);

    m_renderData.projection = Mat3x3::outputProjection(pMonitor->m_pixelSize, HYPRUTILS_TRANSFORM_NORMAL);

    m_renderData.monitorProjection = Mat3x3::identity();
    if (pMonitor->m_transform != WL_OUTPUT_TRANSFORM_NORMAL) {
        const Vector2D tfmd = pMonitor->m_transform % 2 == 1 ? Vector2D{FBO->m_size.y, FBO->m_size.x} : FBO->m_size;
        m_renderData.monitorProjection.translate(FBO->m_size / 2.0).transform(Math::wlTransformToHyprutils(pMonitor->m_transform)).translate(-tfmd / 2.0);
    }

    m_renderData.pCurrentMonData = &m_monitorRenderResources[pMonitor];

    if (!m_shadersInitialized)
        initShaders();

    m_renderData.damage.set(damage);
    m_renderData.finalDamage.set(damage);

    m_fakeFrame = true;

    m_renderData.currentFB = FBO;
    FBO->bind();
    m_offloadedFramebuffer = false;

    m_renderData.mainFB = m_renderData.currentFB;
    m_renderData.outFB  = FBO;

    m_renderData.simplePass = true;

    pushMonitorTransformEnabled(false);
}

void CHyprOpenGLImpl::begin(PHLMONITOR pMonitor, const CRegion& damage_, CFramebuffer* fb, std::optional<CRegion> finalDamage) {
    m_renderData.pMonitor = pMonitor;

    const GLenum RESETSTATUS = glGetGraphicsResetStatus();
    if (RESETSTATUS != GL_NO_ERROR) {
        std::string errStr = "";
        switch (RESETSTATUS) {
            case GL_GUILTY_CONTEXT_RESET: errStr = "GL_GUILTY_CONTEXT_RESET"; break;
            case GL_INNOCENT_CONTEXT_RESET: errStr = "GL_INNOCENT_CONTEXT_RESET"; break;
            case GL_UNKNOWN_CONTEXT_RESET: errStr = "GL_UNKNOWN_CONTEXT_RESET"; break;
            default: errStr = "UNKNOWN??"; break;
        }
        RASSERT(false, "Aborting, glGetGraphicsResetStatus returned {}. Cannot continue until proper GPU reset handling is implemented.", errStr);
        return;
    }

    TRACY_GPU_ZONE("RenderBegin");

    setViewport(0, 0, pMonitor->m_pixelSize.x, pMonitor->m_pixelSize.y);

    m_renderData.projection = Mat3x3::outputProjection(pMonitor->m_pixelSize, HYPRUTILS_TRANSFORM_NORMAL);

    m_renderData.monitorProjection = pMonitor->m_projMatrix;

    if (m_monitorRenderResources.contains(pMonitor) && m_monitorRenderResources.at(pMonitor).offloadFB.m_size != pMonitor->m_pixelSize)
        destroyMonitorResources(pMonitor);

    m_renderData.pCurrentMonData = &m_monitorRenderResources[pMonitor];

    if (!m_shadersInitialized)
        initShaders();

    const auto DRM_FORMAT = fb ? fb->m_drmFormat : pMonitor->m_output->state->state().drmFormat;

    // ensure a framebuffer for the monitor exists
    if (m_renderData.pCurrentMonData->offloadFB.m_size != pMonitor->m_pixelSize || DRM_FORMAT != m_renderData.pCurrentMonData->offloadFB.m_drmFormat) {
        m_renderData.pCurrentMonData->stencilTex->allocate();

        m_renderData.pCurrentMonData->offloadFB.alloc(pMonitor->m_pixelSize.x, pMonitor->m_pixelSize.y, DRM_FORMAT);
        m_renderData.pCurrentMonData->mirrorFB.alloc(pMonitor->m_pixelSize.x, pMonitor->m_pixelSize.y, DRM_FORMAT);
        m_renderData.pCurrentMonData->mirrorSwapFB.alloc(pMonitor->m_pixelSize.x, pMonitor->m_pixelSize.y, DRM_FORMAT);

        m_renderData.pCurrentMonData->offloadFB.addStencil(m_renderData.pCurrentMonData->stencilTex);
        m_renderData.pCurrentMonData->mirrorFB.addStencil(m_renderData.pCurrentMonData->stencilTex);
        m_renderData.pCurrentMonData->mirrorSwapFB.addStencil(m_renderData.pCurrentMonData->stencilTex);
    }

    if (m_renderData.pCurrentMonData->monitorMirrorFB.isAllocated() && m_renderData.pMonitor->m_mirrors.empty())
        m_renderData.pCurrentMonData->monitorMirrorFB.release();

    m_renderData.damage.set(damage_);
    m_renderData.finalDamage.set(finalDamage.value_or(damage_));

    m_fakeFrame = fb;

    if (m_reloadScreenShader) {
        m_reloadScreenShader = false;
        static auto PSHADER  = CConfigValue<std::string>("decoration:screen_shader");
        applyScreenShader(*PSHADER);
    }

    m_renderData.pCurrentMonData->offloadFB.bind();
    m_renderData.currentFB = &m_renderData.pCurrentMonData->offloadFB;
    m_offloadedFramebuffer = true;

    m_renderData.mainFB = m_renderData.currentFB;
    m_renderData.outFB  = fb ? fb : g_pHyprRenderer->getCurrentRBO()->getFB();

    pushMonitorTransformEnabled(false);
}

void CHyprOpenGLImpl::end() {
    static auto PZOOMDISABLEAA = CConfigValue<Hyprlang::INT>("cursor:zoom_disable_aa");

    TRACY_GPU_ZONE("RenderEnd");

    // end the render, copy the data to the main framebuffer
    if LIKELY (m_offloadedFramebuffer) {
        m_renderData.damage = m_renderData.finalDamage;
        pushMonitorTransformEnabled(true);

        CBox monbox = {0, 0, m_renderData.pMonitor->m_transformedSize.x, m_renderData.pMonitor->m_transformedSize.y};

        if LIKELY (g_pHyprRenderer->m_renderMode == RENDER_MODE_NORMAL && m_renderData.mouseZoomFactor == 1.0f)
            m_renderData.pMonitor->m_zoomController.m_resetCameraState = true;
        m_renderData.pMonitor->m_zoomController.applyZoomTransform(monbox, m_renderData);

        m_applyFinalShader = !m_renderData.blockScreenShader;
        if UNLIKELY (m_renderData.mouseZoomFactor != 1.F && m_renderData.mouseZoomUseMouse && *PZOOMDISABLEAA)
            m_renderData.useNearestNeighbor = true;

        // copy the damaged areas into the mirror buffer
        // we can't use the offloadFB for mirroring, as it contains artifacts from blurring
        if UNLIKELY (!m_renderData.pMonitor->m_mirrors.empty() && !m_fakeFrame)
            saveBufferForMirror(monbox);

        m_renderData.outFB->bind();
        blend(false);

        if LIKELY (m_finalScreenShader->program() < 1 && !g_pHyprRenderer->m_crashingInProgress)
            renderTexturePrimitive(m_renderData.pCurrentMonData->offloadFB.getTexture(), monbox);
        else
            renderTexture(m_renderData.pCurrentMonData->offloadFB.getTexture(), monbox, {});

        blend(true);

        m_renderData.useNearestNeighbor = false;
        m_applyFinalShader              = false;
        popMonitorTransformEnabled();
    }

    // invalidate our render FBs to signal to the driver we don't need them anymore
    m_renderData.pCurrentMonData->mirrorFB.bind();
    m_renderData.pCurrentMonData->mirrorFB.invalidate({GL_STENCIL_ATTACHMENT, GL_COLOR_ATTACHMENT0});
    m_renderData.pCurrentMonData->mirrorSwapFB.bind();
    m_renderData.pCurrentMonData->mirrorSwapFB.invalidate({GL_STENCIL_ATTACHMENT, GL_COLOR_ATTACHMENT0});
    m_renderData.pCurrentMonData->offloadFB.bind();
    m_renderData.pCurrentMonData->offloadFB.invalidate({GL_STENCIL_ATTACHMENT, GL_COLOR_ATTACHMENT0});
    m_renderData.pCurrentMonData->offMainFB.bind();
    m_renderData.pCurrentMonData->offMainFB.invalidate({GL_STENCIL_ATTACHMENT, GL_COLOR_ATTACHMENT0});

    // reset our data
    m_renderData.pMonitor.reset();
    m_renderData.mouseZoomFactor   = 1.f;
    m_renderData.mouseZoomUseMouse = true;
    m_renderData.blockScreenShader = false;
    m_renderData.currentFB         = nullptr;
    m_renderData.mainFB            = nullptr;
    m_renderData.outFB             = nullptr;
    popMonitorTransformEnabled();

    // if we dropped to offMain, release it now.
    // if there is a plugin constantly using it, this might be a bit slow,
    // but I haven't seen a single plugin yet use these, so it's better to drop a bit of vram.
    if UNLIKELY (m_renderData.pCurrentMonData->offMainFB.isAllocated())
        m_renderData.pCurrentMonData->offMainFB.release();

    // check for gl errors
    const GLenum ERR = glGetError();

    if UNLIKELY (ERR == GL_CONTEXT_LOST) /* We don't have infra to recover from this */
        RASSERT(false, "glGetError at Opengl::end() returned GL_CONTEXT_LOST. Cannot continue until proper GPU reset handling is implemented.");
}

void CHyprOpenGLImpl::setDamage(const CRegion& damage_, std::optional<CRegion> finalDamage) {
    m_renderData.damage.set(damage_);
    m_renderData.finalDamage.set(finalDamage.value_or(damage_));
}

// TODO notify user if bundled shader is newer than ~/.config override
static std::string loadShader(const std::string& filename) {
    const auto home = Hyprutils::Path::getHome();
    if (home.has_value()) {
        const auto src = NFsUtils::readFileAsString(home.value() + "/hypr/shaders/" + filename);
        if (src.has_value())
            return src.value();
    }
    for (auto& e : ASSET_PATHS) {
        const auto src = NFsUtils::readFileAsString(std::string{e} + "/hypr/shaders/" + filename);
        if (src.has_value())
            return src.value();
    }
    if (SHADERS.contains(filename))
        return SHADERS.at(filename);
    throw std::runtime_error(std::format("Couldn't load shader {}", filename));
}

static void loadShaderInclude(const std::string& filename, std::map<std::string, std::string>& includes) {
    includes.insert({filename, loadShader(filename)});
}

static void processShaderIncludes(std::string& source, const std::map<std::string, std::string>& includes) {
    for (auto it = includes.begin(); it != includes.end(); ++it) {
        Hyprutils::String::replaceInString(source, "#include \"" + it->first + "\"", it->second);
    }
}

static const uint8_t MAX_INCLUDE_DEPTH = 3;

static std::string   processShader(const std::string& filename, const std::map<std::string, std::string>& includes, const uint8_t includeDepth = 1) {
    auto source = loadShader(filename);
    for (auto i = 0; i < includeDepth; i++) {
        processShaderIncludes(source, includes);
    }
    return source;
}

bool CHyprOpenGLImpl::initShaders() {
    auto                               shaders = makeShared<SPreparedShaders>();
    std::map<std::string, std::string> includes;
    const bool                         isDynamic = m_shadersInitialized;
    static const auto                  PCM       = CConfigValue<Hyprlang::INT>("render:cm_enabled");

    try {
        loadShaderInclude("get_rgb_pixel.glsl", includes);
        loadShaderInclude("get_rgba_pixel.glsl", includes);
        loadShaderInclude("get_rgbx_pixel.glsl", includes);
        loadShaderInclude("discard.glsl", includes);
        loadShaderInclude("do_discard.glsl", includes);
        loadShaderInclude("tint.glsl", includes);
        loadShaderInclude("do_tint.glsl", includes);
        loadShaderInclude("rounding.glsl", includes);
        loadShaderInclude("do_rounding.glsl", includes);
        loadShaderInclude("surface_CM.glsl", includes);
        loadShaderInclude("CM.glsl", includes);
        loadShaderInclude("do_CM.glsl", includes);
        loadShaderInclude("tonemap.glsl", includes);
        loadShaderInclude("do_tonemap.glsl", includes);
        loadShaderInclude("sdr_mod.glsl", includes);
        loadShaderInclude("do_sdr_mod.glsl", includes);
        loadShaderInclude("primaries_xyz.glsl", includes);
        loadShaderInclude("primaries_xyz_uniform.glsl", includes);
        loadShaderInclude("primaries_xyz_const.glsl", includes);
        loadShaderInclude("gain.glsl", includes);
        loadShaderInclude("border.glsl", includes);

        shaders->TEXVERTSRC    = processShader("tex300.vert", includes);
        shaders->TEXVERTSRC320 = processShader("tex320.vert", includes);

        if (!*PCM)
            m_cmSupported = false;
        else {
            std::vector<SFragShaderDesc> CM_SHADERS = {{
                {SH_FRAG_CM_BLURPREPARE, "CMblurprepare.frag"},
                {SH_FRAG_CM_BORDER1, "CMborder.frag"},
            }};

            bool                         success = false;
            for (const auto& desc : CM_SHADERS) {
                const auto fragSrc = processShader(desc.file, includes, MAX_INCLUDE_DEPTH);

                if (!(success = shaders->frag[desc.id]->createProgram(shaders->TEXVERTSRC, fragSrc, true, true)))
                    break;
            }

            if (m_shadersInitialized && m_cmSupported && !success)
                g_pHyprNotificationOverlay->addNotification(I18n::i18nEngine()->localize(I18n::TXT_KEY_NOTIF_CM_RELOAD_FAILED), CHyprColor{}, 15000, ICON_WARNING);

            m_cmSupported = success;

            if (!m_cmSupported)
                Log::logger->log(
                    Log::ERR,
                    "WARNING: CM Shader failed compiling, color management will not work. It's likely because your GPU is an old piece of garbage, don't file bug reports "
                    "about this!");
        }

        std::vector<SFragShaderDesc> FRAG_SHADERS = {{
            {SH_FRAG_QUAD, "quad.frag"},
            {SH_FRAG_PASSTHRURGBA, "passthru.frag"},
            {SH_FRAG_MATTE, "rgbamatte.frag"},
            {SH_FRAG_GLITCH, "glitch.frag"},
            {SH_FRAG_EXT, "ext.frag"},
            {SH_FRAG_BLUR1, "blur1.frag"},
            {SH_FRAG_BLUR2, "blur2.frag"},
            {SH_FRAG_BLURPREPARE, "blurprepare.frag"},
            {SH_FRAG_BLURFINISH, "blurfinish.frag"},
            {SH_FRAG_SHADOW, "shadow.frag"},
            {SH_FRAG_BORDER1, "border.frag"},
        }};

        for (const auto& desc : FRAG_SHADERS) {
            const auto fragSrc = processShader(desc.file, includes, MAX_INCLUDE_DEPTH);

            if (!shaders->frag[desc.id]->createProgram(shaders->TEXVERTSRC, fragSrc, isDynamic))
                return false;
        }

    } catch (const std::exception& e) {
        if (!m_shadersInitialized)
            throw e;

        Log::logger->log(Log::ERR, "Shaders update failed: {}", e.what());
        return false;
    }

    m_shaders            = shaders;
    m_includes           = includes;
    m_shadersInitialized = true;

    Log::logger->log(Log::DEBUG, "Shaders initialized successfully.");
    return true;
}

void CHyprOpenGLImpl::applyScreenShader(const std::string& path) {

    static auto PDT = CConfigValue<Hyprlang::INT>("debug:damage_tracking");

    m_finalScreenShader->destroy();

    if (path.empty() || path == STRVAL_EMPTY)
        return;

    std::ifstream infile(absolutePath(path, g_pConfigManager->getMainConfigPath()));

    if (!infile.good()) {
        g_pConfigManager->addParseError("Screen shader parser: Screen shader path not found");
        return;
    }

    std::string fragmentShader((std::istreambuf_iterator<char>(infile)), (std::istreambuf_iterator<char>()));

    if (!m_finalScreenShader->createProgram(              //
            fragmentShader.starts_with("#version 320 es") // do not break existing custom shaders
                ?
                m_shaders->TEXVERTSRC320 :
                m_shaders->TEXVERTSRC,
            fragmentShader, true)) {
        // Error will have been sent by now by the underlying cause
        return;
    }

    if (m_finalScreenShader->getUniformLocation(SHADER_TIME) != -1)
        m_finalScreenShader->setInitialTime(m_globalTimer.getSeconds());

    static auto uniformRequireNoDamage = [this](eShaderUniform uniform, const std::string& name) {
        if (*PDT == 0)
            return;
        if (m_finalScreenShader->getUniformLocation(uniform) == -1)
            return;

        // The screen shader uses the uniform
        // Since the screen shader could change every frame, damage tracking *needs* to be disabled
        g_pConfigManager->addParseError(std::format("Screen shader: Screen shader uses uniform '{}', which requires debug:damage_tracking to be switched off.\n"
                                                    "WARNING:(Disabling damage tracking will *massively* increase GPU utilization!",
                                                    name));
    };

    // Allow glitch shader to use time uniform whighout damage tracking
    if (!g_pHyprRenderer->m_crashingInProgress)
        uniformRequireNoDamage(SHADER_TIME, "time");

    uniformRequireNoDamage(SHADER_POINTER, "pointer_position");
    uniformRequireNoDamage(SHADER_POINTER_PRESSED_POSITIONS, "pointer_pressed_positions");
    uniformRequireNoDamage(SHADER_POINTER_PRESSED_TIMES, "pointer_pressed_times");
    uniformRequireNoDamage(SHADER_POINTER_PRESSED_KILLED, "pointer_pressed_killed");
    uniformRequireNoDamage(SHADER_POINTER_PRESSED_TOUCHED, "pointer_pressed_touched");
    uniformRequireNoDamage(SHADER_POINTER_LAST_ACTIVE, "pointer_last_active");
    uniformRequireNoDamage(SHADER_POINTER_HIDDEN, "pointer_hidden");
    uniformRequireNoDamage(SHADER_POINTER_KILLING, "pointer_killing");
    uniformRequireNoDamage(SHADER_POINTER_SHAPE, "pointer_shape");
    uniformRequireNoDamage(SHADER_POINTER_SHAPE_PREVIOUS, "pointer_shape_previous");
}

void CHyprOpenGLImpl::clear(const CHyprColor& color) {
    RASSERT(m_renderData.pMonitor, "Tried to render without begin()!");

    TRACY_GPU_ZONE("RenderClear");

    glClearColor(color.r, color.g, color.b, color.a);

    if (!m_renderData.damage.empty()) {
        m_renderData.damage.forEachRect([this](const auto& RECT) {
            scissor(&RECT);
            glClear(GL_COLOR_BUFFER_BIT);
        });
    }
}

void CHyprOpenGLImpl::blend(bool enabled) {
    if (enabled) {
        setCapStatus(GL_BLEND, true);
        glBlendFunc(GL_ONE, GL_ONE_MINUS_SRC_ALPHA); // everything is premultiplied
    } else
        setCapStatus(GL_BLEND, false);

    m_blend = enabled;
}

void CHyprOpenGLImpl::scissor(const CBox& originalBox, bool transform) {
    RASSERT(m_renderData.pMonitor, "Tried to scissor without begin()!");

    // only call glScissor if the box has changed
    static CBox m_lastScissorBox = {};

    if (transform) {
        CBox       box = originalBox;
        const auto TR  = Math::wlTransformToHyprutils(Math::invertTransform(m_renderData.pMonitor->m_transform));
        box.transform(TR, m_renderData.pMonitor->m_transformedSize.x, m_renderData.pMonitor->m_transformedSize.y);

        if (box != m_lastScissorBox) {
            glScissor(box.x, box.y, box.width, box.height);
            m_lastScissorBox = box;
        }

        setCapStatus(GL_SCISSOR_TEST, true);
        return;
    }

    if (originalBox != m_lastScissorBox) {
        glScissor(originalBox.x, originalBox.y, originalBox.width, originalBox.height);
        m_lastScissorBox = originalBox;
    }

    setCapStatus(GL_SCISSOR_TEST, true);
}

void CHyprOpenGLImpl::scissor(const pixman_box32* pBox, bool transform) {
    RASSERT(m_renderData.pMonitor, "Tried to scissor without begin()!");

    if (!pBox) {
        setCapStatus(GL_SCISSOR_TEST, false);
        return;
    }

    CBox newBox = {pBox->x1, pBox->y1, pBox->x2 - pBox->x1, pBox->y2 - pBox->y1};

    scissor(newBox, transform);
}

void CHyprOpenGLImpl::scissor(const int x, const int y, const int w, const int h, bool transform) {
    CBox box = {x, y, w, h};
    scissor(box, transform);
}

void CHyprOpenGLImpl::renderRect(const CBox& box, const CHyprColor& col, SRectRenderData data) {
    if (!data.damage)
        data.damage = &m_renderData.damage;

    if (data.blur)
        renderRectWithBlurInternal(box, col, data);
    else
        renderRectWithDamageInternal(box, col, data);
}

void CHyprOpenGLImpl::renderRectWithBlurInternal(const CBox& box, const CHyprColor& col, const SRectRenderData& data) {
    if (data.damage->empty())
        return;

    CRegion damage{m_renderData.damage};
    damage.intersect(box);

    CFramebuffer* POUTFB = data.xray ? &m_renderData.pCurrentMonData->blurFB : blurMainFramebufferWithDamage(data.blurA, &damage);

    m_renderData.currentFB->bind();

    CBox MONITORBOX = {0, 0, m_renderData.pMonitor->m_transformedSize.x, m_renderData.pMonitor->m_transformedSize.y};
    pushMonitorTransformEnabled(true);
    const auto SAVEDRENDERMODIF = m_renderData.renderModif;
    m_renderData.renderModif    = {}; // fix shit
    renderTexture(POUTFB->getTexture(), MONITORBOX,
                  STextureRenderData{.damage = &damage, .a = data.blurA, .round = data.round, .roundingPower = 2.F, .allowCustomUV = false, .allowDim = false, .noAA = false});
    popMonitorTransformEnabled();
    m_renderData.renderModif = SAVEDRENDERMODIF;

    renderRectWithDamageInternal(box, col, data);
}

void CHyprOpenGLImpl::renderRectWithDamageInternal(const CBox& box, const CHyprColor& col, const SRectRenderData& data) {
    RASSERT((box.width > 0 && box.height > 0), "Tried to render rect with width/height < 0!");
    RASSERT(m_renderData.pMonitor, "Tried to render rect without begin()!");

    TRACY_GPU_ZONE("RenderRectWithDamage");

    CBox newBox = box;
    m_renderData.renderModif.applyToBox(newBox);

    Mat3x3 matrix = m_renderData.monitorProjection.projectBox(
        newBox, Math::wlTransformToHyprutils(Math::invertTransform(!m_monitorTransformEnabled ? WL_OUTPUT_TRANSFORM_NORMAL : m_renderData.pMonitor->m_transform)), newBox.rot);
    Mat3x3 glMatrix = m_renderData.projection.copy().multiply(matrix);

    auto   shader = useShader(m_shaders->frag[SH_FRAG_QUAD]);
    shader->setUniformMatrix3fv(SHADER_PROJ, 1, GL_TRUE, glMatrix.getMatrix());

    // premultiply the color as well as we don't work with straight alpha
    shader->setUniformFloat4(SHADER_COLOR, col.r * col.a, col.g * col.a, col.b * col.a, col.a);

    CBox transformedBox = box;
    transformedBox.transform(Math::wlTransformToHyprutils(Math::invertTransform(m_renderData.pMonitor->m_transform)), m_renderData.pMonitor->m_transformedSize.x,
                             m_renderData.pMonitor->m_transformedSize.y);

    const auto TOPLEFT  = Vector2D(transformedBox.x, transformedBox.y);
    const auto FULLSIZE = Vector2D(transformedBox.width, transformedBox.height);

    // Rounded corners
    shader->setUniformFloat2(SHADER_TOP_LEFT, sc<float>(TOPLEFT.x), sc<float>(TOPLEFT.y));
    shader->setUniformFloat2(SHADER_FULL_SIZE, sc<float>(FULLSIZE.x), sc<float>(FULLSIZE.y));
    shader->setUniformFloat(SHADER_RADIUS, data.round);
    shader->setUniformFloat(SHADER_ROUNDING_POWER, data.roundingPower);

    glBindVertexArray(shader->getUniformLocation(SHADER_SHADER_VAO));

    if (m_renderData.clipBox.width != 0 && m_renderData.clipBox.height != 0) {
        CRegion damageClip{m_renderData.clipBox.x, m_renderData.clipBox.y, m_renderData.clipBox.width, m_renderData.clipBox.height};
        damageClip.intersect(*data.damage);

        if (!damageClip.empty()) {
            damageClip.forEachRect([this](const auto& RECT) {
                scissor(&RECT);
                glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);
            });
        }
    } else {
        data.damage->forEachRect([this](const auto& RECT) {
            scissor(&RECT);
            glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);
        });
    }

    glBindVertexArray(0);
    scissor(nullptr);
}

void CHyprOpenGLImpl::renderTexture(SP<CTexture> tex, const CBox& box, STextureRenderData data) {
    RASSERT(m_renderData.pMonitor, "Tried to render texture without begin()!");

    if (!data.damage) {
        if (m_renderData.damage.empty())
            return;

        data.damage = &m_renderData.damage;
    }

    if (data.blur)
        renderTextureWithBlurInternal(tex, box, data);
    else
        renderTextureInternal(tex, box, data);

    scissor(nullptr);
}

static std::map<std::pair<uint32_t, uint32_t>, std::array<GLfloat, 9>> primariesConversionCache;

static bool isSDR2HDR(const NColorManagement::SImageDescription& imageDescription, const NColorManagement::SImageDescription& targetImageDescription) {
    // might be too strict
    return (imageDescription.transferFunction == NColorManagement::CM_TRANSFER_FUNCTION_SRGB ||
            imageDescription.transferFunction == NColorManagement::CM_TRANSFER_FUNCTION_GAMMA22) &&
        (targetImageDescription.transferFunction == NColorManagement::CM_TRANSFER_FUNCTION_ST2084_PQ ||
         targetImageDescription.transferFunction == NColorManagement::CM_TRANSFER_FUNCTION_HLG);
}

static bool isHDR2SDR(const NColorManagement::SImageDescription& imageDescription, const NColorManagement::SImageDescription& targetImageDescription) {
    // might be too strict
    return (imageDescription.transferFunction == NColorManagement::CM_TRANSFER_FUNCTION_ST2084_PQ ||
            imageDescription.transferFunction == NColorManagement::CM_TRANSFER_FUNCTION_HLG) &&
        (targetImageDescription.transferFunction == NColorManagement::CM_TRANSFER_FUNCTION_SRGB ||
         targetImageDescription.transferFunction == NColorManagement::CM_TRANSFER_FUNCTION_GAMMA22);
}

void CHyprOpenGLImpl::passCMUniforms(WP<CShader> shader, const NColorManagement::PImageDescription imageDescription,
                                     const NColorManagement::PImageDescription targetImageDescription, bool modifySDR, float sdrMinLuminance, int sdrMaxLuminance) {
    static auto PSDREOTF = CConfigValue<Hyprlang::INT>("render:cm_sdr_eotf");

    if (m_renderData.surface.valid() &&
        ((!m_renderData.surface->m_colorManagement.valid() && *PSDREOTF >= 1) ||
         (*PSDREOTF == 2 && m_renderData.surface->m_colorManagement.valid() &&
          imageDescription->value().transferFunction == NColorManagement::eTransferFunction::CM_TRANSFER_FUNCTION_SRGB))) {
        shader->setUniformInt(SHADER_SOURCE_TF, NColorManagement::eTransferFunction::CM_TRANSFER_FUNCTION_GAMMA22);
    } else
        shader->setUniformInt(SHADER_SOURCE_TF, imageDescription->value().transferFunction);

    shader->setUniformInt(SHADER_TARGET_TF, targetImageDescription->value().transferFunction);

    const auto                   targetPrimaries      = targetImageDescription->getPrimaries();
    const auto                   mat                  = targetPrimaries->value().toXYZ().mat();
    const std::array<GLfloat, 9> glTargetPrimariesXYZ = {
        mat[0][0], mat[1][0], mat[2][0], //
        mat[0][1], mat[1][1], mat[2][1], //
        mat[0][2], mat[1][2], mat[2][2], //
    };
    shader->setUniformMatrix3fv(SHADER_TARGET_PRIMARIES_XYZ, 1, false, glTargetPrimariesXYZ);

    const bool needsSDRmod = modifySDR && isSDR2HDR(imageDescription->value(), targetImageDescription->value());
    const bool needsHDRmod = !needsSDRmod && isHDR2SDR(imageDescription->value(), targetImageDescription->value());

    shader->setUniformFloat2(SHADER_SRC_TF_RANGE, imageDescription->value().getTFMinLuminance(needsSDRmod ? sdrMinLuminance : -1),
                             imageDescription->value().getTFMaxLuminance(needsSDRmod ? sdrMaxLuminance : -1));
    shader->setUniformFloat2(SHADER_DST_TF_RANGE, targetImageDescription->value().getTFMinLuminance(needsSDRmod ? sdrMinLuminance : -1),
                             targetImageDescription->value().getTFMaxLuminance(needsSDRmod ? sdrMaxLuminance : -1));

    shader->setUniformFloat(SHADER_SRC_REF_LUMINANCE, imageDescription->value().luminances.reference);
    shader->setUniformFloat(SHADER_DST_REF_LUMINANCE, targetImageDescription->value().luminances.reference);

    const float maxLuminance = needsHDRmod ?
        imageDescription->value().getTFMaxLuminance(-1) :
        (imageDescription->value().luminances.max > 0 ? imageDescription->value().luminances.max : imageDescription->value().luminances.reference);
    shader->setUniformFloat(SHADER_MAX_LUMINANCE, maxLuminance * targetImageDescription->value().luminances.reference / imageDescription->value().luminances.reference);
    shader->setUniformFloat(SHADER_DST_MAX_LUMINANCE, targetImageDescription->value().luminances.max > 0 ? targetImageDescription->value().luminances.max : 10000);
    shader->setUniformFloat(SHADER_SDR_SATURATION, needsSDRmod && m_renderData.pMonitor->m_sdrSaturation > 0 ? m_renderData.pMonitor->m_sdrSaturation : 1.0f);
    shader->setUniformFloat(SHADER_SDR_BRIGHTNESS, needsSDRmod && m_renderData.pMonitor->m_sdrBrightness > 0 ? m_renderData.pMonitor->m_sdrBrightness : 1.0f);
    const auto cacheKey = std::make_pair(imageDescription->id(), targetImageDescription->id());
    if (!primariesConversionCache.contains(cacheKey)) {
        auto                         conversion      = imageDescription->getPrimaries()->convertMatrix(targetImageDescription->getPrimaries());
        const auto                   mat             = conversion.mat();
        const std::array<GLfloat, 9> glConvertMatrix = {
            mat[0][0], mat[1][0], mat[2][0], //
            mat[0][1], mat[1][1], mat[2][1], //
            mat[0][2], mat[1][2], mat[2][2], //
        };
        primariesConversionCache.insert(std::make_pair(cacheKey, glConvertMatrix));
    }
    shader->setUniformMatrix3fv(SHADER_CONVERT_MATRIX, 1, false, primariesConversionCache[cacheKey]);
}

void CHyprOpenGLImpl::passCMUniforms(WP<CShader> shader, const PImageDescription imageDescription) {
    passCMUniforms(shader, imageDescription, m_renderData.pMonitor->m_imageDescription, true, m_renderData.pMonitor->m_sdrMinLuminance, m_renderData.pMonitor->m_sdrMaxLuminance);
}

void CHyprOpenGLImpl::renderTextureInternal(SP<CTexture> tex, const CBox& box, const STextureRenderData& data) {
    RASSERT(m_renderData.pMonitor, "Tried to render texture without begin()!");
    RASSERT((tex->m_texID > 0), "Attempted to draw nullptr texture!");

    TRACY_GPU_ZONE("RenderTextureInternalWithDamage");

    float alpha = std::clamp(data.a, 0.f, 1.f);

    if (data.damage->empty())
        return;

    CBox newBox = box;
    m_renderData.renderModif.applyToBox(newBox);

    static const auto PDT            = CConfigValue<Hyprlang::INT>("debug:damage_tracking");
    static const auto PPASS          = CConfigValue<Hyprlang::INT>("render:cm_fs_passthrough");
    static const auto PENABLECM      = CConfigValue<Hyprlang::INT>("render:cm_enabled");
    static const auto PCURSORTIMEOUT = CConfigValue<Hyprlang::FLOAT>("cursor:inactive_timeout");

    // get the needed transform for this texture
    const auto                  MONITOR_INVERTED = Math::wlTransformToHyprutils(Math::invertTransform(m_renderData.pMonitor->m_transform));
    Hyprutils::Math::eTransform TRANSFORM        = tex->m_transform;

    if (m_monitorTransformEnabled)
        TRANSFORM = Math::composeTransform(MONITOR_INVERTED, TRANSFORM);

    Mat3x3      matrix   = m_renderData.monitorProjection.projectBox(newBox, TRANSFORM, newBox.rot);
    Mat3x3      glMatrix = m_renderData.projection.copy().multiply(matrix);

    WP<CShader> shader;

    bool        usingFinalShader = false;

    const bool  CRASHING = m_applyFinalShader && g_pHyprRenderer->m_crashingInProgress;

    uint8_t     shaderFeatures = 0;

    if (CRASHING) {
        shader           = m_shaders->frag[SH_FRAG_GLITCH];
        usingFinalShader = true;
    } else if (m_applyFinalShader && m_finalScreenShader->program()) {
        shader           = m_finalScreenShader;
        usingFinalShader = true;
    } else {
        if (m_applyFinalShader) {
            shader           = m_shaders->frag[SH_FRAG_PASSTHRURGBA];
            usingFinalShader = true;
        } else {
            switch (tex->m_type) {
                case TEXTURE_RGBA: shaderFeatures |= SH_FEAT_RGBA; break;
                case TEXTURE_RGBX: shaderFeatures &= ~SH_FEAT_RGBA; break;

                case TEXTURE_EXTERNAL: shader = m_shaders->frag[SH_FRAG_EXT]; break; // might be unused
                default: RASSERT(false, "tex->m_iTarget unsupported!");
            }
        }
    }

    if (m_renderData.currentWindow && m_renderData.currentWindow->m_ruleApplicator->RGBX().valueOrDefault())
        shaderFeatures &= ~SH_FEAT_RGBA;

    glActiveTexture(GL_TEXTURE0);
    tex->bind();

    tex->setTexParameter(GL_TEXTURE_WRAP_S, data.wrapX);
    tex->setTexParameter(GL_TEXTURE_WRAP_T, data.wrapY);

    if (m_renderData.useNearestNeighbor) {
        tex->setTexParameter(GL_TEXTURE_MAG_FILTER, GL_NEAREST);
        tex->setTexParameter(GL_TEXTURE_MIN_FILTER, GL_NEAREST);
    } else {
        tex->setTexParameter(GL_TEXTURE_MAG_FILTER, tex->magFilter);
        tex->setTexParameter(GL_TEXTURE_MIN_FILTER, tex->minFilter);
    }

    const bool isHDRSurface      = m_renderData.surface.valid() && m_renderData.surface->m_colorManagement.valid() ? m_renderData.surface->m_colorManagement->isHDR() : false;
    const bool canPassHDRSurface = isHDRSurface && !m_renderData.surface->m_colorManagement->isWindowsScRGB(); // windows scRGB requires CM shader

    const auto imageDescription = m_renderData.surface.valid() && m_renderData.surface->m_colorManagement.valid() ?
        CImageDescription::from(m_renderData.surface->m_colorManagement->imageDescription()) :
        (data.cmBackToSRGB ? data.cmBackToSRGBSource->m_imageDescription : DEFAULT_IMAGE_DESCRIPTION);

    const bool skipCM = !*PENABLECM || !m_cmSupported                                                        /* CM unsupported or disabled */
        || m_renderData.pMonitor->doesNoShaderCM()                                                           /* no shader needed */
        || (imageDescription->id() == m_renderData.pMonitor->m_imageDescription->id() && !data.cmBackToSRGB) /* Source and target have the same image description */
        || (((*PPASS && canPassHDRSurface) ||
             (*PPASS == 1 && !isHDRSurface && m_renderData.pMonitor->m_cmType != NCMType::CM_HDR && m_renderData.pMonitor->m_cmType != NCMType::CM_HDR_EDID)) &&
            m_renderData.pMonitor->inFullscreenMode()) /* Fullscreen window with pass cm enabled */;

    if (data.discardActive)
        shaderFeatures |= SH_FEAT_DISCARD;

    if (!usingFinalShader) {
        if (data.allowDim && m_renderData.currentWindow && (m_renderData.currentWindow->m_notRespondingTint->value() > 0 || m_renderData.currentWindow->m_dimPercent->value() > 0))
            shaderFeatures |= SH_FEAT_TINT;

        if (data.round > 0)
            shaderFeatures |= SH_FEAT_ROUNDING;

        if (!skipCM) {
            shaderFeatures |= SH_FEAT_CM;

            const bool  needsSDRmod  = isSDR2HDR(imageDescription->value(), m_renderData.pMonitor->m_imageDescription->value());
            const bool  needsHDRmod  = !needsSDRmod && isHDR2SDR(imageDescription->value(), m_renderData.pMonitor->m_imageDescription->value());
            const float maxLuminance = needsHDRmod ?
                imageDescription->value().getTFMaxLuminance(-1) :
                (imageDescription->value().luminances.max > 0 ? imageDescription->value().luminances.max : imageDescription->value().luminances.reference);
            const auto  dstMaxLuminance =
                m_renderData.pMonitor->m_imageDescription->value().luminances.max > 0 ? m_renderData.pMonitor->m_imageDescription->value().luminances.max : 10000;

            if (maxLuminance >= dstMaxLuminance * 1.01)
                shaderFeatures |= SH_FEAT_TONEMAP;

            if (!data.cmBackToSRGB &&
                (imageDescription->value().transferFunction == CM_TRANSFER_FUNCTION_SRGB || imageDescription->value().transferFunction == CM_TRANSFER_FUNCTION_GAMMA22) &&
                m_renderData.pMonitor->m_imageDescription->value().transferFunction == CM_TRANSFER_FUNCTION_ST2084_PQ &&
                ((m_renderData.pMonitor->m_sdrSaturation > 0 && m_renderData.pMonitor->m_sdrSaturation != 1.0f) ||
                 (m_renderData.pMonitor->m_sdrBrightness > 0 && m_renderData.pMonitor->m_sdrBrightness != 1.0f)))
                shaderFeatures |= SH_FEAT_SDR_MOD;
        }
    }

    if (!shader)
        shader = getSurfaceShader(shaderFeatures);

    shader = useShader(shader);

    if (!skipCM && !usingFinalShader) {
        if (data.cmBackToSRGB) {
            static auto PSDREOTF      = CConfigValue<Hyprlang::INT>("render:cm_sdr_eotf");
            auto        chosenSdrEotf = *PSDREOTF != 3 ? NColorManagement::CM_TRANSFER_FUNCTION_GAMMA22 : NColorManagement::CM_TRANSFER_FUNCTION_SRGB;
            passCMUniforms(shader, imageDescription, CImageDescription::from(NColorManagement::SImageDescription{.transferFunction = chosenSdrEotf}), true, -1, -1);
        } else
            passCMUniforms(shader, imageDescription);
    }

    shader->setUniformMatrix3fv(SHADER_PROJ, 1, GL_TRUE, glMatrix.getMatrix());
    shader->setUniformInt(SHADER_TEX, 0);

    if ((usingFinalShader && *PDT == 0) || CRASHING)
        shader->setUniformFloat(SHADER_TIME, m_globalTimer.getSeconds() - shader->getInitialTime());
    else if (usingFinalShader)
        shader->setUniformFloat(SHADER_TIME, 0.f);

    if (usingFinalShader) {
        shader->setUniformInt(SHADER_WL_OUTPUT, m_renderData.pMonitor->m_id);
        shader->setUniformFloat2(SHADER_FULL_SIZE, m_renderData.pMonitor->m_pixelSize.x, m_renderData.pMonitor->m_pixelSize.y);
        shader->setUniformFloat(SHADER_POINTER_INACTIVE_TIMEOUT, *PCURSORTIMEOUT);
        shader->setUniformInt(SHADER_POINTER_HIDDEN, g_pHyprRenderer->m_cursorHiddenByCondition);
        shader->setUniformInt(SHADER_POINTER_KILLING, g_pInputManager->getClickMode() == CLICKMODE_KILL);
        shader->setUniformInt(SHADER_POINTER_SHAPE, g_pHyprRenderer->m_lastCursorData.shape);
        shader->setUniformInt(SHADER_POINTER_SHAPE_PREVIOUS, g_pHyprRenderer->m_lastCursorData.shapePrevious);
        shader->setUniformFloat(SHADER_POINTER_SIZE, g_pCursorManager->getScaledSize());
    }

    if (usingFinalShader && *PDT == 0) {
        PHLMONITORREF pMonitor = m_renderData.pMonitor;
        Vector2D      p        = ((g_pInputManager->getMouseCoordsInternal() - pMonitor->m_position) * pMonitor->m_scale);
        p                      = p.transform(Math::wlTransformToHyprutils(pMonitor->m_transform), pMonitor->m_pixelSize);
        shader->setUniformFloat2(SHADER_POINTER, p.x / pMonitor->m_pixelSize.x, p.y / pMonitor->m_pixelSize.y);

        std::vector<float> pressedPos = m_pressedHistoryPositions | std::views::transform([&](const Vector2D& vec) {
                                            Vector2D pPressed = ((vec - pMonitor->m_position) * pMonitor->m_scale);
                                            pPressed          = pPressed.transform(Math::wlTransformToHyprutils(pMonitor->m_transform), pMonitor->m_pixelSize);
                                            return std::array<float, 2>{pPressed.x / pMonitor->m_pixelSize.x, pPressed.y / pMonitor->m_pixelSize.y};
                                        }) |
            std::views::join | std::ranges::to<std::vector<float>>();

        shader->setUniform2fv(SHADER_POINTER_PRESSED_POSITIONS, pressedPos.size(), pressedPos);

        std::vector<float> pressedTime =
            m_pressedHistoryTimers | std::views::transform([](const CTimer& timer) { return timer.getSeconds(); }) | std::ranges::to<std::vector<float>>();

        shader->setUniform1fv(SHADER_POINTER_PRESSED_TIMES, pressedTime.size(), pressedTime);

        shader->setUniformInt(SHADER_POINTER_PRESSED_KILLED, m_pressedHistoryKilled);
        shader->setUniformInt(SHADER_POINTER_PRESSED_TOUCHED, m_pressedHistoryTouched);

        shader->setUniformFloat(SHADER_POINTER_LAST_ACTIVE, g_pInputManager->m_lastCursorMovement.getSeconds());
        shader->setUniformFloat(SHADER_POINTER_SWITCH_TIME, g_pHyprRenderer->m_lastCursorData.switchedTimer.getSeconds());

    } else if (usingFinalShader) {
        shader->setUniformFloat2(SHADER_POINTER, 0.f, 0.f);

        static const std::vector<float> pressedPosDefault(POINTER_PRESSED_HISTORY_LENGTH * 2uz, 0.f);
        static const std::vector<float> pressedTimeDefault(POINTER_PRESSED_HISTORY_LENGTH, 0.f);

        shader->setUniform2fv(SHADER_POINTER_PRESSED_POSITIONS, pressedPosDefault.size(), pressedPosDefault);
        shader->setUniform1fv(SHADER_POINTER_PRESSED_TIMES, pressedTimeDefault.size(), pressedTimeDefault);
        shader->setUniformInt(SHADER_POINTER_PRESSED_KILLED, 0);

        shader->setUniformFloat(SHADER_POINTER_LAST_ACTIVE, 0.f);
        shader->setUniformFloat(SHADER_POINTER_SWITCH_TIME, 0.f);
    }

    if (CRASHING) {
        shader->setUniformFloat(SHADER_DISTORT, g_pHyprRenderer->m_crashingDistort);
        shader->setUniformFloat2(SHADER_FULL_SIZE, m_renderData.pMonitor->m_pixelSize.x, m_renderData.pMonitor->m_pixelSize.y);
    }

    if (!usingFinalShader) {
        shader->setUniformFloat(SHADER_ALPHA, alpha);

        if (data.discardActive) {
            shader->setUniformInt(SHADER_DISCARD_OPAQUE, !!(m_renderData.discardMode & DISCARD_OPAQUE));
            shader->setUniformInt(SHADER_DISCARD_ALPHA, !!(m_renderData.discardMode & DISCARD_ALPHA));
            shader->setUniformFloat(SHADER_DISCARD_ALPHA_VALUE, m_renderData.discardOpacity);
        } else {
            shader->setUniformInt(SHADER_DISCARD_OPAQUE, 0);
            shader->setUniformInt(SHADER_DISCARD_ALPHA, 0);
        }
    }

    CBox transformedBox = newBox;
    transformedBox.transform(Math::wlTransformToHyprutils(Math::invertTransform(m_renderData.pMonitor->m_transform)), m_renderData.pMonitor->m_transformedSize.x,
                             m_renderData.pMonitor->m_transformedSize.y);

    const auto TOPLEFT  = Vector2D(transformedBox.x, transformedBox.y);
    const auto FULLSIZE = Vector2D(transformedBox.width, transformedBox.height);

    if (!usingFinalShader) {
        // Rounded corners
        shader->setUniformFloat2(SHADER_TOP_LEFT, TOPLEFT.x, TOPLEFT.y);
        shader->setUniformFloat2(SHADER_FULL_SIZE, FULLSIZE.x, FULLSIZE.y);
        shader->setUniformFloat(SHADER_RADIUS, data.round);
        shader->setUniformFloat(SHADER_ROUNDING_POWER, data.roundingPower);

        if (data.allowDim && m_renderData.currentWindow) {
            if (m_renderData.currentWindow->m_notRespondingTint->value() > 0) {
                const auto DIM = m_renderData.currentWindow->m_notRespondingTint->value();
                shader->setUniformInt(SHADER_APPLY_TINT, 1);
                shader->setUniformFloat3(SHADER_TINT, 1.f - DIM, 1.f - DIM, 1.f - DIM);
            } else if (m_renderData.currentWindow->m_dimPercent->value() > 0) {
                shader->setUniformInt(SHADER_APPLY_TINT, 1);
                const auto DIM = m_renderData.currentWindow->m_dimPercent->value();
                shader->setUniformFloat3(SHADER_TINT, 1.f - DIM, 1.f - DIM, 1.f - DIM);
            } else
                shader->setUniformInt(SHADER_APPLY_TINT, 0);
        } else
            shader->setUniformInt(SHADER_APPLY_TINT, 0);
    }

    glBindVertexArray(shader->getUniformLocation(SHADER_SHADER_VAO));
    glBindBuffer(GL_ARRAY_BUFFER, shader->getUniformLocation(SHADER_SHADER_VBO));

    // this tells GPU can keep reading the old block for previous draws while the CPU writes to a new one.
    // to avoid stalls if renderTextureInternal is called multiple times on same renderpass
    // at the cost of some temporar vram usage.
    glBufferData(GL_ARRAY_BUFFER, sizeof(fullVerts), nullptr, GL_DYNAMIC_DRAW);

    auto verts = fullVerts;

    if (data.allowCustomUV && m_renderData.primarySurfaceUVTopLeft != Vector2D(-1, -1)) {
        const float u0 = m_renderData.primarySurfaceUVTopLeft.x;
        const float v0 = m_renderData.primarySurfaceUVTopLeft.y;
        const float u1 = m_renderData.primarySurfaceUVBottomRight.x;
        const float v1 = m_renderData.primarySurfaceUVBottomRight.y;

        verts[0].u = u0;
        verts[0].v = v0;
        verts[1].u = u0;
        verts[1].v = v1;
        verts[2].u = u1;
        verts[2].v = v0;
        verts[3].u = u1;
        verts[3].v = v1;
    }

    glBufferSubData(GL_ARRAY_BUFFER, 0, sizeof(verts), verts.data());

    if (!m_renderData.clipBox.empty() || !m_renderData.clipRegion.empty()) {
        CRegion damageClip = m_renderData.clipBox;

        if (!m_renderData.clipRegion.empty()) {
            if (m_renderData.clipBox.empty())
                damageClip = m_renderData.clipRegion;
            else
                damageClip.intersect(m_renderData.clipRegion);
        }

        if (!damageClip.empty()) {
            damageClip.forEachRect([this](const auto& RECT) {
                scissor(&RECT);
                glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);
            });
        }
    } else {
        data.damage->forEachRect([this](const auto& RECT) {
            scissor(&RECT);
            glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);
        });
    }

    glBindVertexArray(0);
    glBindBuffer(GL_ARRAY_BUFFER, 0);
    tex->unbind();
}

void CHyprOpenGLImpl::renderTexturePrimitive(SP<CTexture> tex, const CBox& box) {
    RASSERT(m_renderData.pMonitor, "Tried to render texture without begin()!");
    RASSERT((tex->m_texID > 0), "Attempted to draw nullptr texture!");

    TRACY_GPU_ZONE("RenderTexturePrimitive");

    if (m_renderData.damage.empty())
        return;

    CBox newBox = box;
    m_renderData.renderModif.applyToBox(newBox);

    // get transform
    const auto TRANSFORM = Math::wlTransformToHyprutils(Math::invertTransform(!m_monitorTransformEnabled ? WL_OUTPUT_TRANSFORM_NORMAL : m_renderData.pMonitor->m_transform));
    Mat3x3     matrix    = m_renderData.monitorProjection.projectBox(newBox, TRANSFORM, newBox.rot);
    Mat3x3     glMatrix  = m_renderData.projection.copy().multiply(matrix);

    glActiveTexture(GL_TEXTURE0);
    tex->bind();

    // ensure the final blit uses the desired sampling filter
    // when cursor zoom is active we want nearest-neighbor (no anti-aliasing)
    if (m_renderData.useNearestNeighbor) {
        tex->setTexParameter(GL_TEXTURE_MAG_FILTER, GL_NEAREST);
        tex->setTexParameter(GL_TEXTURE_MIN_FILTER, GL_NEAREST);
    } else {
        tex->setTexParameter(GL_TEXTURE_MAG_FILTER, tex->magFilter);
        tex->setTexParameter(GL_TEXTURE_MIN_FILTER, tex->minFilter);
    }

    auto shader = useShader(m_shaders->frag[SH_FRAG_PASSTHRURGBA]);
    shader->setUniformMatrix3fv(SHADER_PROJ, 1, GL_TRUE, glMatrix.getMatrix());
    shader->setUniformInt(SHADER_TEX, 0);
    glBindVertexArray(shader->getUniformLocation(SHADER_SHADER_VAO));

    m_renderData.damage.forEachRect([this](const auto& RECT) {
        scissor(&RECT);
        glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);
    });

    scissor(nullptr);
    glBindVertexArray(0);
    tex->unbind();
}

void CHyprOpenGLImpl::renderTextureMatte(SP<CTexture> tex, const CBox& box, CFramebuffer& matte) {
    RASSERT(m_renderData.pMonitor, "Tried to render texture without begin()!");
    RASSERT((tex->m_texID > 0), "Attempted to draw nullptr texture!");

    TRACY_GPU_ZONE("RenderTextureMatte");

    if (m_renderData.damage.empty())
        return;

    CBox newBox = box;
    m_renderData.renderModif.applyToBox(newBox);

    // get transform
    const auto TRANSFORM = Math::wlTransformToHyprutils(Math::invertTransform(!m_monitorTransformEnabled ? WL_OUTPUT_TRANSFORM_NORMAL : m_renderData.pMonitor->m_transform));
    Mat3x3     matrix    = m_renderData.monitorProjection.projectBox(newBox, TRANSFORM, newBox.rot);
    Mat3x3     glMatrix  = m_renderData.projection.copy().multiply(matrix);

    auto       shader = useShader(m_shaders->frag[SH_FRAG_MATTE]);
    shader->setUniformMatrix3fv(SHADER_PROJ, 1, GL_TRUE, glMatrix.getMatrix());
    shader->setUniformInt(SHADER_TEX, 0);
    shader->setUniformInt(SHADER_ALPHA_MATTE, 1);

    glActiveTexture(GL_TEXTURE0);
    tex->bind();

    glActiveTexture(GL_TEXTURE0 + 1);
    auto matteTex = matte.getTexture();
    matteTex->bind();

    glBindVertexArray(shader->getUniformLocation(SHADER_SHADER_VAO));

    m_renderData.damage.forEachRect([this](const auto& RECT) {
        scissor(&RECT);
        glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);
    });

    scissor(nullptr);
    glBindVertexArray(0);
    tex->unbind();
}

// This probably isn't the fastest
// but it works... well, I guess?
//
// Dual (or more) kawase blur
CFramebuffer* CHyprOpenGLImpl::blurMainFramebufferWithDamage(float a, CRegion* originalDamage) {
    if (!m_renderData.currentFB->getTexture()) {
        Log::logger->log(Log::ERR, "BUG THIS: null fb texture while attempting to blur main fb?! (introspection off?!)");
        return &m_renderData.pCurrentMonData->mirrorFB; // return something to sample from at least
    }

    return blurFramebufferWithDamage(a, originalDamage, *m_renderData.currentFB);
}

CFramebuffer* CHyprOpenGLImpl::blurFramebufferWithDamage(float a, CRegion* originalDamage, CFramebuffer& source) {
    TRACY_GPU_ZONE("RenderBlurFramebufferWithDamage");

    const auto BLENDBEFORE = m_blend;
    blend(false);
    setCapStatus(GL_STENCIL_TEST, false);

    // get transforms for the full monitor
    const auto TRANSFORM  = Math::wlTransformToHyprutils(Math::invertTransform(m_renderData.pMonitor->m_transform));
    CBox       MONITORBOX = {0, 0, m_renderData.pMonitor->m_transformedSize.x, m_renderData.pMonitor->m_transformedSize.y};
    Mat3x3     matrix     = m_renderData.monitorProjection.projectBox(MONITORBOX, TRANSFORM);
    Mat3x3     glMatrix   = m_renderData.projection.copy().multiply(matrix);

    // get the config settings
    static auto PBLURSIZE             = CConfigValue<Hyprlang::INT>("decoration:blur:size");
    static auto PBLURPASSES           = CConfigValue<Hyprlang::INT>("decoration:blur:passes");
    static auto PBLURVIBRANCY         = CConfigValue<Hyprlang::FLOAT>("decoration:blur:vibrancy");
    static auto PBLURVIBRANCYDARKNESS = CConfigValue<Hyprlang::FLOAT>("decoration:blur:vibrancy_darkness");

    const auto  BLUR_PASSES = std::clamp(*PBLURPASSES, sc<int64_t>(1), sc<int64_t>(8));

    // prep damage
    CRegion damage{*originalDamage};
    damage.transform(Math::wlTransformToHyprutils(Math::invertTransform(m_renderData.pMonitor->m_transform)), m_renderData.pMonitor->m_transformedSize.x,
                     m_renderData.pMonitor->m_transformedSize.y);
    damage.expand(std::clamp(*PBLURSIZE, sc<int64_t>(1), sc<int64_t>(40)) * pow(2, BLUR_PASSES));

    // helper
    const auto    PMIRRORFB     = &m_renderData.pCurrentMonData->mirrorFB;
    const auto    PMIRRORSWAPFB = &m_renderData.pCurrentMonData->mirrorSwapFB;

    CFramebuffer* currentRenderToFB = PMIRRORFB;

    // Begin with base color adjustments - global brightness and contrast
    // TODO: make this a part of the first pass maybe to save on a drawcall?
    {
        static auto PBLURCONTRAST   = CConfigValue<Hyprlang::FLOAT>("decoration:blur:contrast");
        static auto PBLURBRIGHTNESS = CConfigValue<Hyprlang::FLOAT>("decoration:blur:brightness");

        PMIRRORSWAPFB->bind();

        glActiveTexture(GL_TEXTURE0);

        auto currentTex = source.getTexture();

        currentTex->bind();
        currentTex->setTexParameter(GL_TEXTURE_MIN_FILTER, GL_LINEAR);

        WP<CShader> shader;

        // From FB to sRGB
        const bool skipCM = !m_cmSupported || m_renderData.pMonitor->m_imageDescription->id() == DEFAULT_IMAGE_DESCRIPTION->id();
        if (!skipCM) {
            shader = useShader(m_shaders->frag[SH_FRAG_CM_BLURPREPARE]);
            passCMUniforms(shader, m_renderData.pMonitor->m_imageDescription, DEFAULT_IMAGE_DESCRIPTION);
            shader->setUniformFloat(SHADER_SDR_SATURATION,
                                    m_renderData.pMonitor->m_sdrSaturation > 0 &&
                                            m_renderData.pMonitor->m_imageDescription->value().transferFunction == NColorManagement::CM_TRANSFER_FUNCTION_ST2084_PQ ?
                                        m_renderData.pMonitor->m_sdrSaturation :
                                        1.0f);
            shader->setUniformFloat(SHADER_SDR_BRIGHTNESS,
                                    m_renderData.pMonitor->m_sdrBrightness > 0 &&
                                            m_renderData.pMonitor->m_imageDescription->value().transferFunction == NColorManagement::CM_TRANSFER_FUNCTION_ST2084_PQ ?
                                        m_renderData.pMonitor->m_sdrBrightness :
                                        1.0f);
        } else
            shader = useShader(m_shaders->frag[SH_FRAG_BLURPREPARE]);

        shader->setUniformMatrix3fv(SHADER_PROJ, 1, GL_TRUE, glMatrix.getMatrix());
        shader->setUniformFloat(SHADER_CONTRAST, *PBLURCONTRAST);
        shader->setUniformFloat(SHADER_BRIGHTNESS, *PBLURBRIGHTNESS);
        shader->setUniformInt(SHADER_TEX, 0);

        glBindVertexArray(shader->getUniformLocation(SHADER_SHADER_VAO));

        if (!damage.empty()) {
            damage.forEachRect([this](const auto& RECT) {
                scissor(&RECT, false /* this region is already transformed */);
                glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);
            });
        }

        glBindVertexArray(0);
        currentRenderToFB = PMIRRORSWAPFB;
    }

    // declare the draw func
    auto drawPass = [&](WP<CShader> shader, ePreparedFragmentShader frag, CRegion* pDamage) {
        if (currentRenderToFB == PMIRRORFB)
            PMIRRORSWAPFB->bind();
        else
            PMIRRORFB->bind();

        glActiveTexture(GL_TEXTURE0);

        auto currentTex = currentRenderToFB->getTexture();

        currentTex->bind();

        currentTex->setTexParameter(GL_TEXTURE_MIN_FILTER, GL_LINEAR);

        // prep two shaders
        shader->setUniformMatrix3fv(SHADER_PROJ, 1, GL_TRUE, glMatrix.getMatrix());
        shader->setUniformFloat(SHADER_RADIUS, *PBLURSIZE * a); // this makes the blursize change with a
        if (frag == SH_FRAG_BLUR1) {
            shader->setUniformFloat2(SHADER_HALFPIXEL, 0.5f / (m_renderData.pMonitor->m_pixelSize.x / 2.f), 0.5f / (m_renderData.pMonitor->m_pixelSize.y / 2.f));
            shader->setUniformInt(SHADER_PASSES, BLUR_PASSES);
            shader->setUniformFloat(SHADER_VIBRANCY, *PBLURVIBRANCY);
            shader->setUniformFloat(SHADER_VIBRANCY_DARKNESS, *PBLURVIBRANCYDARKNESS);
        } else
            shader->setUniformFloat2(SHADER_HALFPIXEL, 0.5f / (m_renderData.pMonitor->m_pixelSize.x * 2.f), 0.5f / (m_renderData.pMonitor->m_pixelSize.y * 2.f));
        shader->setUniformInt(SHADER_TEX, 0);

        glBindVertexArray(shader->getUniformLocation(SHADER_SHADER_VAO));

        if (!pDamage->empty()) {
            pDamage->forEachRect([this](const auto& RECT) {
                scissor(&RECT, false /* this region is already transformed */);
                glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);
            });
        }

        glBindVertexArray(0);

        if (currentRenderToFB != PMIRRORFB)
            currentRenderToFB = PMIRRORFB;
        else
            currentRenderToFB = PMIRRORSWAPFB;
    };

    // draw the things.
    // first draw is swap -> mirr
    PMIRRORFB->bind();
    PMIRRORSWAPFB->getTexture()->bind();

    // damage region will be scaled, make a temp
    CRegion tempDamage{damage};

    // and draw
    auto shader = useShader(m_shaders->frag[SH_FRAG_BLUR1]);
    for (auto i = 1; i <= BLUR_PASSES; ++i) {
        tempDamage = damage.copy().scale(1.f / (1 << i));
        drawPass(shader, SH_FRAG_BLUR1, &tempDamage); // down
    }

    shader = useShader(m_shaders->frag[SH_FRAG_BLUR2]);
    for (auto i = BLUR_PASSES - 1; i >= 0; --i) {
        tempDamage = damage.copy().scale(1.f / (1 << i)); // when upsampling we make the region twice as big
        drawPass(shader, SH_FRAG_BLUR2, &tempDamage);     // up
    }

    // finalize the image
    {
        static auto PBLURNOISE      = CConfigValue<Hyprlang::FLOAT>("decoration:blur:noise");
        static auto PBLURBRIGHTNESS = CConfigValue<Hyprlang::FLOAT>("decoration:blur:brightness");

        if (currentRenderToFB == PMIRRORFB)
            PMIRRORSWAPFB->bind();
        else
            PMIRRORFB->bind();

        glActiveTexture(GL_TEXTURE0);

        auto currentTex = currentRenderToFB->getTexture();

        currentTex->bind();

        currentTex->setTexParameter(GL_TEXTURE_MIN_FILTER, GL_LINEAR);

        auto shader = useShader(m_shaders->frag[SH_FRAG_BLURFINISH]);
        shader->setUniformMatrix3fv(SHADER_PROJ, 1, GL_TRUE, glMatrix.getMatrix());
        shader->setUniformFloat(SHADER_NOISE, *PBLURNOISE);
        shader->setUniformFloat(SHADER_BRIGHTNESS, *PBLURBRIGHTNESS);

        shader->setUniformInt(SHADER_TEX, 0);

        glBindVertexArray(shader->getUniformLocation(SHADER_SHADER_VAO));

        if (!damage.empty()) {
            damage.forEachRect([this](const auto& RECT) {
                scissor(&RECT, false /* this region is already transformed */);
                glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);
            });
        }

        glBindVertexArray(0);

        if (currentRenderToFB != PMIRRORFB)
            currentRenderToFB = PMIRRORFB;
        else
            currentRenderToFB = PMIRRORSWAPFB;
    }

    // finish
    PMIRRORFB->getTexture()->unbind();

    blend(BLENDBEFORE);

    return currentRenderToFB;
}

void CHyprOpenGLImpl::markBlurDirtyForMonitor(PHLMONITOR pMonitor) {
    m_monitorRenderResources[pMonitor].blurFBDirty = true;
}

void CHyprOpenGLImpl::preRender(PHLMONITOR pMonitor) {
    static auto PBLURNEWOPTIMIZE = CConfigValue<Hyprlang::INT>("decoration:blur:new_optimizations");
    static auto PBLURXRAY        = CConfigValue<Hyprlang::INT>("decoration:blur:xray");
    static auto PBLUR            = CConfigValue<Hyprlang::INT>("decoration:blur:enabled");

    if (!*PBLURNEWOPTIMIZE || !m_monitorRenderResources[pMonitor].blurFBDirty || !*PBLUR)
        return;

    // ignore if solitary present, nothing to blur
    if (!pMonitor->m_solitaryClient.expired())
        return;

    // check if we need to update the blur fb
    // if there are no windows that would benefit from it,
    // we will ignore that the blur FB is dirty.

    auto windowShouldBeBlurred = [&](PHLWINDOW pWindow) -> bool {
        if (!pWindow)
            return false;

        if (pWindow->m_ruleApplicator->noBlur().valueOrDefault())
            return false;

        if (pWindow->wlSurface()->small() && !pWindow->wlSurface()->m_fillIgnoreSmall)
            return true;

        const auto  PSURFACE = pWindow->wlSurface()->resource();

        const auto  PWORKSPACE = pWindow->m_workspace;
        const float A          = pWindow->m_alpha->value() * pWindow->m_activeInactiveAlpha->value() * PWORKSPACE->m_alpha->value();

        if (A >= 1.f) {
            // if (PSURFACE->opaque)
            //   return false;

            CRegion        inverseOpaque;

            pixman_box32_t surfbox = {0, 0, PSURFACE->m_current.size.x, PSURFACE->m_current.size.y};
            CRegion        opaqueRegion{PSURFACE->m_current.opaque};
            inverseOpaque.set(opaqueRegion).invert(&surfbox).intersect(0, 0, PSURFACE->m_current.size.x, PSURFACE->m_current.size.y);

            if (inverseOpaque.empty())
                return false;
        }

        return true;
    };

    bool hasWindows = false;
    for (auto const& w : g_pCompositor->m_windows) {
        if (w->m_workspace == pMonitor->m_activeWorkspace && !w->isHidden() && w->m_isMapped && (!w->m_isFloating || *PBLURXRAY)) {

            // check if window is valid
            if (!windowShouldBeBlurred(w))
                continue;

            hasWindows = true;
            break;
        }
    }

    for (auto const& m : g_pCompositor->m_monitors) {
        for (auto const& lsl : m->m_layerSurfaceLayers) {
            for (auto const& ls : lsl) {
                if (!ls->m_layerSurface || ls->m_ruleApplicator->xray().valueOrDefault() != 1)
                    continue;

                // if (ls->layerSurface->surface->opaque && ls->alpha->value() >= 1.f)
                //     continue;

                hasWindows = true;
                break;
            }
        }
    }

    if (!hasWindows)
        return;

    g_pHyprRenderer->damageMonitor(pMonitor);
    m_monitorRenderResources[pMonitor].blurFBShouldRender = true;
}

void CHyprOpenGLImpl::preBlurForCurrentMonitor() {

    TRACY_GPU_ZONE("RenderPreBlurForCurrentMonitor");

    const auto SAVEDRENDERMODIF = m_renderData.renderModif;
    m_renderData.renderModif    = {}; // fix shit

    // make the fake dmg
    CRegion    fakeDamage{0, 0, m_renderData.pMonitor->m_transformedSize.x, m_renderData.pMonitor->m_transformedSize.y};
    const auto POUTFB = blurMainFramebufferWithDamage(1, &fakeDamage);

    // render onto blurFB
    m_renderData.pCurrentMonData->blurFB.alloc(m_renderData.pMonitor->m_pixelSize.x, m_renderData.pMonitor->m_pixelSize.y,
                                               m_renderData.pMonitor->m_output->state->state().drmFormat);
    m_renderData.pCurrentMonData->blurFB.bind();

    clear(CHyprColor(0, 0, 0, 0));

    pushMonitorTransformEnabled(true);
    renderTextureInternal(POUTFB->getTexture(), CBox{0, 0, m_renderData.pMonitor->m_transformedSize.x, m_renderData.pMonitor->m_transformedSize.y},
                          STextureRenderData{.damage = &fakeDamage, .a = 1, .round = 0, .roundingPower = 2.F, .discardActive = false, .allowCustomUV = false, .noAA = true});
    popMonitorTransformEnabled();

    m_renderData.currentFB->bind();

    m_renderData.pCurrentMonData->blurFBDirty = false;

    m_renderData.renderModif = SAVEDRENDERMODIF;

    m_monitorRenderResources[m_renderData.pMonitor].blurFBShouldRender = false;
}

void CHyprOpenGLImpl::preWindowPass() {
    if (!preBlurQueued())
        return;

    g_pHyprRenderer->m_renderPass.add(makeUnique<CPreBlurElement>());
}

bool CHyprOpenGLImpl::preBlurQueued() {
    static auto PBLURNEWOPTIMIZE = CConfigValue<Hyprlang::INT>("decoration:blur:new_optimizations");
    static auto PBLUR            = CConfigValue<Hyprlang::INT>("decoration:blur:enabled");

    return m_renderData.pCurrentMonData->blurFBDirty && *PBLURNEWOPTIMIZE && *PBLUR && m_renderData.pCurrentMonData->blurFBShouldRender;
}

bool CHyprOpenGLImpl::shouldUseNewBlurOptimizations(PHLLS pLayer, PHLWINDOW pWindow) {
    static auto PBLURNEWOPTIMIZE = CConfigValue<Hyprlang::INT>("decoration:blur:new_optimizations");
    static auto PBLURXRAY        = CConfigValue<Hyprlang::INT>("decoration:blur:xray");

    if (!m_renderData.pCurrentMonData->blurFB.getTexture())
        return false;

    if (pWindow && pWindow->m_ruleApplicator->xray().hasValue() && !pWindow->m_ruleApplicator->xray().valueOrDefault())
        return false;

    if (pLayer && pLayer->m_ruleApplicator->xray().valueOrDefault() == 0)
        return false;

    if ((*PBLURNEWOPTIMIZE && pWindow && !pWindow->m_isFloating && !pWindow->onSpecialWorkspace()) || *PBLURXRAY)
        return true;

    if ((pLayer && pLayer->m_ruleApplicator->xray().valueOrDefault() == 1) || (pWindow && pWindow->m_ruleApplicator->xray().valueOrDefault()))
        return true;

    return false;
}

void CHyprOpenGLImpl::renderTextureWithBlurInternal(SP<CTexture> tex, const CBox& box, const STextureRenderData& data) {
    RASSERT(m_renderData.pMonitor, "Tried to render texture with blur without begin()!");

    TRACY_GPU_ZONE("RenderTextureWithBlur");

    // make a damage region for this window
    CRegion texDamage{m_renderData.damage};
    texDamage.intersect(box.x, box.y, box.width, box.height);

    // While renderTextureInternalWithDamage will clip the blur as well,
    // clipping texDamage here allows blur generation to be optimized.
    if (!m_renderData.clipRegion.empty())
        texDamage.intersect(m_renderData.clipRegion);

    if (texDamage.empty())
        return;

    m_renderData.renderModif.applyToRegion(texDamage);

    // amazing hack: the surface has an opaque region!
    CRegion inverseOpaque;
    if (data.a >= 1.f && data.surface && std::round(data.surface->m_current.size.x * m_renderData.pMonitor->m_scale) == box.w &&
        std::round(data.surface->m_current.size.y * m_renderData.pMonitor->m_scale) == box.h) {
        pixman_box32_t surfbox = {0, 0, data.surface->m_current.size.x * data.surface->m_current.scale, data.surface->m_current.size.y * data.surface->m_current.scale};
        inverseOpaque          = data.surface->m_current.opaque;
        inverseOpaque.invert(&surfbox).intersect(0, 0, data.surface->m_current.size.x * data.surface->m_current.scale,
                                                 data.surface->m_current.size.y * data.surface->m_current.scale);

        if (inverseOpaque.empty()) {
            renderTextureInternal(tex, box, data);
            return;
        }
    } else
        inverseOpaque = {0, 0, box.width, box.height};

    inverseOpaque.scale(m_renderData.pMonitor->m_scale);

    //   vvv TODO: layered blur fbs?
    const bool    USENEWOPTIMIZE = shouldUseNewBlurOptimizations(m_renderData.currentLS.lock(), m_renderData.currentWindow.lock()) && !data.blockBlurOptimization;

    CFramebuffer* POUTFB = nullptr;
    if (!USENEWOPTIMIZE) {
        inverseOpaque.translate(box.pos());
        m_renderData.renderModif.applyToRegion(inverseOpaque);
        inverseOpaque.intersect(texDamage);
        POUTFB = blurMainFramebufferWithDamage(data.a, &inverseOpaque);
    } else
        POUTFB = &m_renderData.pCurrentMonData->blurFB;

    m_renderData.currentFB->bind();

    const auto NEEDS_STENCIL = m_renderData.discardMode != 0;

    if (NEEDS_STENCIL) {
        scissor(nullptr); // allow the entire window and stencil to render
        glClearStencil(0);
        glClear(GL_STENCIL_BUFFER_BIT);

        setCapStatus(GL_STENCIL_TEST, true);

        glStencilFunc(GL_ALWAYS, 1, 0xFF);
        glStencilOp(GL_KEEP, GL_KEEP, GL_REPLACE);

        glColorMask(GL_FALSE, GL_FALSE, GL_FALSE, GL_FALSE);
        if (USENEWOPTIMIZE && !(m_renderData.discardMode & DISCARD_ALPHA))
            renderRect(box, CHyprColor(0, 0, 0, 0), SRectRenderData{.round = data.round, .roundingPower = data.roundingPower});
        else
            renderTexture(tex, box,
                          STextureRenderData{.a             = data.a,
                                             .round         = data.round,
                                             .roundingPower = data.roundingPower,
                                             .discardActive = true,
                                             .allowCustomUV = true,
                                             .wrapX         = data.wrapX,
                                             .wrapY         = data.wrapY}); // discard opaque
        glColorMask(GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE);

        glStencilFunc(GL_EQUAL, 1, 0xFF);
        glStencilOp(GL_KEEP, GL_KEEP, GL_REPLACE);
    }

    // stencil done. Render everything.
    const auto LASTTL = m_renderData.primarySurfaceUVTopLeft;
    const auto LASTBR = m_renderData.primarySurfaceUVBottomRight;

    CBox       transformedBox = box;
    transformedBox.transform(Math::wlTransformToHyprutils(Math::invertTransform(m_renderData.pMonitor->m_transform)), m_renderData.pMonitor->m_transformedSize.x,
                             m_renderData.pMonitor->m_transformedSize.y);

    CBox monitorSpaceBox = {transformedBox.pos().x / m_renderData.pMonitor->m_pixelSize.x * m_renderData.pMonitor->m_transformedSize.x,
                            transformedBox.pos().y / m_renderData.pMonitor->m_pixelSize.y * m_renderData.pMonitor->m_transformedSize.y,
                            transformedBox.width / m_renderData.pMonitor->m_pixelSize.x * m_renderData.pMonitor->m_transformedSize.x,
                            transformedBox.height / m_renderData.pMonitor->m_pixelSize.y * m_renderData.pMonitor->m_transformedSize.y};

    m_renderData.primarySurfaceUVTopLeft     = monitorSpaceBox.pos() / m_renderData.pMonitor->m_transformedSize;
    m_renderData.primarySurfaceUVBottomRight = (monitorSpaceBox.pos() + monitorSpaceBox.size()) / m_renderData.pMonitor->m_transformedSize;

    static auto PBLURIGNOREOPACITY = CConfigValue<Hyprlang::INT>("decoration:blur:ignore_opacity");
    pushMonitorTransformEnabled(true);
    if (!USENEWOPTIMIZE)
        setRenderModifEnabled(false);
    renderTextureInternal(POUTFB->getTexture(), box,
                          STextureRenderData{
                              .damage        = &texDamage,
                              .a             = (*PBLURIGNOREOPACITY ? data.blurA : data.a * data.blurA) * data.overallA,
                              .round         = data.round,
                              .roundingPower = data.roundingPower,
                              .discardActive = false,
                              .allowCustomUV = true,
                              .noAA          = false,
                              .wrapX         = data.wrapX,
                              .wrapY         = data.wrapY,
                          });
    if (!USENEWOPTIMIZE)
        setRenderModifEnabled(true);
    popMonitorTransformEnabled();

    m_renderData.primarySurfaceUVTopLeft     = LASTTL;
    m_renderData.primarySurfaceUVBottomRight = LASTBR;

    // draw window
    setCapStatus(GL_STENCIL_TEST, false);
    renderTextureInternal(tex, box,
                          STextureRenderData{
                              .damage        = &texDamage,
                              .a             = data.a * data.overallA,
                              .round         = data.round,
                              .roundingPower = data.roundingPower,
                              .discardActive = false,
                              .allowCustomUV = true,
                              .allowDim      = true,
                              .noAA          = false,
                              .wrapX         = data.wrapX,
                              .wrapY         = data.wrapY,
                          });

    m_renderData.currentFB->invalidate({GL_STENCIL_ATTACHMENT});
    scissor(nullptr);
}

void CHyprOpenGLImpl::renderBorder(const CBox& box, const CGradientValueData& grad, SBorderRenderData data) {
    RASSERT((box.width > 0 && box.height > 0), "Tried to render rect with width/height < 0!");
    RASSERT(m_renderData.pMonitor, "Tried to render rect without begin()!");

    TRACY_GPU_ZONE("RenderBorder");

    if (m_renderData.damage.empty())
        return;

    CBox newBox = box;
    m_renderData.renderModif.applyToBox(newBox);

    if (data.borderSize < 1)
        return;

    int scaledBorderSize = std::round(data.borderSize * m_renderData.pMonitor->m_scale);
    scaledBorderSize     = std::round(scaledBorderSize * m_renderData.renderModif.combinedScale());

    // adjust box
    newBox.x -= scaledBorderSize;
    newBox.y -= scaledBorderSize;
    newBox.width += 2 * scaledBorderSize;
    newBox.height += 2 * scaledBorderSize;

    float  round = data.round + (data.round == 0 ? 0 : scaledBorderSize);

    Mat3x3 matrix = m_renderData.monitorProjection.projectBox(
        newBox, Math::wlTransformToHyprutils(Math::invertTransform(!m_monitorTransformEnabled ? WL_OUTPUT_TRANSFORM_NORMAL : m_renderData.pMonitor->m_transform)), newBox.rot);
    Mat3x3     glMatrix = m_renderData.projection.copy().multiply(matrix);

    const auto BLEND = m_blend;
    blend(true);

    WP<CShader> shader;

    const bool  skipCM = !m_cmSupported || m_renderData.pMonitor->m_imageDescription->id() == DEFAULT_IMAGE_DESCRIPTION->id();
    if (!skipCM) {
        shader = useShader(m_shaders->frag[SH_FRAG_CM_BORDER1]);
        passCMUniforms(shader, DEFAULT_IMAGE_DESCRIPTION);
    } else
        shader = useShader(m_shaders->frag[SH_FRAG_BORDER1]);

    shader->setUniformMatrix3fv(SHADER_PROJ, 1, GL_TRUE, glMatrix.getMatrix());
    shader->setUniform4fv(SHADER_GRADIENT, grad.m_colorsOkLabA.size() / 4, grad.m_colorsOkLabA);
    shader->setUniformInt(SHADER_GRADIENT_LENGTH, grad.m_colorsOkLabA.size() / 4);
    shader->setUniformFloat(SHADER_ANGLE, sc<int>(grad.m_angle / (std::numbers::pi / 180.0)) % 360 * (std::numbers::pi / 180.0));
    shader->setUniformFloat(SHADER_ALPHA, data.a);
    shader->setUniformInt(SHADER_GRADIENT2_LENGTH, 0);

    CBox transformedBox = newBox;
    transformedBox.transform(Math::wlTransformToHyprutils(Math::invertTransform(m_renderData.pMonitor->m_transform)), m_renderData.pMonitor->m_transformedSize.x,
                             m_renderData.pMonitor->m_transformedSize.y);

    const auto TOPLEFT  = Vector2D(transformedBox.x, transformedBox.y);
    const auto FULLSIZE = Vector2D(transformedBox.width, transformedBox.height);

    shader->setUniformFloat2(SHADER_TOP_LEFT, sc<float>(TOPLEFT.x), sc<float>(TOPLEFT.y));
    shader->setUniformFloat2(SHADER_FULL_SIZE, sc<float>(FULLSIZE.x), sc<float>(FULLSIZE.y));
    shader->setUniformFloat2(SHADER_FULL_SIZE_UNTRANSFORMED, sc<float>(newBox.width), sc<float>(newBox.height));
    shader->setUniformFloat(SHADER_RADIUS, round);
    shader->setUniformFloat(SHADER_RADIUS_OUTER, data.outerRound == -1 ? round : data.outerRound);
    shader->setUniformFloat(SHADER_ROUNDING_POWER, data.roundingPower);
    shader->setUniformFloat(SHADER_THICK, scaledBorderSize);

    glBindVertexArray(shader->getUniformLocation(SHADER_SHADER_VAO));

    // calculate the border's region, which we need to render over. No need to run the shader on
    // things outside there
    CRegion borderRegion = m_renderData.damage.copy().intersect(newBox);
    borderRegion.subtract(box.copy().expand(-scaledBorderSize - round));

    if (m_renderData.clipBox.width != 0 && m_renderData.clipBox.height != 0)
        borderRegion.intersect(m_renderData.clipBox);

    if (!borderRegion.empty()) {
        borderRegion.forEachRect([this](const auto& RECT) {
            scissor(&RECT);
            glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);
        });
    }

    glBindVertexArray(0);

    blend(BLEND);
}

void CHyprOpenGLImpl::renderBorder(const CBox& box, const CGradientValueData& grad1, const CGradientValueData& grad2, float lerp, SBorderRenderData data) {
    RASSERT((box.width > 0 && box.height > 0), "Tried to render rect with width/height < 0!");
    RASSERT(m_renderData.pMonitor, "Tried to render rect without begin()!");

    TRACY_GPU_ZONE("RenderBorder2");

    if (m_renderData.damage.empty())
        return;

    CBox newBox = box;
    m_renderData.renderModif.applyToBox(newBox);

    if (data.borderSize < 1)
        return;

    int scaledBorderSize = std::round(data.borderSize * m_renderData.pMonitor->m_scale);
    scaledBorderSize     = std::round(scaledBorderSize * m_renderData.renderModif.combinedScale());

    // adjust box
    newBox.x -= scaledBorderSize;
    newBox.y -= scaledBorderSize;
    newBox.width += 2 * scaledBorderSize;
    newBox.height += 2 * scaledBorderSize;

    float  round = data.round + (data.round == 0 ? 0 : scaledBorderSize);

    Mat3x3 matrix = m_renderData.monitorProjection.projectBox(
        newBox, Math::wlTransformToHyprutils(Math::invertTransform(!m_monitorTransformEnabled ? WL_OUTPUT_TRANSFORM_NORMAL : m_renderData.pMonitor->m_transform)), newBox.rot);
    Mat3x3     glMatrix = m_renderData.projection.copy().multiply(matrix);

    const auto BLEND = m_blend;
    blend(true);

    WP<CShader> shader;
    const bool  skipCM = !m_cmSupported || m_renderData.pMonitor->m_imageDescription->id() == DEFAULT_IMAGE_DESCRIPTION->id();
    if (!skipCM) {
        shader = useShader(m_shaders->frag[SH_FRAG_CM_BORDER1]);
        passCMUniforms(shader, DEFAULT_IMAGE_DESCRIPTION);
    } else
        shader = useShader(m_shaders->frag[SH_FRAG_BORDER1]);

    shader->setUniformMatrix3fv(SHADER_PROJ, 1, GL_TRUE, glMatrix.getMatrix());
    shader->setUniform4fv(SHADER_GRADIENT, grad1.m_colorsOkLabA.size() / 4, grad1.m_colorsOkLabA);
    shader->setUniformInt(SHADER_GRADIENT_LENGTH, grad1.m_colorsOkLabA.size() / 4);
    shader->setUniformFloat(SHADER_ANGLE, sc<int>(grad1.m_angle / (std::numbers::pi / 180.0)) % 360 * (std::numbers::pi / 180.0));
    if (!grad2.m_colorsOkLabA.empty())
        shader->setUniform4fv(SHADER_GRADIENT2, grad2.m_colorsOkLabA.size() / 4, grad2.m_colorsOkLabA);
    shader->setUniformInt(SHADER_GRADIENT2_LENGTH, grad2.m_colorsOkLabA.size() / 4);
    shader->setUniformFloat(SHADER_ANGLE2, sc<int>(grad2.m_angle / (std::numbers::pi / 180.0)) % 360 * (std::numbers::pi / 180.0));
    shader->setUniformFloat(SHADER_ALPHA, data.a);
    shader->setUniformFloat(SHADER_GRADIENT_LERP, lerp);

    CBox transformedBox = newBox;
    transformedBox.transform(Math::wlTransformToHyprutils(Math::invertTransform(m_renderData.pMonitor->m_transform)), m_renderData.pMonitor->m_transformedSize.x,
                             m_renderData.pMonitor->m_transformedSize.y);

    const auto TOPLEFT  = Vector2D(transformedBox.x, transformedBox.y);
    const auto FULLSIZE = Vector2D(transformedBox.width, transformedBox.height);

    shader->setUniformFloat2(SHADER_TOP_LEFT, sc<float>(TOPLEFT.x), sc<float>(TOPLEFT.y));
    shader->setUniformFloat2(SHADER_FULL_SIZE, sc<float>(FULLSIZE.x), sc<float>(FULLSIZE.y));
    shader->setUniformFloat2(SHADER_FULL_SIZE_UNTRANSFORMED, sc<float>(newBox.width), sc<float>(newBox.height));
    shader->setUniformFloat(SHADER_RADIUS, round);
    shader->setUniformFloat(SHADER_RADIUS_OUTER, data.outerRound == -1 ? round : data.outerRound);
    shader->setUniformFloat(SHADER_ROUNDING_POWER, data.roundingPower);
    shader->setUniformFloat(SHADER_THICK, scaledBorderSize);

    glBindVertexArray(shader->getUniformLocation(SHADER_SHADER_VAO));

    // calculate the border's region, which we need to render over. No need to run the shader on
    // things outside there
    CRegion borderRegion = m_renderData.damage.copy().intersect(newBox);
    borderRegion.subtract(box.copy().expand(-scaledBorderSize - round));

    if (m_renderData.clipBox.width != 0 && m_renderData.clipBox.height != 0)
        borderRegion.intersect(m_renderData.clipBox);

    if (!borderRegion.empty()) {
        borderRegion.forEachRect([this](const auto& RECT) {
            scissor(&RECT);
            glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);
        });
    }

    glBindVertexArray(0);
    blend(BLEND);
}

void CHyprOpenGLImpl::renderRoundedShadow(const CBox& box, int round, float roundingPower, int range, const CHyprColor& color, float a) {
    RASSERT(m_renderData.pMonitor, "Tried to render shadow without begin()!");
    RASSERT((box.width > 0 && box.height > 0), "Tried to render shadow with width/height < 0!");

    if (m_renderData.damage.empty())
        return;

    TRACY_GPU_ZONE("RenderShadow");

    CBox newBox = box;
    m_renderData.renderModif.applyToBox(newBox);

    static auto PSHADOWPOWER = CConfigValue<Hyprlang::INT>("decoration:shadow:render_power");

    const auto  SHADOWPOWER = std::clamp(sc<int>(*PSHADOWPOWER), 1, 4);

    const auto  col = color;

    Mat3x3      matrix = m_renderData.monitorProjection.projectBox(
        newBox, Math::wlTransformToHyprutils(Math::invertTransform(!m_monitorTransformEnabled ? WL_OUTPUT_TRANSFORM_NORMAL : m_renderData.pMonitor->m_transform)), newBox.rot);
    Mat3x3 glMatrix = m_renderData.projection.copy().multiply(matrix);

    blend(true);

    auto       shader = useShader(m_shaders->frag[SH_FRAG_SHADOW]);
    const bool skipCM = !m_cmSupported || m_renderData.pMonitor->m_imageDescription->id() == DEFAULT_IMAGE_DESCRIPTION->id();
    shader->setUniformInt(SHADER_SKIP_CM, skipCM);
    if (!skipCM)
        passCMUniforms(shader, DEFAULT_IMAGE_DESCRIPTION);

    shader->setUniformMatrix3fv(SHADER_PROJ, 1, GL_TRUE, glMatrix.getMatrix());
    shader->setUniformFloat4(SHADER_COLOR, col.r, col.g, col.b, col.a * a);

    const auto TOPLEFT     = Vector2D(range + round, range + round);
    const auto BOTTOMRIGHT = Vector2D(newBox.width - (range + round), newBox.height - (range + round));
    const auto FULLSIZE    = Vector2D(newBox.width, newBox.height);

    // Rounded corners
    shader->setUniformFloat2(SHADER_TOP_LEFT, sc<float>(TOPLEFT.x), sc<float>(TOPLEFT.y));
    shader->setUniformFloat2(SHADER_BOTTOM_RIGHT, sc<float>(BOTTOMRIGHT.x), sc<float>(BOTTOMRIGHT.y));
    shader->setUniformFloat2(SHADER_FULL_SIZE, sc<float>(FULLSIZE.x), sc<float>(FULLSIZE.y));
    shader->setUniformFloat(SHADER_RADIUS, range + round);
    shader->setUniformFloat(SHADER_ROUNDING_POWER, roundingPower);
    shader->setUniformFloat(SHADER_RANGE, range);
    shader->setUniformFloat(SHADER_SHADOW_POWER, SHADOWPOWER);

    glBindVertexArray(shader->getUniformLocation(SHADER_SHADER_VAO));

    if (m_renderData.clipBox.width != 0 && m_renderData.clipBox.height != 0) {
        CRegion damageClip{m_renderData.clipBox.x, m_renderData.clipBox.y, m_renderData.clipBox.width, m_renderData.clipBox.height};
        damageClip.intersect(m_renderData.damage);

        if (!damageClip.empty()) {
            damageClip.forEachRect([this](const auto& RECT) {
                scissor(&RECT);
                glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);
            });
        }
    } else {
        m_renderData.damage.forEachRect([this](const auto& RECT) {
            scissor(&RECT);
            glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);
        });
    }

    glBindVertexArray(0);
}

void CHyprOpenGLImpl::saveBufferForMirror(const CBox& box) {

    if (!m_renderData.pCurrentMonData->monitorMirrorFB.isAllocated())
        m_renderData.pCurrentMonData->monitorMirrorFB.alloc(m_renderData.pMonitor->m_pixelSize.x, m_renderData.pMonitor->m_pixelSize.y,
                                                            m_renderData.pMonitor->m_output->state->state().drmFormat);

    m_renderData.pCurrentMonData->monitorMirrorFB.bind();

    blend(false);

    renderTexture(m_renderData.currentFB->getTexture(), box,
                  STextureRenderData{
                      .a             = 1.f,
                      .round         = 0,
                      .discardActive = false,
                      .allowCustomUV = false,
                  });

    blend(true);

    m_renderData.currentFB->bind();
}

void CHyprOpenGLImpl::renderMirrored() {

    auto         monitor  = m_renderData.pMonitor;
    auto         mirrored = monitor->m_mirrorOf;

    const double scale  = std::min(monitor->m_transformedSize.x / mirrored->m_transformedSize.x, monitor->m_transformedSize.y / mirrored->m_transformedSize.y);
    CBox         monbox = {0, 0, mirrored->m_transformedSize.x * scale, mirrored->m_transformedSize.y * scale};

    // transform box as it will be drawn on a transformed projection
    monbox.transform(Math::wlTransformToHyprutils(mirrored->m_transform), mirrored->m_transformedSize.x * scale, mirrored->m_transformedSize.y * scale);

    monbox.x = (monitor->m_transformedSize.x - monbox.w) / 2;
    monbox.y = (monitor->m_transformedSize.y - monbox.h) / 2;

    const auto PFB = &m_monitorRenderResources[mirrored].monitorMirrorFB;
    if (!PFB->isAllocated() || !PFB->getTexture())
        return;

    g_pHyprRenderer->m_renderPass.add(makeUnique<CClearPassElement>(CClearPassElement::SClearData{CHyprColor(0, 0, 0, 0)}));

    CTexPassElement::SRenderData data;
    data.tex               = PFB->getTexture();
    data.box               = monbox;
    data.replaceProjection = Mat3x3::identity()
                                 .translate(monitor->m_pixelSize / 2.0)
                                 .transform(Math::wlTransformToHyprutils(monitor->m_transform))
                                 .transform(Math::wlTransformToHyprutils(Math::invertTransform(mirrored->m_transform)))
                                 .translate(-monitor->m_transformedSize / 2.0);

    g_pHyprRenderer->m_renderPass.add(makeUnique<CTexPassElement>(std::move(data)));
}

void CHyprOpenGLImpl::renderSplash(cairo_t* const CAIRO, cairo_surface_t* const CAIROSURFACE, double offsetY, const Vector2D& size) {
    static auto           PSPLASHCOLOR = CConfigValue<Hyprlang::INT>("misc:col.splash");
    static auto           PSPLASHFONT  = CConfigValue<std::string>("misc:splash_font_family");
    static auto           FALLBACKFONT = CConfigValue<std::string>("misc:font_family");

    const auto            FONTFAMILY = *PSPLASHFONT != STRVAL_EMPTY ? *PSPLASHFONT : *FALLBACKFONT;
    const auto            FONTSIZE   = sc<int>(size.y / 76);
    const auto            COLOR      = CHyprColor(*PSPLASHCOLOR);

    PangoLayout*          layoutText = pango_cairo_create_layout(CAIRO);
    PangoFontDescription* pangoFD    = pango_font_description_new();

    pango_font_description_set_family_static(pangoFD, FONTFAMILY.c_str());
    pango_font_description_set_absolute_size(pangoFD, FONTSIZE * PANGO_SCALE);
    pango_font_description_set_style(pangoFD, PANGO_STYLE_NORMAL);
    pango_font_description_set_weight(pangoFD, PANGO_WEIGHT_NORMAL);
    pango_layout_set_font_description(layoutText, pangoFD);

    cairo_set_source_rgba(CAIRO, COLOR.r, COLOR.g, COLOR.b, COLOR.a);

    int textW = 0, textH = 0;
    pango_layout_set_text(layoutText, g_pCompositor->m_currentSplash.c_str(), -1);
    pango_layout_get_size(layoutText, &textW, &textH);
    textW /= PANGO_SCALE;
    textH /= PANGO_SCALE;

    cairo_move_to(CAIRO, (size.x - textW) / 2.0, size.y - textH - offsetY);
    pango_cairo_show_layout(CAIRO, layoutText);

    pango_font_description_free(pangoFD);
    g_object_unref(layoutText);

    cairo_surface_flush(CAIROSURFACE);
}

std::string CHyprOpenGLImpl::resolveAssetPath(const std::string& filename) {
    std::string fullPath;
    for (auto& e : ASSET_PATHS) {
        std::string     p = std::string{e} + "/hypr/" + filename;
        std::error_code ec;
        if (std::filesystem::exists(p, ec)) {
            fullPath = p;
            break;
        } else
            Log::logger->log(Log::DEBUG, "resolveAssetPath: looking at {} unsuccessful: ec {}", filename, ec.message());
    }

    if (fullPath.empty()) {
        m_failedAssetsNo++;
        Log::logger->log(Log::ERR, "resolveAssetPath: looking for {} failed (no provider found)", filename);
        return "";
    }

    return fullPath;
}

SP<CTexture> CHyprOpenGLImpl::loadAsset(const std::string& filename) {

    const std::string fullPath = resolveAssetPath(filename);

    if (fullPath.empty())
        return m_missingAssetTexture;

    const auto CAIROSURFACE = cairo_image_surface_create_from_png(fullPath.c_str());

    if (!CAIROSURFACE) {
        m_failedAssetsNo++;
        Log::logger->log(Log::ERR, "loadAsset: failed to load {} (corrupt / inaccessible / not png)", fullPath);
        return m_missingAssetTexture;
    }

    auto tex = texFromCairo(CAIROSURFACE);

    cairo_surface_destroy(CAIROSURFACE);

    return tex;
}

SP<CTexture> CHyprOpenGLImpl::texFromCairo(cairo_surface_t* cairo) {
    const auto CAIROFORMAT = cairo_image_surface_get_format(cairo);
    auto       tex         = makeShared<CTexture>();

    tex->allocate();
    tex->m_size = {cairo_image_surface_get_width(cairo), cairo_image_surface_get_height(cairo)};

    const GLint glIFormat = CAIROFORMAT == CAIRO_FORMAT_RGB96F ? GL_RGB32F : GL_RGBA;
    const GLint glFormat  = CAIROFORMAT == CAIRO_FORMAT_RGB96F ? GL_RGB : GL_RGBA;
    const GLint glType    = CAIROFORMAT == CAIRO_FORMAT_RGB96F ? GL_FLOAT : GL_UNSIGNED_BYTE;

    const auto  DATA = cairo_image_surface_get_data(cairo);
    tex->bind();
    tex->setTexParameter(GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    tex->setTexParameter(GL_TEXTURE_MIN_FILTER, GL_LINEAR);

    if (CAIROFORMAT != CAIRO_FORMAT_RGB96F) {
        tex->setTexParameter(GL_TEXTURE_SWIZZLE_R, GL_BLUE);
        tex->setTexParameter(GL_TEXTURE_SWIZZLE_B, GL_RED);
    }

    glTexImage2D(GL_TEXTURE_2D, 0, glIFormat, tex->m_size.x, tex->m_size.y, 0, glFormat, glType, DATA);

    return tex;
}

SP<CTexture> CHyprOpenGLImpl::renderText(const std::string& text, CHyprColor col, int pt, bool italic, const std::string& fontFamily, int maxWidth, int weight) {
    SP<CTexture>          tex = makeShared<CTexture>();

    static auto           FONT = CConfigValue<std::string>("misc:font_family");

    const auto            FONTFAMILY = fontFamily.empty() ? *FONT : fontFamily;
    const auto            FONTSIZE   = pt;
    const auto            COLOR      = col;

    auto                  CAIROSURFACE = cairo_image_surface_create(CAIRO_FORMAT_ARGB32, 1920, 1080 /* arbitrary, just for size */);
    auto                  CAIRO        = cairo_create(CAIROSURFACE);

    PangoLayout*          layoutText = pango_cairo_create_layout(CAIRO);
    PangoFontDescription* pangoFD    = pango_font_description_new();

    pango_font_description_set_family_static(pangoFD, FONTFAMILY.c_str());
    pango_font_description_set_absolute_size(pangoFD, FONTSIZE * PANGO_SCALE);
    pango_font_description_set_style(pangoFD, italic ? PANGO_STYLE_ITALIC : PANGO_STYLE_NORMAL);
    pango_font_description_set_weight(pangoFD, sc<PangoWeight>(weight));
    pango_layout_set_font_description(layoutText, pangoFD);

    cairo_set_source_rgba(CAIRO, COLOR.r, COLOR.g, COLOR.b, COLOR.a);

    int textW = 0, textH = 0;
    pango_layout_set_text(layoutText, text.c_str(), -1);

    if (maxWidth > 0) {
        pango_layout_set_width(layoutText, maxWidth * PANGO_SCALE);
        pango_layout_set_ellipsize(layoutText, PANGO_ELLIPSIZE_END);
    }

    pango_layout_get_size(layoutText, &textW, &textH);
    textW /= PANGO_SCALE;
    textH /= PANGO_SCALE;

    pango_font_description_free(pangoFD);
    g_object_unref(layoutText);
    cairo_destroy(CAIRO);
    cairo_surface_destroy(CAIROSURFACE);

    CAIROSURFACE = cairo_image_surface_create(CAIRO_FORMAT_ARGB32, textW, textH);
    CAIRO        = cairo_create(CAIROSURFACE);

    layoutText = pango_cairo_create_layout(CAIRO);
    pangoFD    = pango_font_description_new();

    pango_font_description_set_family_static(pangoFD, FONTFAMILY.c_str());
    pango_font_description_set_absolute_size(pangoFD, FONTSIZE * PANGO_SCALE);
    pango_font_description_set_style(pangoFD, italic ? PANGO_STYLE_ITALIC : PANGO_STYLE_NORMAL);
    pango_font_description_set_weight(pangoFD, sc<PangoWeight>(weight));
    pango_layout_set_font_description(layoutText, pangoFD);
    pango_layout_set_text(layoutText, text.c_str(), -1);

    cairo_set_source_rgba(CAIRO, COLOR.r, COLOR.g, COLOR.b, COLOR.a);

    cairo_move_to(CAIRO, 0, 0);
    pango_cairo_show_layout(CAIRO, layoutText);

    pango_font_description_free(pangoFD);
    g_object_unref(layoutText);

    cairo_surface_flush(CAIROSURFACE);

    tex->allocate();
    tex->m_size = {cairo_image_surface_get_width(CAIROSURFACE), cairo_image_surface_get_height(CAIROSURFACE)};

    const auto DATA = cairo_image_surface_get_data(CAIROSURFACE);
    tex->bind();
    tex->setTexParameter(GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    tex->setTexParameter(GL_TEXTURE_MIN_FILTER, GL_LINEAR);
    tex->setTexParameter(GL_TEXTURE_SWIZZLE_R, GL_BLUE);
    tex->setTexParameter(GL_TEXTURE_SWIZZLE_B, GL_RED);
    glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, tex->m_size.x, tex->m_size.y, 0, GL_RGBA, GL_UNSIGNED_BYTE, DATA);

    cairo_destroy(CAIRO);
    cairo_surface_destroy(CAIROSURFACE);

    return tex;
}

void CHyprOpenGLImpl::initMissingAssetTexture() {
    SP<CTexture> tex = makeShared<CTexture>();
    tex->allocate();

    const auto CAIROSURFACE = cairo_image_surface_create(CAIRO_FORMAT_ARGB32, 512, 512);
    const auto CAIRO        = cairo_create(CAIROSURFACE);

    cairo_set_antialias(CAIRO, CAIRO_ANTIALIAS_NONE);
    cairo_save(CAIRO);
    cairo_set_source_rgba(CAIRO, 0, 0, 0, 1);
    cairo_set_operator(CAIRO, CAIRO_OPERATOR_SOURCE);
    cairo_paint(CAIRO);
    cairo_set_source_rgba(CAIRO, 1, 0, 1, 1);
    cairo_rectangle(CAIRO, 256, 0, 256, 256);
    cairo_fill(CAIRO);
    cairo_rectangle(CAIRO, 0, 256, 256, 256);
    cairo_fill(CAIRO);
    cairo_restore(CAIRO);

    cairo_surface_flush(CAIROSURFACE);

    tex->m_size = {512, 512};

    // copy the data to an OpenGL texture we have
    const GLint glFormat = GL_RGBA;
    const GLint glType   = GL_UNSIGNED_BYTE;

    const auto  DATA = cairo_image_surface_get_data(CAIROSURFACE);
    tex->bind();
    tex->setTexParameter(GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    tex->setTexParameter(GL_TEXTURE_MIN_FILTER, GL_LINEAR);
    tex->setTexParameter(GL_TEXTURE_SWIZZLE_R, GL_BLUE);
    tex->setTexParameter(GL_TEXTURE_SWIZZLE_B, GL_RED);
    glTexImage2D(GL_TEXTURE_2D, 0, glFormat, tex->m_size.x, tex->m_size.y, 0, glFormat, glType, DATA);

    cairo_surface_destroy(CAIROSURFACE);
    cairo_destroy(CAIRO);

    m_missingAssetTexture = tex;
}

WP<CShader> CHyprOpenGLImpl::useShader(WP<CShader> prog) {
    if (m_currentProgram == prog->program())
        return prog;

    glUseProgram(prog->program());
    m_currentProgram = prog->program();

    return prog;
}

void CHyprOpenGLImpl::initAssets() {
    initMissingAssetTexture();

    m_screencopyDeniedTexture = renderText("Permission denied to share screen", Colors::WHITE, 20);
}

void CHyprOpenGLImpl::ensureLockTexturesRendered(bool load) {
    static bool loaded = false;

    if (loaded == load)
        return;

    loaded = load;

    if (load) {
        // this will cause a small hitch. I don't think we can do much, other than wasting VRAM and having this loaded all the time.
        m_lockDeadTexture  = loadAsset("lockdead.png");
        m_lockDead2Texture = loadAsset("lockdead2.png");

        const auto VT = g_pCompositor->getVTNr();

        m_lockTtyTextTexture = renderText(std::format("Running on tty {}", VT.has_value() ? std::to_string(*VT) : "unknown"), CHyprColor{0.9F, 0.9F, 0.9F, 0.7F}, 20, true);
    } else {
        m_lockDeadTexture.reset();
        m_lockDead2Texture.reset();
        m_lockTtyTextTexture.reset();
    }
}

void CHyprOpenGLImpl::requestBackgroundResource() {
    if (m_backgroundResource)
        return;

    static auto PNOWALLPAPER    = CConfigValue<Hyprlang::INT>("misc:disable_hyprland_logo");
    static auto PFORCEWALLPAPER = CConfigValue<Hyprlang::INT>("misc:force_default_wallpaper");

    const auto  FORCEWALLPAPER = std::clamp(*PFORCEWALLPAPER, sc<int64_t>(-1), sc<int64_t>(2));

    if (*PNOWALLPAPER)
        return;

    static bool        once    = true;
    static std::string texPath = "wall";

    if (once) {
        // get the adequate tex
        if (FORCEWALLPAPER == -1) {
            std::mt19937_64                 engine(time(nullptr));
            std::uniform_int_distribution<> distribution(0, 2);

            texPath += std::to_string(distribution(engine));
        } else
            texPath += std::to_string(std::clamp(*PFORCEWALLPAPER, sc<int64_t>(0), sc<int64_t>(2)));

        texPath += ".png";

        texPath = resolveAssetPath(texPath);

        once = false;
    }

    if (texPath.empty()) {
        m_backgroundResourceFailed = true;
        return;
    }

    m_backgroundResource = makeAtomicShared<Hyprgraphics::CImageResource>(texPath);

    // doesn't have to be ASP as it's passed
    SP<CMainLoopExecutor> executor = makeShared<CMainLoopExecutor>([] {
        for (const auto& m : g_pCompositor->m_monitors) {
            g_pHyprRenderer->damageMonitor(m);
        }
    });

    m_backgroundResource->m_events.finished.listenStatic([executor] {
        // this is in the worker thread.
        executor->signal();
    });

    g_pAsyncResourceGatherer->enqueue(m_backgroundResource);
}

void CHyprOpenGLImpl::createBGTextureForMonitor(PHLMONITOR pMonitor) {
    RASSERT(m_renderData.pMonitor, "Tried to createBGTex without begin()!");

    Log::logger->log(Log::DEBUG, "Creating a texture for BGTex");

    static auto PRENDERTEX = CConfigValue<Hyprlang::INT>("misc:disable_hyprland_logo");
    static auto PNOSPLASH  = CConfigValue<Hyprlang::INT>("misc:disable_splash_rendering");

    if (*PRENDERTEX || m_backgroundResourceFailed)
        return;

    if (!m_backgroundResource) {
        // queue the asset to be created
        requestBackgroundResource();
        return;
    }

    if (!m_backgroundResource->m_ready)
        return;

    // release the last tex if exists
    const auto PFB = &m_monitorBGFBs[pMonitor];
    PFB->release();

    PFB->alloc(pMonitor->m_pixelSize.x, pMonitor->m_pixelSize.y, pMonitor->m_output->state->state().drmFormat);

    // create a new one with cairo
    SP<CTexture> tex = makeShared<CTexture>();

    tex->allocate();

    const auto CAIROSURFACE = cairo_image_surface_create(CAIRO_FORMAT_ARGB32, pMonitor->m_pixelSize.x, pMonitor->m_pixelSize.y);
    const auto CAIRO        = cairo_create(CAIROSURFACE);

    cairo_set_antialias(CAIRO, CAIRO_ANTIALIAS_GOOD);
    cairo_save(CAIRO);
    cairo_set_source_rgba(CAIRO, 0, 0, 0, 0);
    cairo_set_operator(CAIRO, CAIRO_OPERATOR_SOURCE);
    cairo_paint(CAIRO);
    cairo_restore(CAIRO);

    if (!*PNOSPLASH)
        renderSplash(CAIRO, CAIROSURFACE, 0.02 * pMonitor->m_pixelSize.y, pMonitor->m_pixelSize);

    cairo_surface_flush(CAIROSURFACE);

    tex->m_size = pMonitor->m_pixelSize;

    // copy the data to an OpenGL texture we have
    const GLint glFormat = GL_RGBA;
    const GLint glType   = GL_UNSIGNED_BYTE;

    const auto  DATA = cairo_image_surface_get_data(CAIROSURFACE);
    tex->bind();
    tex->setTexParameter(GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    tex->setTexParameter(GL_TEXTURE_MIN_FILTER, GL_LINEAR);
    tex->setTexParameter(GL_TEXTURE_SWIZZLE_R, GL_BLUE);
    tex->setTexParameter(GL_TEXTURE_SWIZZLE_B, GL_RED);

    glTexImage2D(GL_TEXTURE_2D, 0, glFormat, tex->m_size.x, tex->m_size.y, 0, glFormat, glType, DATA);

    cairo_surface_destroy(CAIROSURFACE);
    cairo_destroy(CAIRO);

    // render the texture to our fb
    PFB->bind();
    CRegion fakeDamage{0, 0, INT16_MAX, INT16_MAX};

    blend(true);
    clear(CHyprColor{0, 0, 0, 1});

    SP<CTexture> backgroundTexture = texFromCairo(m_backgroundResource->m_asset.cairoSurface->cairo());

    // first render the background
    if (backgroundTexture) {
        const double MONRATIO = m_renderData.pMonitor->m_transformedSize.x / m_renderData.pMonitor->m_transformedSize.y;
        const double WPRATIO  = backgroundTexture->m_size.x / backgroundTexture->m_size.y;
        Vector2D     origin;
        double       scale = 1.0;

        if (MONRATIO > WPRATIO) {
            scale    = m_renderData.pMonitor->m_transformedSize.x / backgroundTexture->m_size.x;
            origin.y = (m_renderData.pMonitor->m_transformedSize.y - backgroundTexture->m_size.y * scale) / 2.0;
        } else {
            scale    = m_renderData.pMonitor->m_transformedSize.y / backgroundTexture->m_size.y;
            origin.x = (m_renderData.pMonitor->m_transformedSize.x - backgroundTexture->m_size.x * scale) / 2.0;
        }

        CBox texbox = CBox{origin, backgroundTexture->m_size * scale};
        renderTextureInternal(backgroundTexture, texbox, {.damage = &fakeDamage, .a = 1.0});
    }

    CBox monbox = {{}, pMonitor->m_pixelSize};
    renderTextureInternal(tex, monbox, {.damage = &fakeDamage, .a = 1.0});

    // bind back
    if (m_renderData.currentFB)
        m_renderData.currentFB->bind();

    Log::logger->log(Log::DEBUG, "Background created for monitor {}", pMonitor->m_name);

    // clear the resource after we're done using it
    g_pEventLoopManager->doLater([this] { m_backgroundResource.reset(); });

    // set the animation to start for fading this background in nicely
    pMonitor->m_backgroundOpacity->setValueAndWarp(0.F);
    *pMonitor->m_backgroundOpacity = 1.F;
}

void CHyprOpenGLImpl::clearWithTex() {
    RASSERT(m_renderData.pMonitor, "Tried to render BGtex without begin()!");

    static auto PBACKGROUNDCOLOR = CConfigValue<Hyprlang::INT>("misc:background_color");

    auto        TEXIT = m_monitorBGFBs.find(m_renderData.pMonitor);

    if (TEXIT == m_monitorBGFBs.end()) {
        createBGTextureForMonitor(m_renderData.pMonitor.lock());
        g_pHyprRenderer->m_renderPass.add(makeUnique<CClearPassElement>(CClearPassElement::SClearData{CHyprColor(*PBACKGROUNDCOLOR)}));
    }

    if (TEXIT != m_monitorBGFBs.end()) {
        CTexPassElement::SRenderData data;
        data.box          = {0, 0, m_renderData.pMonitor->m_transformedSize.x, m_renderData.pMonitor->m_transformedSize.y};
        data.a            = m_renderData.pMonitor->m_backgroundOpacity->value();
        data.flipEndFrame = true;
        data.tex          = TEXIT->second.getTexture();
        g_pHyprRenderer->m_renderPass.add(makeUnique<CTexPassElement>(std::move(data)));
    }
}

void CHyprOpenGLImpl::destroyMonitorResources(PHLMONITORREF pMonitor) {
    g_pHyprRenderer->makeEGLCurrent();

    if (!g_pHyprOpenGL)
        return;

    auto RESIT = g_pHyprOpenGL->m_monitorRenderResources.find(pMonitor);
    if (RESIT != g_pHyprOpenGL->m_monitorRenderResources.end()) {
        RESIT->second.mirrorFB.release();
        RESIT->second.offloadFB.release();
        RESIT->second.mirrorSwapFB.release();
        RESIT->second.monitorMirrorFB.release();
        RESIT->second.blurFB.release();
        RESIT->second.offMainFB.release();
        RESIT->second.stencilTex->destroyTexture();
        g_pHyprOpenGL->m_monitorRenderResources.erase(RESIT);
    }

    auto TEXIT = g_pHyprOpenGL->m_monitorBGFBs.find(pMonitor);
    if (TEXIT != g_pHyprOpenGL->m_monitorBGFBs.end()) {
        TEXIT->second.release();
        g_pHyprOpenGL->m_monitorBGFBs.erase(TEXIT);
    }

    if (pMonitor)
        Log::logger->log(Log::DEBUG, "Monitor {} -> destroyed all render data", pMonitor->m_name);
}

void CHyprOpenGLImpl::saveMatrix() {
    m_renderData.savedProjection = m_renderData.projection;
}

void CHyprOpenGLImpl::setMatrixScaleTranslate(const Vector2D& translate, const float& scale) {
    m_renderData.projection.scale(scale).translate(translate);
}

void CHyprOpenGLImpl::restoreMatrix() {
    m_renderData.projection = m_renderData.savedProjection;
}

void CHyprOpenGLImpl::bindOffMain() {
    if (!m_renderData.pCurrentMonData->offMainFB.isAllocated()) {
        m_renderData.pCurrentMonData->offMainFB.alloc(m_renderData.pMonitor->m_pixelSize.x, m_renderData.pMonitor->m_pixelSize.y,
                                                      m_renderData.pMonitor->m_output->state->state().drmFormat);

        m_renderData.pCurrentMonData->offMainFB.addStencil(m_renderData.pCurrentMonData->stencilTex);
    }

    m_renderData.pCurrentMonData->offMainFB.bind();
    clear(CHyprColor(0, 0, 0, 0));
    m_renderData.currentFB = &m_renderData.pCurrentMonData->offMainFB;
}

void CHyprOpenGLImpl::renderOffToMain(CFramebuffer* off) {
    CBox monbox = {0, 0, m_renderData.pMonitor->m_transformedSize.x, m_renderData.pMonitor->m_transformedSize.y};
    renderTexturePrimitive(off->getTexture(), monbox);
}

void CHyprOpenGLImpl::bindBackOnMain() {
    m_renderData.mainFB->bind();
    m_renderData.currentFB = m_renderData.mainFB;
}

void CHyprOpenGLImpl::pushMonitorTransformEnabled(bool enabled) {
    m_monitorTransformStack.push(enabled);
    m_monitorTransformEnabled = enabled;
}

void CHyprOpenGLImpl::popMonitorTransformEnabled() {
    m_monitorTransformStack.pop();
    m_monitorTransformEnabled = m_monitorTransformStack.top();
}

void CHyprOpenGLImpl::setRenderModifEnabled(bool enabled) {
    m_renderData.renderModif.enabled = enabled;
}

void CHyprOpenGLImpl::setViewport(GLint x, GLint y, GLsizei width, GLsizei height) {
    if (m_lastViewport.x == x && m_lastViewport.y == y && m_lastViewport.width == width && m_lastViewport.height == height)
        return;

    glViewport(x, y, width, height);
    m_lastViewport = {.x = x, .y = y, .width = width, .height = height};
}

void CHyprOpenGLImpl::setCapStatus(int cap, bool status) {
    const auto getCapIndex = [cap]() {
        switch (cap) {
            case GL_BLEND: return CAP_STATUS_BLEND;
            case GL_SCISSOR_TEST: return CAP_STATUS_SCISSOR_TEST;
            case GL_STENCIL_TEST: return CAP_STATUS_STENCIL_TEST;
            default: return CAP_STATUS_END;
        }
    };

    auto idx = getCapIndex();

    if (idx == CAP_STATUS_END) {
        if (status)
            glEnable(cap);
        else
            glDisable(cap);

        return;
    }

    if (m_capStatus[idx] == status)
        return;

    if (status) {
        m_capStatus[idx] = status;
        glEnable(cap);
    } else {
        m_capStatus[idx] = status;
        glDisable(cap);
    }
}

uint32_t CHyprOpenGLImpl::getPreferredReadFormat(PHLMONITOR pMonitor) {
    static const auto PFORCE8BIT = CConfigValue<Hyprlang::INT>("misc:screencopy_force_8b");

    if (!*PFORCE8BIT)
        return pMonitor->m_output->state->state().drmFormat;

    auto fmt = pMonitor->m_output->state->state().drmFormat;

    if (fmt == DRM_FORMAT_BGRA1010102 || fmt == DRM_FORMAT_ARGB2101010 || fmt == DRM_FORMAT_XRGB2101010 || fmt == DRM_FORMAT_BGRX1010102 || fmt == DRM_FORMAT_XBGR2101010)
        return DRM_FORMAT_XRGB8888;

    return fmt;
}

bool CHyprOpenGLImpl::explicitSyncSupported() {
    return m_exts.EGL_ANDROID_native_fence_sync_ext;
}

WP<CShader> CHyprOpenGLImpl::getSurfaceShader(uint8_t features) {
    if (!m_shaders->fragVariants.contains(features)) {

        auto shader                    = makeShared<CShader>();
        auto includes                  = m_includes;
        includes["get_rgb_pixel.glsl"] = includes[features & SH_FEAT_RGBA ? "get_rgba_pixel.glsl" : "get_rgbx_pixel.glsl"];
        if (!(features & SH_FEAT_DISCARD)) {
            includes["discard.glsl"]    = "";
            includes["do_discard.glsl"] = "";
        }
        if (!(features & SH_FEAT_TINT)) {
            includes["tint.glsl"]    = "";
            includes["do_tint.glsl"] = "";
        }
        if (!(features & SH_FEAT_ROUNDING)) {
            includes["rounding.glsl"]    = "";
            includes["do_rounding.glsl"] = "";
        }
        if (!(features & SH_FEAT_CM)) {
            includes["surface_CM.glsl"] = "";
            includes["CM.glsl"]         = "";
            includes["do_CM.glsl"]      = "";
        }
        if (!(features & SH_FEAT_TONEMAP)) {
            includes["tonemap.glsl"]    = "";
            includes["do_tonemap.glsl"] = "";
        }
        if (!(features & SH_FEAT_SDR_MOD)) {
            includes["sdr_mod.glsl"]    = "";
            includes["do_sdr_mod.glsl"] = "";
        }
        if (!(features & SH_FEAT_TONEMAP || features & SH_FEAT_SDR_MOD))
            includes["primaries_xyz.glsl"] = includes["primaries_xyz_const.glsl"];

        Log::logger->log(Log::INFO, "getSurfaceShader: compiling feature set {}", features);
        const auto fragSrc = processShader("surface.frag", includes, MAX_INCLUDE_DEPTH);
        if (shader->createProgram(m_shaders->TEXVERTSRC, fragSrc, true, true)) {
            m_shaders->fragVariants[features] = shader;
            return shader;
        } else {
            Log::logger->log(Log::ERR, "getSurfaceShader failed for {}. Falling back to old branching", features);
            m_shaders->fragVariants[features] = nullptr;
        }
    }

    ASSERT(m_shaders->fragVariants[features]);
    return m_shaders->fragVariants[features];
}

std::vector<SDRMFormat> CHyprOpenGLImpl::getDRMFormats() {
    return m_drmFormats;
}

void SRenderModifData::applyToBox(CBox& box) {
    if (!enabled)
        return;

    for (auto const& [type, val] : modifs) {
        try {
            switch (type) {
                case RMOD_TYPE_SCALE: box.scale(std::any_cast<float>(val)); break;
                case RMOD_TYPE_SCALECENTER: box.scaleFromCenter(std::any_cast<float>(val)); break;
                case RMOD_TYPE_TRANSLATE: box.translate(std::any_cast<Vector2D>(val)); break;
                case RMOD_TYPE_ROTATE: box.rot += std::any_cast<float>(val); break;
                case RMOD_TYPE_ROTATECENTER: {
                    const auto   THETA = std::any_cast<float>(val);
                    const double COS   = std::cos(THETA);
                    const double SIN   = std::sin(THETA);
                    box.rot += THETA;
                    const auto OLDPOS = box.pos();
                    box.x             = OLDPOS.x * COS - OLDPOS.y * SIN;
                    box.y             = OLDPOS.y * COS + OLDPOS.x * SIN;
                }
            }
        } catch (std::bad_any_cast& e) { Log::logger->log(Log::ERR, "BUG THIS OR PLUGIN ERROR: caught a bad_any_cast in SRenderModifData::applyToBox!"); }
    }
}

void SRenderModifData::applyToRegion(CRegion& rg) {
    if (!enabled)
        return;

    for (auto const& [type, val] : modifs) {
        try {
            switch (type) {
                case RMOD_TYPE_SCALE: rg.scale(std::any_cast<float>(val)); break;
                case RMOD_TYPE_SCALECENTER: rg.scale(std::any_cast<float>(val)); break;
                case RMOD_TYPE_TRANSLATE: rg.translate(std::any_cast<Vector2D>(val)); break;
                case RMOD_TYPE_ROTATE: /* TODO */
                case RMOD_TYPE_ROTATECENTER: break;
            }
        } catch (std::bad_any_cast& e) { Log::logger->log(Log::ERR, "BUG THIS OR PLUGIN ERROR: caught a bad_any_cast in SRenderModifData::applyToRegion!"); }
    }
}

float SRenderModifData::combinedScale() {
    if (!enabled)
        return 1;

    float scale = 1.f;
    for (auto const& [type, val] : modifs) {
        try {
            switch (type) {
                case RMOD_TYPE_SCALE: scale *= std::any_cast<float>(val); break;
                case RMOD_TYPE_SCALECENTER:
                case RMOD_TYPE_TRANSLATE:
                case RMOD_TYPE_ROTATE:
                case RMOD_TYPE_ROTATECENTER: break;
            }
        } catch (std::bad_any_cast& e) { Log::logger->log(Log::ERR, "BUG THIS OR PLUGIN ERROR: caught a bad_any_cast in SRenderModifData::combinedScale!"); }
    }
    return scale;
}

UP<CEGLSync> CEGLSync::create() {
    RASSERT(g_pHyprOpenGL->m_exts.EGL_ANDROID_native_fence_sync_ext, "Tried to create an EGL sync when syncs are not supported on the gpu");

    EGLSyncKHR sync = g_pHyprOpenGL->m_proc.eglCreateSyncKHR(g_pHyprOpenGL->m_eglDisplay, EGL_SYNC_NATIVE_FENCE_ANDROID, nullptr);

    if UNLIKELY (sync == EGL_NO_SYNC_KHR) {
        Log::logger->log(Log::ERR, "eglCreateSyncKHR failed");
        return nullptr;
    }

    // we need to flush otherwise we might not get a valid fd
    glFlush();

    int fd = g_pHyprOpenGL->m_proc.eglDupNativeFenceFDANDROID(g_pHyprOpenGL->m_eglDisplay, sync);
    if UNLIKELY (fd == EGL_NO_NATIVE_FENCE_FD_ANDROID) {
        Log::logger->log(Log::ERR, "eglDupNativeFenceFDANDROID failed");
        return nullptr;
    }

    UP<CEGLSync> eglSync(new CEGLSync);
    eglSync->m_fd    = CFileDescriptor(fd);
    eglSync->m_sync  = sync;
    eglSync->m_valid = true;

    return eglSync;
}

CEGLSync::~CEGLSync() {
    if UNLIKELY (m_sync == EGL_NO_SYNC_KHR)
        return;

    if UNLIKELY (g_pHyprOpenGL && g_pHyprOpenGL->m_proc.eglDestroySyncKHR(g_pHyprOpenGL->m_eglDisplay, m_sync) != EGL_TRUE)
        Log::logger->log(Log::ERR, "eglDestroySyncKHR failed");
}

CFileDescriptor& CEGLSync::fd() {
    return m_fd;
}

CFileDescriptor&& CEGLSync::takeFd() {
    return std::move(m_fd);
}

bool CEGLSync::isValid() {
    return m_valid && m_sync != EGL_NO_SYNC_KHR && m_fd.isValid();
}
