/*
 * iDescriptor: A free and open-source idevice management tool.
 *
 * Copyright (C) 2025 Uncore <https://github.com/uncor3>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

#include "devicedatabase.h"
#include <string>

// https://github.com/libimobiledevice/libirecovery/blob/master/src/libirecovery.c
const DeviceDatabaseInfo DeviceDatabase::m_devices[] = {
    /* iPhone */
    {"iPhone1,1", "m68ap", 0x00, 0x8900, "iPhone 2G", "iPhone 2G"},
    {"iPhone1,2", "n82ap", 0x04, 0x8900, "iPhone 3G", "iPhone 3G"},
    {"iPhone2,1", "n88ap", 0x00, 0x8920, "iPhone 3Gs", "iPhone 3Gs"},
    {"iPhone3,1", "n90ap", 0x00, 0x8930, "iPhone 4 (GSM)", "iPhone 4"},
    {"iPhone3,2", "n90bap", 0x04, 0x8930, "iPhone 4 (GSM) R2 2012", "iPhone 4"},
    {"iPhone3,3", "n92ap", 0x06, 0x8930, "iPhone 4 (CDMA)", "iPhone 4"},
    {"iPhone4,1", "n94ap", 0x08, 0x8940, "iPhone 4s", "iPhone 4s"},
    {"iPhone5,1", "n41ap", 0x00, 0x8950, "iPhone 5 (GSM)", "iPhone 5"},
    {"iPhone5,2", "n42ap", 0x02, 0x8950, "iPhone 5 (Global)", "iPhone 5"},
    {"iPhone5,3", "n48ap", 0x0a, 0x8950, "iPhone 5c (GSM)", "iPhone 5c"},
    {"iPhone5,4", "n49ap", 0x0e, 0x8950, "iPhone 5c (Global)", "iPhone 5c"},
    {"iPhone6,1", "n51ap", 0x00, 0x8960, "iPhone 5s (GSM)", "iPhone 5s"},
    {"iPhone6,2", "n53ap", 0x02, 0x8960, "iPhone 5s (Global)", "iPhone 5s"},
    {"iPhone7,1", "n56ap", 0x04, 0x7000, "iPhone 6 Plus", "iPhone 6 Plus"},
    {"iPhone7,2", "n61ap", 0x06, 0x7000, "iPhone 6", "iPhone 6"},
    {"iPhone8,1", "n71ap", 0x04, 0x8000, "iPhone 6s", "iPhone 6s"},
    {"iPhone8,1", "n71map", 0x04, 0x8003, "iPhone 6s", "iPhone 6s"},
    {"iPhone8,2", "n66ap", 0x06, 0x8000, "iPhone 6s Plus", "iPhone 6s Plus"},
    {"iPhone8,2", "n66map", 0x06, 0x8003, "iPhone 6s Plus", "iPhone 6s Plus"},
    {"iPhone8,4", "n69ap", 0x02, 0x8003, "iPhone SE (1st gen)", "iPhone SE"},
    {"iPhone8,4", "n69uap", 0x02, 0x8000, "iPhone SE (1st gen)", "iPhone SE"},
    {"iPhone9,1", "d10ap", 0x08, 0x8010, "iPhone 7 (Global)", "iPhone 7"},
    {"iPhone9,2", "d11ap", 0x0a, 0x8010, "iPhone 7 Plus (Global)",
     "iPhone 7 Plus"},
    {"iPhone9,3", "d101ap", 0x0c, 0x8010, "iPhone 7 (GSM)", "iPhone 7"},
    {"iPhone9,4", "d111ap", 0x0e, 0x8010, "iPhone 7 Plus (GSM)",
     "iPhone 7 Plus"},
    {"iPhone10,1", "d20ap", 0x02, 0x8015, "iPhone 8 (Global)", "iPhone 8"},
    {"iPhone10,2", "d21ap", 0x04, 0x8015, "iPhone 8 Plus (Global)",
     "iPhone 8 Plus"},
    {"iPhone10,3", "d22ap", 0x06, 0x8015, "iPhone X (Global)", "iPhone X"},
    {"iPhone10,4", "d201ap", 0x0a, 0x8015, "iPhone 8 (GSM)", "iPhone 8"},
    {"iPhone10,5", "d211ap", 0x0c, 0x8015, "iPhone 8 Plus (GSM)",
     "iPhone 8 Plus"},
    {"iPhone10,6", "d221ap", 0x0e, 0x8015, "iPhone X (GSM)", "iPhone X"},
    {"iPhone11,2", "d321ap", 0x0e, 0x8020, "iPhone XS", "iPhone XS"},
    {"iPhone11,4", "d331ap", 0x0a, 0x8020, "iPhone XS Max (China)",
     "iPhone XS Max"},
    {"iPhone11,6", "d331pap", 0x1a, 0x8020, "iPhone XS Max", "iPhone XS Max"},
    {"iPhone11,8", "n841ap", 0x0c, 0x8020, "iPhone XR", "iPhone XR"},
    {"iPhone12,1", "n104ap", 0x04, 0x8030, "iPhone 11", "iPhone 11"},
    {"iPhone12,3", "d421ap", 0x06, 0x8030, "iPhone 11 Pro", "iPhone 11 Pro"},
    {"iPhone12,5", "d431ap", 0x02, 0x8030, "iPhone 11 Pro Max",
     "iPhone 11 Pro Max"},
    {"iPhone12,8", "d79ap", 0x10, 0x8030, "iPhone SE (2nd gen)",
     "iPhone SE 2nd gen"},
    {"iPhone13,1", "d52gap", 0x0A, 0x8101, "iPhone 12 mini", "iPhone 12 mini"},
    {"iPhone13,2", "d53gap", 0x0C, 0x8101, "iPhone 12", "iPhone 12"},
    {"iPhone13,3", "d53pap", 0x0E, 0x8101, "iPhone 12 Pro", "iPhone 12 Pro"},
    {"iPhone13,4", "d54pap", 0x08, 0x8101, "iPhone 12 Pro Max",
     "iPhone 12 Pro Max"},
    {"iPhone14,2", "d63ap", 0x0C, 0x8110, "iPhone 13 Pro", "iPhone 13 Pro"},
    {"iPhone14,3", "d64ap", 0x0E, 0x8110, "iPhone 13 Pro Max",
     "iPhone 13 Pro Max"},
    {"iPhone14,4", "d16ap", 0x08, 0x8110, "iPhone 13 mini", "iPhone 13 mini"},
    {"iPhone14,5", "d17ap", 0x0A, 0x8110, "iPhone 13", "iPhone 13"},
    {"iPhone14,6", "d49ap", 0x10, 0x8110, "iPhone SE (3rd gen)",
     "iPhone SE 3rd gen"},
    {"iPhone14,7", "d27ap", 0x18, 0x8110, "iPhone 14", "iPhone 14"},
    {"iPhone14,8", "d28ap", 0x1A, 0x8110, "iPhone 14 Plus", "iPhone 14 Plus"},
    {"iPhone15,2", "d73ap", 0x0C, 0x8120, "iPhone 14 Pro", "iPhone 14 Pro"},
    {"iPhone15,3", "d74ap", 0x0E, 0x8120, "iPhone 14 Pro Max",
     "iPhone 14 Pro Max"},
    {"iPhone15,4", "d37ap", 0x08, 0x8120, "iPhone 15", "iPhone 15"},
    {"iPhone15,5", "d38ap", 0x0A, 0x8120, "iPhone 15 Plus", "iPhone 15 Plus"},
    {"iPhone16,1", "d83ap", 0x04, 0x8130, "iPhone 15 Pro", "iPhone 15 Pro"},
    {"iPhone16,2", "d84ap", 0x06, 0x8130, "iPhone 15 Pro Max",
     "iPhone 15 Pro Max"},
    {"iPhone17,1", "d93ap", 0x0C, 0x8140, "iPhone 16 Pro", "iPhone 16 Pro"},
    {"iPhone17,2", "d94ap", 0x0E, 0x8140, "iPhone 16 Pro Max",
     "iPhone 16 Pro Max"},
    {"iPhone17,3", "d47ap", 0x08, 0x8140, "iPhone 16", "iPhone 16"},
    {"iPhone17,4", "d48ap", 0x0A, 0x8140, "iPhone 16 Plus", "iPhone 16 Plus"},
    {"iPhone17,5", "v59ap", 0x04, 0x8140, "iPhone 16e", "iPhone 16e"},
    {"iPhone18,1", "v53ap", 0x0C, 0x8150, "iPhone 17 Pro", "iPhone 17 Pro"},
    {"iPhone18,2", "v54ap", 0x0E, 0x8150, "iPhone 17 Pro Max",
     "iPhone 17 Pro Max"},
    {"iPhone18,3", "v57ap", 0x08, 0x8150, "iPhone 17", "iPhone 17"},
    {"iPhone18,4", "d23ap", 0x0A, 0x8150, "iPhone Air", "iPhone Air"},
    /* iPod */
    {"iPod1,1", "n45ap", 0x02, 0x8900, "iPod Touch (1st gen)",
     "iPod Touch 1st gen"},
    {"iPod2,1", "n72ap", 0x00, 0x8720, "iPod Touch (2nd gen)",
     "iPod Touch 2nd gen"},
    {"iPod3,1", "n18ap", 0x02, 0x8922, "iPod Touch (3rd gen)",
     "iPod Touch 3rd gen"},
    {"iPod4,1", "n81ap", 0x08, 0x8930, "iPod Touch (4th gen)",
     "iPod Touch 4th gen"},
    {"iPod5,1", "n78ap", 0x00, 0x8942, "iPod Touch (5th gen)",
     "iPod Touch 5th gen"},
    {"iPod7,1", "n102ap", 0x10, 0x7000, "iPod Touch (6th gen)",
     "iPod Touch 6th gen"},
    {"iPod9,1", "n112ap", 0x16, 0x8010, "iPod Touch (7th gen)",
     "iPod Touch 7th gen"},
    /* iPad */
    {"iPad1,1", "k48ap", 0x02, 0x8930, "iPad"},
    {"iPad2,1", "k93ap", 0x04, 0x8940, "iPad 2 (WiFi)", "iPad 2"},
    {"iPad2,2", "k94ap", 0x06, 0x8940, "iPad 2 (GSM)", "iPad 2"},
    {"iPad2,3", "k95ap", 0x02, 0x8940, "iPad 2 (CDMA)", "iPad 2"},
    {"iPad2,4", "k93aap", 0x06, 0x8942, "iPad 2 (WiFi) R2 2012", "iPad 2"},
    {"iPad2,5", "p105ap", 0x0a, 0x8942, "iPad mini (WiFi)", "iPad mini"},
    {"iPad2,6", "p106ap", 0x0c, 0x8942, "iPad mini (GSM)", "iPad mini"},
    {"iPad2,7", "p107ap", 0x0e, 0x8942, "iPad mini (Global)", "iPad mini"},
    {"iPad3,1", "j1ap", 0x00, 0x8945, "iPad (3rd gen, WiFi)", "iPad (3rd gen)"},
    {"iPad3,2", "j2ap", 0x02, 0x8945, "iPad (3rd gen, CDMA)", "iPad (3rd gen)"},
    {"iPad3,3", "j2aap", 0x04, 0x8945, "iPad (3rd gen, GSM)", "iPad (3rd gen)"},
    {"iPad3,4", "p101ap", 0x00, 0x8955, "iPad (4th gen, WiFi)",
     "iPad (4th gen)"},
    {"iPad3,5", "p102ap", 0x02, 0x8955, "iPad (4th gen, GSM)",
     "iPad (4th gen)"},
    {"iPad3,6", "p103ap", 0x04, 0x8955, "iPad (4th gen, Global)",
     "iPad (4th gen)"},
    {"iPad4,1", "j71ap", 0x10, 0x8960, "iPad Air (WiFi)", "iPad Air"},
    {"iPad4,2", "j72ap", 0x12, 0x8960, "iPad Air (Cellular)", "iPad Air"},
    {"iPad4,3", "j73ap", 0x14, 0x8960, "iPad Air (China)", "iPad Air"},
    {"iPad4,4", "j85ap", 0x0a, 0x8960, "iPad mini 2 (WiFi)", "iPad mini 2"},
    {"iPad4,5", "j86ap", 0x0c, 0x8960, "iPad mini 2 (Cellular)", "iPad mini 2"},
    {"iPad4,6", "j87ap", 0x0e, 0x8960, "iPad mini 2 (China)", "iPad mini 2"},
    {"iPad4,7", "j85map", 0x32, 0x8960, "iPad mini 3 (WiFi)", "iPad mini 3"},
    {"iPad4,8", "j86map", 0x34, 0x8960, "iPad mini 3 (Cellular)",
     "iPad mini 3"},
    {"iPad4,9", "j87map", 0x36, 0x8960, "iPad mini 3 (China)", "iPad mini 3"},
    {"iPad5,1", "j96ap", 0x08, 0x7000, "iPad mini 4 (WiFi)", "iPad mini 4"},
    {"iPad5,2", "j97ap", 0x0A, 0x7000, "iPad mini 4 (Cellular)", "iPad mini 4"},
    {"iPad5,3", "j81ap", 0x06, 0x7001, "iPad Air 2 (WiFi)", "iPad Air 2"},
    {"iPad5,4", "j82ap", 0x02, 0x7001, "iPad Air 2 (Cellular)", "iPad Air 2"},
    {"iPad6,3", "j127ap", 0x08, 0x8001, "iPad Pro 9.7-inch (WiFi)",
     "iPad Pro 9.7-inch"},
    {"iPad6,4", "j128ap", 0x0a, 0x8001, "iPad Pro 9.7-inch (Cellular)",
     "iPad Pro 9.7-inch"},
    {"iPad6,7", "j98aap", 0x10, 0x8001, "iPad Pro 12.9-inch (1st gen, WiFi)",
     "iPad Pro 12.9-inch (1st gen)"},
    {"iPad6,8", "j99aap", 0x12, 0x8001,
     "iPad Pro 12.9-inch (1st gen, Cellular)", "iPad Pro 12.9-inch (1st gen)"},
    {"iPad6,11", "j71sap", 0x10, 0x8000, "iPad (5th gen, WiFi)",
     "iPad (5th gen)"},
    {"iPad6,11", "j71tap", 0x10, 0x8003, "iPad (5th gen, WiFi)",
     "iPad (5th gen)"},
    {"iPad6,12", "j72sap", 0x12, 0x8000, "iPad (5th gen, Cellular)",
     "iPad (5th gen)"},
    {"iPad6,12", "j72tap", 0x12, 0x8003, "iPad (5th gen, Cellular)",
     "iPad (5th gen)"},
    {"iPad7,1", "j120ap", 0x0C, 0x8011, "iPad Pro 12.9-inch (2nd gen, WiFi)",
     "iPad Pro 12.9-inch (2nd gen)"},
    {"iPad7,2", "j121ap", 0x0E, 0x8011,
     "iPad Pro 12.9-inch (2nd gen, Cellular)", "iPad Pro 12.9-inch (2nd gen)"},
    {"iPad7,3", "j207ap", 0x04, 0x8011, "iPad Pro 10.5-inch (WiFi)",
     "iPad Pro 10.5-inch"},
    {"iPad7,4", "j208ap", 0x06, 0x8011, "iPad Pro 10.5-inch (Cellular)",
     "iPad Pro 10.5-inch"},
    {"iPad7,5", "j71bap", 0x18, 0x8010, "iPad (6th gen, WiFi)",
     "iPad (6th gen)"},
    {"iPad7,6", "j72bap", 0x1A, 0x8010, "iPad (6th gen, Cellular)",
     "iPad (6th gen)"},
    {"iPad7,11", "j171ap", 0x1C, 0x8010, "iPad (7th gen, WiFi)",
     "iPad (7th gen)"},
    {"iPad7,12", "j172ap", 0x1E, 0x8010, "iPad (7th gen, Cellular)",
     "iPad (7th gen)"},
    {"iPad8,1", "j317ap", 0x0C, 0x8027, "iPad Pro 11-inch (1st gen, WiFi)",
     "iPad Pro 11-inch (1st gen)"},
    {"iPad8,2", "j317xap", 0x1C, 0x8027,
     "iPad Pro 11-inch (1st gen, WiFi, 1TB)"},
    {"iPad8,3", "j318ap", 0x0E, 0x8027, "iPad Pro 11-inch (1st gen, Cellular)"},
    {"iPad8,4", "j318xap", 0x1E, 0x8027,
     "iPad Pro 11-inch (1st gen, Cellular, 1TB)"},
    {"iPad8,5", "j320ap", 0x08, 0x8027, "iPad Pro 12.9-inch (3rd gen, WiFi)"},
    {"iPad8,6", "j320xap", 0x18, 0x8027,
     "iPad Pro 12.9-inch (3rd gen, WiFi, 1TB)"},
    {"iPad8,7", "j321ap", 0x0A, 0x8027,
     "iPad Pro 12.9-inch (3rd gen, Cellular)"},
    {"iPad8,8", "j321xap", 0x1A, 0x8027,
     "iPad Pro 12.9-inch (3rd gen, Cellular, 1TB)"},
    {"iPad8,9", "j417ap", 0x3C, 0x8027, "iPad Pro 11-inch (2nd gen, WiFi)"},
    {"iPad8,10", "j418ap", 0x3E, 0x8027,
     "iPad Pro 11-inch (2nd gen, Cellular)"},
    {"iPad8,11", "j420ap", 0x38, 0x8027, "iPad Pro 12.9-inch (4th gen, WiFi)"},
    {"iPad8,12", "j421ap", 0x3A, 0x8027,
     "iPad Pro 12.9-inch (4th gen, Cellular)"},
    {"iPad11,1", "j210ap", 0x14, 0x8020, "iPad mini (5th gen, WiFi)"},
    {"iPad11,2", "j211ap", 0x16, 0x8020, "iPad mini (5th gen, Cellular)"},
    {"iPad11,3", "j217ap", 0x1C, 0x8020, "iPad Air (3rd gen, WiFi)"},
    {"iPad11,4", "j218ap", 0x1E, 0x8020, "iPad Air (3rd gen, Cellular)"},
    {"iPad11,6", "j171aap", 0x24, 0x8020, "iPad (8th gen, WiFi)"},
    {"iPad11,7", "j172aap", 0x26, 0x8020, "iPad (8th gen, Cellular)"},
    {"iPad12,1", "j181ap", 0x18, 0x8030, "iPad (9th gen, WiFi)"},
    {"iPad12,2", "j182ap", 0x1A, 0x8030, "iPad (9th gen, Cellular)"},
    {"iPad13,1", "j307ap", 0x04, 0x8101, "iPad Air (4th gen, WiFi)"},
    {"iPad13,2", "j308ap", 0x06, 0x8101, "iPad Air (4th gen, Cellular)"},
    {"iPad13,4", "j517ap", 0x08, 0x8103, "iPad Pro 11-inch (3rd gen, WiFi)"},
    {"iPad13,5", "j517xap", 0x0A, 0x8103,
     "iPad Pro 11-inch (3rd gen, WiFi, 2TB)"},
    {"iPad13,6", "j518ap", 0x0C, 0x8103,
     "iPad Pro 11-inch (3rd gen, Cellular)"},
    {"iPad13,7", "j518xap", 0x0E, 0x8103,
     "iPad Pro 11-inch (3rd gen, Cellular, 2TB)"},
    {"iPad13,8", "j522ap", 0x18, 0x8103, "iPad Pro 12.9-inch (5th gen, WiFi)"},
    {"iPad13,9", "j522xap", 0x1A, 0x8103,
     "iPad Pro 12.9-inch (5th gen, WiFi, 2TB)"},
    {"iPad13,10", "j523ap", 0x1C, 0x8103,
     "iPad Pro 12.9-inch (5th gen, Cellular)"},
    {"iPad13,11", "j523xap", 0x1E, 0x8103,
     "iPad Pro 12.9-inch (5th gen, Cellular, 2TB)"},
    {"iPad13,16", "j407ap", 0x10, 0x8103, "iPad Air (5th gen, WiFi)"},
    {"iPad13,17", "j408ap", 0x12, 0x8103, "iPad Air (5th gen, Cellular)"},
    {"iPad13,18", "j271ap", 0x14, 0x8101, "iPad (10th gen, WiFi)"},
    {"iPad13,19", "j272ap", 0x16, 0x8101, "iPad (10th gen, Cellular)"},
    {"iPad14,1", "j310ap", 0x04, 0x8110, "iPad mini (6th gen, WiFi)"},
    {"iPad14,2", "j311ap", 0x06, 0x8110, "iPad mini (6th gen, Cellular)"},
    {"iPad14,3", "j617ap", 0x08, 0x8112, "iPad Pro 11-inch (4th gen, WiFi)"},
    {"iPad14,4", "j618ap", 0x0A, 0x8112,
     "iPad Pro 11-inch (4th gen, Cellular)"},
    {"iPad14,5", "j620ap", 0x0C, 0x8112, "iPad Pro 12.9-inch (6th gen, WiFi)"},
    {"iPad14,6", "j621ap", 0x0E, 0x8112,
     "iPad Pro 12.9-inch (6th gen, Cellular)"},
    {"iPad14,8", "j507ap", 0x10, 0x8112, "iPad Air 11-inch (M2, WiFi)"},
    {"iPad14,9", "j508ap", 0x12, 0x8112, "iPad Air 11-inch (M2, Cellular)"},
    {"iPad14,10", "j537ap", 0x14, 0x8112, "iPad Air 13-inch (M2, WiFi)"},
    {"iPad14,11", "j538ap", 0x16, 0x8112, "iPad Air 13-inch (M2, Cellular)"},
    {"iPad15,3", "j607ap", 0x08, 0x8122, "iPad Air 11-inch (M3, WiFi)"},
    {"iPad15,4", "j608ap", 0x0A, 0x8122, "iPad Air 11-inch (M3, Cellular)"},
    {"iPad15,5", "j637ap", 0x0C, 0x8122, "iPad Air 13-inch (M3, WiFi)"},
    {"iPad15,6", "j638ap", 0x0E, 0x8122, "iPad Air 13-inch (M3, Cellular)"},
    {"iPad15,7", "j481ap", 0x10, 0x8120, "iPad (A16, WiFi)"},
    {"iPad15,8", "j482ap", 0x12, 0x8120, "iPad (A16, Cellular)"},
    {"iPad16,1", "j410ap", 0x08, 0x8130, "iPad mini (A17 Pro, WiFi)"},
    {"iPad16,2", "j411ap", 0x0A, 0x8130, "iPad mini (A17 Pro, Cellular)"},
    {"iPad16,3", "j717ap", 0x08, 0x8132, "iPad Pro 11-inch (M4, WiFi)"},
    {"iPad16,4", "j718ap", 0x0A, 0x8132, "iPad Pro 11-inch (M4, Cellular)"},
    {"iPad16,5", "j720ap", 0x0C, 0x8132, "iPad Pro 13-inch (M4, WiFi)"},
    {"iPad16,6", "j721ap", 0x0E, 0x8132, "iPad Pro 13-inch (M4, Cellular)"},
    /* Apple TV */
    {"AppleTV2,1", "k66ap", 0x10, 0x8930, "Apple TV 2"},
    {"AppleTV3,1", "j33ap", 0x08, 0x8942, "Apple TV 3"},
    {"AppleTV3,2", "j33iap", 0x00, 0x8947, "Apple TV 3 (2013)"},
    {"AppleTV5,3", "j42dap", 0x34, 0x7000, "Apple TV 4"},
    {"AppleTV6,2", "j105aap", 0x02, 0x8011, "Apple TV 4K"},
    {"AppleTV11,1", "j305ap", 0x08, 0x8020, "Apple TV 4K (2nd gen)"},
    {"AppleTV14,1", "j255ap", 0x02, 0x8110, "Apple TV 4K (3rd gen)"},
    /* HomePod */
    {"AudioAccessory1,1", "b238aap", 0x38, 0x7000, "HomePod (1st gen)"},
    {"AudioAccessory1,2", "b238ap", 0x1A, 0x7000, "HomePod (1st gen)"},
    {"AudioAccessory5,1", "b520ap", 0x22, 0x8006, "HomePod mini"},
    {"AudioAccessory6,1", "b620ap", 0x18, 0x8301, "HomePod (2nd gen)"},
    /* Apple Watch */
    {"Watch1,1", "n27aap", 0x02, 0x7002, "Apple Watch 38mm (1st gen)"},
    {"Watch1,2", "n28aap", 0x04, 0x7002, "Apple Watch 42mm (1st gen)"},
    {"Watch2,6", "n27dap", 0x02, 0x8002, "Apple Watch Series 1 (38mm)"},
    {"Watch2,7", "n28dap", 0x04, 0x8002, "Apple Watch Series 1 (42mm)"},
    {"Watch2,3", "n74ap", 0x0C, 0x8002, "Apple Watch Series 2 (38mm)"},
    {"Watch2,4", "n75ap", 0x0E, 0x8002, "Apple Watch Series 2 (42mm)"},
    {"Watch3,1", "n111sap", 0x1C, 0x8004,
     "Apple Watch Series 3 (38mm Cellular)"},
    {"Watch3,2", "n111bap", 0x1E, 0x8004,
     "Apple Watch Series 3 (42mm Cellular)"},
    {"Watch3,3", "n121sap", 0x18, 0x8004, "Apple Watch Series 3 (38mm)"},
    {"Watch3,4", "n121bap", 0x1A, 0x8004, "Apple Watch Series 3 (42mm)"},
    {"Watch4,1", "n131sap", 0x08, 0x8006, "Apple Watch Series 4 (40mm)"},
    {"Watch4,2", "n131bap", 0x0A, 0x8006, "Apple Watch Series 4 (44mm)"},
    {"Watch4,3", "n141sap", 0x0C, 0x8006,
     "Apple Watch Series 4 (40mm Cellular)"},
    {"Watch4,4", "n141bap", 0x0E, 0x8006,
     "Apple Watch Series 4 (44mm Cellular)"},
    {"Watch5,1", "n144sap", 0x10, 0x8006, "Apple Watch Series 5 (40mm)"},
    {"Watch5,2", "n144bap", 0x12, 0x8006, "Apple Watch Series 5 (44mm)"},
    {"Watch5,3", "n146sap", 0x14, 0x8006,
     "Apple Watch Series 5 (40mm Cellular)"},
    {"Watch5,4", "n146bap", 0x16, 0x8006,
     "Apple Watch Series 5 (44mm Cellular)"},
    {"Watch5,9", "n140sap", 0x28, 0x8006, "Apple Watch SE (40mm)"},
    {"Watch5,10", "n140bap", 0x2A, 0x8006, "Apple Watch SE (44mm)"},
    {"Watch5,11", "n142sap", 0x2C, 0x8006, "Apple Watch SE (40mm Cellular)"},
    {"Watch5,12", "n142bap", 0x2E, 0x8006, "Apple Watch SE (44mm Cellular)"},
    {"Watch6,1", "n157sap", 0x08, 0x8301, "Apple Watch Series 6 (40mm)"},
    {"Watch6,2", "n157bap", 0x0A, 0x8301, "Apple Watch Series 6 (44mm)"},
    {"Watch6,3", "n158sap", 0x0C, 0x8301,
     "Apple Watch Series 6 (40mm Cellular)"},
    {"Watch6,4", "n158bap", 0x0E, 0x8301,
     "Apple Watch Series 6 (44mm Cellular)"},
    {"Watch6,6", "n187sap", 0x10, 0x8301, "Apple Watch Series 7 (41mm)"},
    {"Watch6,7", "n187bap", 0x12, 0x8301, "Apple Watch Series 7 (45mm)"},
    {"Watch6,8", "n188sap", 0x14, 0x8301,
     "Apple Watch Series 7 (41mm Cellular)"},
    {"Watch6,9", "n188bap", 0x16, 0x8301,
     "Apple Watch Series 7 (45mm Cellular)"},
    {"Watch6,10", "n143sap", 0x28, 0x8301, "Apple Watch SE 2 (40mm)"},
    {"Watch6,11", "n143bap", 0x2A, 0x8301, "Apple Watch SE 2 (44mm)"},
    {"Watch6,12", "n149sap", 0x2C, 0x8301, "Apple Watch SE 2 (40mm Cellular)"},
    {"Watch6,13", "n149bap", 0x2E, 0x8301, "Apple Watch SE 2 (44mm Cellular)"},
    {"Watch6,14", "n197sap", 0x30, 0x8301, "Apple Watch Series 8 (41mm)"},
    {"Watch6,15", "n197bap", 0x32, 0x8301, "Apple Watch Series 8 (45mm)"},
    {"Watch6,16", "n198sap", 0x34, 0x8301,
     "Apple Watch Series 8 (41mm Cellular)"},
    {"Watch6,17", "n198bap", 0x36, 0x8301,
     "Apple Watch Series 8 (45mm Cellular)"},
    {"Watch6,18", "n199ap", 0x26, 0x8301, "Apple Watch Ultra"},
    {"Watch7,1", "n207sap", 0x08, 0x8310, "Apple Watch Series 9 (41mm)"},
    {"Watch7,2", "n207bap", 0x0A, 0x8310, "Apple Watch Series 9 (45mm)"},
    {"Watch7,3", "n208sap", 0x0C, 0x8310,
     "Apple Watch Series 9 (41mm Cellular)"},
    {"Watch7,4", "n208bap", 0x0E, 0x8310,
     "Apple Watch Series 9 (45mm Cellular)"},
    {"Watch7,5", "n210ap", 0x02, 0x8310, "Apple Watch Ultra 2"},
    {"Watch7,8", "n217sap", 0x10, 0x8310, "Apple Watch Series 10 (42mm)"},
    {"Watch7,9", "n217bap", 0x12, 0x8310, "Apple Watch Series 10 (46mm)"},
    {"Watch7,10", "n218sap", 0x14, 0x8310,
     "Apple Watch Series 10 (42mm Cellular)"},
    {"Watch7,11", "n218bap", 0x16, 0x8310,
     "Apple Watch Series 10 (46mm Cellular)"},
    {"Watch7,12", "n230ap", 0x22, 0x8310, "Apple Watch Ultra 3"},
    {"Watch7,13", "n243sap", 0x28, 0x8310, "Apple Watch SE 3 (40mm)"},
    {"Watch7,14", "n243bap", 0x2A, 0x8310, "Apple Watch SE 3 (44mm)"},
    {"Watch7,15", "n244sap", 0x2C, 0x8310, "Apple Watch SE 3 (40mm Cellular)"},
    {"Watch7,16", "n244bap", 0x2E, 0x8310, "Apple Watch SE 3 (44mm Cellular)"},
    {"Watch7,17", "n227sap", 0x18, 0x8310, "Apple Watch Series 11 (42mm)"},
    {"Watch7,18", "n227bap", 0x1A, 0x8310, "Apple Watch Series 11 (46mm)"},
    {"Watch7,19", "n228sap", 0x1C, 0x8310,
     "Apple Watch Series 11 (42mm Cellular)"},
    {"Watch7,20", "n228bap", 0x1E, 0x8310,
     "Apple Watch Series 11 (46mm Cellular)"},
    /* Apple Silicon Macs */
    {"ADP3,2", "j273aap", 0x42, 0x8027, "Developer Transition Kit (2020)"},
    {"Macmini9,1", "j274ap", 0x22, 0x8103, "Mac mini (M1, 2020)"},
    {"MacBookPro17,1", "j293ap", 0x24, 0x8103,
     "MacBook Pro (M1, 13-inch, 2020)"},
    {"MacBookPro18,1", "j316sap", 0x0A, 0x6000,
     "MacBook Pro (M1 Pro, 16-inch, 2021)"},
    {"MacBookPro18,2", "j316cap", 0x0A, 0x6001,
     "MacBook Pro (M1 Max, 16-inch, 2021)"},
    {"MacBookPro18,3", "j314sap", 0x08, 0x6000,
     "MacBook Pro (M1 Pro, 14-inch, 2021)"},
    {"MacBookPro18,4", "j314cap", 0x08, 0x6001,
     "MacBook Pro (M1 Max, 14-inch, 2021)"},
    {"MacBookAir10,1", "j313ap", 0x26, 0x8103, "MacBook Air (M1, 2020)"},
    {"iMac21,1", "j456ap", 0x28, 0x8103, "iMac 24-inch (M1, Two Ports, 2021)"},
    {"iMac21,2", "j457ap", 0x2A, 0x8103, "iMac 24-inch (M1, Four Ports, 2021)"},
    {"Mac13,1", "j375cap", 0x04, 0x6001, "Mac Studio (M1 Max, 2022)"},
    {"Mac13,2", "j375dap", 0x0C, 0x6002, "Mac Studio (M1 Ultra, 2022)"},
    {"Mac14,2", "j413ap", 0x28, 0x8112, "MacBook Air (M2, 2022)"},
    {"Mac14,7", "j493ap", 0x2A, 0x8112, "MacBook Pro (M2, 13-inch, 2022)"},
    {"Mac14,3", "j473ap", 0x24, 0x8112, "Mac mini (M2, 2023)"},
    {"Mac14,5", "j414cap", 0x04, 0x6021, "MacBook Pro (14-inch, M2 Max, 2023)"},
    {"Mac14,6", "j416cap", 0x06, 0x6021, "MacBook Pro (16-inch, M2 Max, 2023)"},
    {"Mac14,8", "j180dap", 0x08, 0x6022, "Mac Pro (2023)"},
    {"Mac14,9", "j414sap", 0x04, 0x6020, "MacBook Pro (14-inch, M2 Pro, 2023)"},
    {"Mac14,10", "j416sap", 0x06, 0x6020,
     "MacBook Pro (16-inch, M2 Pro, 2023)"},
    {"Mac14,12", "j474sap", 0x02, 0x6020, "Mac mini (M2 Pro, 2023)"},
    {"Mac14,13", "j475cap", 0x0A, 0x6021, "Mac Studio (M2 Max, 2023)"},
    {"Mac14,14", "j475dap", 0x0A, 0x6022, "Mac Studio (M2 Ultra, 2023)"},
    {"Mac14,15", "j415ap", 0x2E, 0x8112, "MacBook Air (M2, 15-inch, 2023)"},
    {"Mac15,3", "j504ap", 0x22, 0x8122, "MacBook Pro (14-inch, M3, Nov 2023)"},
    {"Mac15,4", "j433ap", 0x28, 0x8122, "iMac 24-inch (M3, Two Ports, 2023)"},
    {"Mac15,5", "j434ap", 0x2A, 0x8122, "iMac 24-inch (M3, Four Ports, 2023)"},
    {"Mac15,6", "j514sap", 0x04, 0x6030,
     "MacBook Pro (14-inch, M3 Pro, Nov 2023)"},
    {"Mac15,7", "j516sap", 0x06, 0x6030,
     "MacBook Pro (16-inch, M3 Pro, Nov 2023)"},
    {"Mac15,8", "j514cap", 0x44, 0x6031,
     "MacBook Pro (14-inch, M3 Max, Nov 2023)"},
    {"Mac15,9", "j516cap", 0x46, 0x6031,
     "MacBook Pro (16-inch, M3 Max, Nov 2023)"},
    {"Mac15,10", "j514map", 0x44, 0x6034,
     "MacBook Pro (14-inch, M3 Max, Nov 2023)"},
    {"Mac15,11", "j516map", 0x46, 0x6034,
     "MacBook Pro (16-inch, M3 Max, Nov 2023)"},
    {"Mac15,12", "j613ap", 0x30, 0x8122, "MacBook Air (13-inch, M3, 2024)"},
    {"Mac15,13", "j615ap", 0x32, 0x8122, "MacBook Air (15-inch, M3, 2024)"},
    {"Mac15,14", "j575dap", 0x44, 0x6032, "Mac Studio (M3 Ultra, 2025)"},
    {"Mac16,1", "j604ap", 0x22, 0x8132, "MacBook Pro (14-inch, M4, Nov 2024)"},
    {"Mac16,2", "j623ap", 0x24, 0x8132, "iMac 24-inch (M4, Two Ports, 2024)"},
    {"Mac16,3", "j624ap", 0x26, 0x8132, "iMac 24-inch (M4, Four Ports, 2024)"},
    {"Mac16,5", "j616cap", 0x06, 0x6041,
     "MacBook Pro (16-inch, M4 Max, Nov 2024)"},
    {"Mac16,6", "j614cap", 0x04, 0x6041,
     "MacBook Pro (14-inch, M4 Max, Nov 2024)"},
    {"Mac16,7", "j616sap", 0x06, 0x6040,
     "MacBook Pro (16-inch, M4 Pro, Nov 2024)"},
    {"Mac16,8", "j614sap", 0x04, 0x6040,
     "MacBook Pro (14-inch, M4 Pro, Nov 2024)"},
    {"Mac16,9", "j575cap", 0x02, 0x6041, "Mac Studio (M4 Max, 2025)"},
    {"Mac16,10", "j773gap", 0x2A, 0x8132, "Mac mini (M4, 2024)"},
    {"Mac16,11", "j773sap", 0x02, 0x6040, "Mac mini (M4 Pro, 2024)"},
    {"Mac16,12", "j713ap", 0x2C, 0x8132, "MacBook Air (13-inch, M4, 2025)"},
    {"Mac16,13", "j715ap", 0x2E, 0x8132, "MacBook Air (15-inch, M4, 2025)"},
    /* Apple Silicon VMs (supported by Virtualization.framework on macOS 12) */
    {"VirtualMac2,1", "vma2macosap", 0x20, 0xFE00, "Apple Virtual Machine 1"},
    /* Apple T2 Coprocessor */
    {"iBridge2,1", "j137ap", 0x0A, 0x8012, "Apple T2 iMacPro1,1 (j137)"},
    {"iBridge2,3", "j680ap", 0x0B, 0x8012, "Apple T2 MacBookPro15,1 (j680)"},
    {"iBridge2,4", "j132ap", 0x0C, 0x8012, "Apple T2 MacBookPro15,2 (j132)"},
    {"iBridge2,5", "j174ap", 0x0E, 0x8012, "Apple T2 Macmini8,1 (j174)"},
    {"iBridge2,6", "j160ap", 0x0F, 0x8012, "Apple T2 MacPro7,1 (j160)"},
    {"iBridge2,7", "j780ap", 0x07, 0x8012, "Apple T2 MacBookPro15,3 (j780)"},
    {"iBridge2,8", "j140kap", 0x17, 0x8012, "Apple T2 MacBookAir8,1 (j140k)"},
    {"iBridge2,10", "j213ap", 0x18, 0x8012, "Apple T2 MacBookPro15,4 (j213)"},
    {"iBridge2,12", "j140aap", 0x37, 0x8012, "Apple T2 MacBookAir8,2 (j140a)"},
    {"iBridge2,14", "j152fap", 0x3A, 0x8012, "Apple T2 MacBookPro16,1 (j152f)"},
    {"iBridge2,15", "j230kap", 0x3F, 0x8012, "Apple T2 MacBookAir9,1 (j230k)"},
    {"iBridge2,16", "j214kap", 0x3E, 0x8012, "Apple T2 MacBookPro16,2 (j214k)"},
    {"iBridge2,19", "j185ap", 0x22, 0x8012, "Apple T2 iMac20,1 (j185)"},
    {"iBridge2,20", "j185fap", 0x23, 0x8012, "Apple T2 iMac20,2 (j185f)"},
    {"iBridge2,21", "j223ap", 0x3B, 0x8012, "Apple T2 MacBookPro16,3 (j223)"},
    {"iBridge2,22", "j215ap", 0x38, 0x8012, "Apple T2 MacBookPro16,4 (j215)"},
    /* Apple Displays */
    {"AppleDisplay2,1", "j327ap", 0x22, 0x8030, "Studio Display"},
    /* Apple Vision Pro */
    {"RealityDevice14,1", "n301ap", 0x42, 0x8112, "Apple Vision Pro"},
    {NULL, NULL, -1, -1, NULL}};

const DeviceDatabaseInfo *
DeviceDatabase::findByIdentifier(const std::string &identifier)
{
    for (const auto &device : m_devices) {
        if (device.modelIdentifier == nullptr) {
            break;
        }
        if (identifier == device.modelIdentifier) {
            return &device;
        }
    }
    return nullptr;
}

const DeviceDatabaseInfo *
DeviceDatabase::findByHwModel(const std::string &hwModel)
{
    for (const auto &device : m_devices) {
        if (device.boardId == nullptr) {
            break;
        }
        if (hwModel == device.boardId) {
            return &device;
        }
    }
    return nullptr;
}

std::string DeviceDatabase::parseRegionInfo(const std::string &code)
{
    // North America
    if (code == "LL/A")
        return "United States, Canada";
    if (code == "LL")
        return "United States, Canada";

    // Latin America
    if (code == "LA/A")
        return "Latin America";
    if (code == "BR/A" || code == "BZ/A")
        return "Brazil";
    if (code == "CL/A")
        return "Chile";
    if (code == "CO/A")
        return "Colombia";
    if (code == "MX/A")
        return "Mexico";
    if (code == "AR/A")
        return "Argentina";

    // Asia Pacific
    if (code == "J/A")
        return "Japan";
    if (code == "KH/A")
        return "Thailand, Cambodia";
    if (code == "MY/A")
        return "Malaysia";
    if (code == "ZP/A")
        return "Hong Kong, Macau";
    if (code == "CH/A")
        return "China";
    if (code == "TA/A")
        return "Taiwan";
    if (code == "KR/A")
        return "Korea";
    if (code == "SG/A")
        return "Singapore";
    if (code == "IN/A")
        return "India";
    if (code == "TH/A")
        return "Thailand";
    if (code == "VN/A")
        return "Vietnam";
    if (code == "ID/A")
        return "Indonesia";
    if (code == "PH/A")
        return "Philippines";
    if (code == "NZ/A")
        return "New Zealand";
    if (code == "AU/A" || code == "X/A")
        return "Australia";

    // Europe
    if (code == "ZA/A")
        return "South Africa";
    if (code == "AB/A")
        return "Egypt, Jordan, Saudi Arabia, UAE";
    if (code == "AE/A")
        return "United Arab Emirates";
    if (code == "B/A")
        return "United Kingdom, Ireland";
    if (code == "FB/A")
        return "France, Luxembourg";
    if (code == "FD/A")
        return "Austria, Liechtenstein, Switzerland";
    if (code == "GR/A")
        return "Greece";
    if (code == "HN/A")
        return "India";
    if (code == "IP/A")
        return "Italy";
    if (code == "KN/A")
        return "Denmark, Norway";
    if (code == "KS/A")
        return "Finland, Sweden";
    if (code == "LZ/A")
        return "Paraguay, Uruguay";
    if (code == "MG/A")
        return "Hungary";
    if (code == "PO/A")
        return "Poland";
    if (code == "PP/A")
        return "Philippines";
    if (code == "RO/A")
        return "Romania";
    if (code == "RS/A")
        return "Russia";
    if (code == "SL/A")
        return "Slovakia";
    if (code == "SO/A")
        return "South Africa";
    if (code == "T/A")
        return "Italy";
    if (code == "TU/A")
        return "Turkey";
    if (code == "Y/A")
        return "Spain";
    if (code == "ZD/A")
        return "Germany, Luxembourg";

    // Middle East
    if (code == "HB/A")
        return "Israel";

    // Canada
    if (code == "C/A")
        return "Canada (English, French)";

    return "Unknown Region (" + code + ")";
}