"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const time_1 = require("../time");
const NoteResource_1 = require("../models/NoteResource");
const ResourceService_1 = require("../services/ResourceService");
const shim_1 = require("../shim");
const test_utils_1 = require("../testing/test-utils");
const Folder_1 = require("../models/Folder");
const Note_1 = require("../models/Note");
const Resource_1 = require("../models/Resource");
const SearchEngine_1 = require("./search/SearchEngine");
const utils_1 = require("./e2ee/utils");
describe('services/ResourceService', () => {
    beforeEach(async () => {
        await (0, test_utils_1.setupDatabaseAndSynchronizer)(1);
        await (0, test_utils_1.setupDatabaseAndSynchronizer)(2);
        await (0, test_utils_1.switchClient)(1);
    });
    it('should delete orphaned resources', (async () => {
        const service = new ResourceService_1.default();
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        let note1 = await Note_1.default.save({ title: 'ma note', parent_id: folder1.id });
        note1 = await shim_1.default.attachFileToNote(note1, `${test_utils_1.supportDir}/photo.jpg`);
        const resource1 = (await Resource_1.default.all())[0];
        const resourcePath = Resource_1.default.fullPath(resource1);
        await service.indexNoteResources();
        await service.deleteOrphanResources(0);
        expect(!!(await Resource_1.default.load(resource1.id))).toBe(true);
        await Note_1.default.delete(note1.id);
        await service.deleteOrphanResources(0);
        expect(!!(await Resource_1.default.load(resource1.id))).toBe(true);
        await service.indexNoteResources();
        await service.deleteOrphanResources(1000 * 60);
        expect(!!(await Resource_1.default.load(resource1.id))).toBe(true);
        await service.deleteOrphanResources(0);
        expect(!!(await Resource_1.default.load(resource1.id))).toBe(false);
        expect(await shim_1.default.fsDriver().exists(resourcePath)).toBe(false);
        expect(!(await NoteResource_1.default.all()).length).toBe(true);
    }));
    it('should not delete resource if still associated with at least one note', (async () => {
        const service = new ResourceService_1.default();
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        let note1 = await Note_1.default.save({ title: 'ma note', parent_id: folder1.id });
        const note2 = await Note_1.default.save({ title: 'ma deuxième note', parent_id: folder1.id });
        note1 = await shim_1.default.attachFileToNote(note1, `${test_utils_1.supportDir}/photo.jpg`);
        const resource1 = (await Resource_1.default.all())[0];
        await service.indexNoteResources();
        await Note_1.default.delete(note1.id);
        await service.indexNoteResources();
        await Note_1.default.save({ id: note2.id, body: Resource_1.default.markupTag(resource1) });
        await service.indexNoteResources();
        await service.deleteOrphanResources(0);
        expect(!!(await Resource_1.default.load(resource1.id))).toBe(true);
    }));
    // This is now handled below by more correct tests
    //
    // it('should not delete a resource that has never been associated with any note, because it probably means the resource came via sync, and associated note has not arrived yet', (async () => {
    // 	const service = new ResourceService();
    // 	await shim.createResourceFromPath(`${supportDir}/photo.jpg`);
    // 	await service.indexNoteResources();
    // 	await service.deleteOrphanResources(0);
    // 	expect((await Resource.all()).length).toBe(1);
    // }));
    it('should not delete resource if it is used in an IMG tag', (async () => {
        const service = new ResourceService_1.default();
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        let note1 = await Note_1.default.save({ title: 'ma note', parent_id: folder1.id });
        note1 = await shim_1.default.attachFileToNote(note1, `${test_utils_1.supportDir}/photo.jpg`);
        const resource1 = (await Resource_1.default.all())[0];
        await service.indexNoteResources();
        await Note_1.default.save({ id: note1.id, body: `This is HTML: <img src=":/${resource1.id}"/>` });
        await service.indexNoteResources();
        await service.deleteOrphanResources(0);
        expect(!!(await Resource_1.default.load(resource1.id))).toBe(true);
    }));
    it('should not process twice the same change', (async () => {
        const service = new ResourceService_1.default();
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        const note1 = await Note_1.default.save({ title: 'ma note', parent_id: folder1.id });
        await shim_1.default.attachFileToNote(note1, `${test_utils_1.supportDir}/photo.jpg`);
        await service.indexNoteResources();
        const before = (await NoteResource_1.default.all())[0];
        await time_1.default.sleep(0.1);
        await service.indexNoteResources();
        const after = (await NoteResource_1.default.all())[0];
        expect(before.last_seen_time).toBe(after.last_seen_time);
    }));
    it('should not delete resources that are associated with an encrypted note', (async () => {
        // https://github.com/laurent22/joplin/issues/1433
        //
        // Client 1 and client 2 have E2EE setup.
        //
        // - Client 1 creates note N1 and add resource R1 to it
        // - Client 1 syncs
        // - Client 2 syncs and get N1
        // - Client 2 add resource R2 to N1
        // - Client 2 syncs
        // - Client 1 syncs
        // - Client 1 runs resource indexer - but because N1 hasn't been decrypted yet, it found that R1 is no longer associated with any note
        // - Client 1 decrypts notes, but too late
        //
        // Eventually R1 is deleted because service thinks that it was at some point associated with a note, but no longer.
        const masterKey = await (0, test_utils_1.loadEncryptionMasterKey)();
        await (0, utils_1.setupAndEnableEncryption)((0, test_utils_1.encryptionService)(), masterKey, '123456');
        await (0, utils_1.loadMasterKeysFromSettings)((0, test_utils_1.encryptionService)());
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        const note1 = await Note_1.default.save({ title: 'ma note', parent_id: folder1.id });
        await shim_1.default.attachFileToNote(note1, `${test_utils_1.supportDir}/photo.jpg`); // R1
        await (0, test_utils_1.resourceService)().indexNoteResources();
        await (0, test_utils_1.synchronizer)().start();
        expect(await (0, test_utils_1.allSyncTargetItemsEncrypted)()).toBe(true);
        await (0, test_utils_1.switchClient)(2);
        await (0, test_utils_1.synchronizer)().start();
        await (0, utils_1.setupAndEnableEncryption)((0, test_utils_1.encryptionService)(), masterKey, '123456');
        await (0, utils_1.loadMasterKeysFromSettings)((0, test_utils_1.encryptionService)());
        await (0, test_utils_1.decryptionWorker)().start();
        {
            const n1 = await Note_1.default.load(note1.id);
            await shim_1.default.attachFileToNote(n1, `${test_utils_1.supportDir}/photo.jpg`); // R2
        }
        await (0, test_utils_1.synchronizer)().start();
        await (0, test_utils_1.switchClient)(1);
        await (0, test_utils_1.synchronizer)().start();
        await (0, test_utils_1.resourceService)().indexNoteResources();
        await (0, test_utils_1.resourceService)().deleteOrphanResources(0); // Previously, R1 would be deleted here because it's not indexed
        expect((await Resource_1.default.all()).length).toBe(2);
    }));
    it('should double-check if the resource is still linked before deleting it', (async () => {
        SearchEngine_1.default.instance().setDb((0, test_utils_1.db)()); // /!\ Note that we use the global search engine here, which we shouldn't but will work for now
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        let note1 = await Note_1.default.save({ title: 'ma note', parent_id: folder1.id });
        note1 = await shim_1.default.attachFileToNote(note1, `${test_utils_1.supportDir}/photo.jpg`);
        await (0, test_utils_1.resourceService)().indexNoteResources();
        const bodyWithResource = note1.body;
        await Note_1.default.save({ id: note1.id, body: '' });
        await (0, test_utils_1.resourceService)().indexNoteResources();
        await Note_1.default.save({ id: note1.id, body: bodyWithResource });
        await SearchEngine_1.default.instance().syncTables();
        await (0, test_utils_1.resourceService)().deleteOrphanResources(0);
        expect((await Resource_1.default.all()).length).toBe(1); // It should not have deleted the resource
        const nr = (await NoteResource_1.default.all())[0];
        expect(!!nr.is_associated).toBe(true); // And it should have fixed the situation by re-indexing the note content
    }));
    it('should delete a resource if it is not associated with any note and has never been synced', (async () => {
        // - User creates a note and attaches a resource
        // - User deletes the note
        // - NoteResource service runs - the resource can be deleted
        //
        // This is because, since the resource didn't come via sync, it's
        // guaranteed that it is orphaned, which means it can be safely deleted.
        // See related test below to handle case where the resource actually
        // comes via sync.
        const note = await Note_1.default.save({});
        await shim_1.default.attachFileToNote(note, `${test_utils_1.supportDir}/photo.jpg`);
        await Note_1.default.delete(note.id);
        const resource = (await Resource_1.default.all())[0];
        await (0, test_utils_1.resourceService)().indexNoteResources();
        await (0, test_utils_1.resourceService)().deleteOrphanResources(-10);
        expect(await Resource_1.default.load(resource.id)).toBeFalsy();
    }));
    it('should NOT delete a resource if it arrived via synced, even if it is not associated with any note', (async () => {
        // - C1 creates Resource 1
        // - C1 sync
        // - C2 sync
        // - NoteResource service runs - should find an orphan resource, but not
        //   delete it because the associated note might come later from U1.
        //
        // At this point, C1 has the knowledge about Resource 1 so whether it's
        // eventually deleted or not will come from there. If it's an orphaned
        // resource, it will be deleted on C1 first, then the deletion will be
        // synced to other clients.
        const note = await Note_1.default.save({});
        await shim_1.default.attachFileToNote(note, `${test_utils_1.supportDir}/photo.jpg`);
        await Note_1.default.delete(note.id);
        const resource = (await Resource_1.default.all())[0];
        await (0, test_utils_1.synchronizer)().start();
        await (0, test_utils_1.switchClient)(2);
        await (0, test_utils_1.synchronizer)().start();
        await (0, test_utils_1.resourceService)().indexNoteResources();
        await (0, test_utils_1.resourceService)().deleteOrphanResources(0);
        expect(await Resource_1.default.load(resource.id)).toBeTruthy();
    }));
    it('should still create associations for notes in the trash', async () => {
        const note = await Note_1.default.save({});
        await shim_1.default.attachFileToNote(note, `${test_utils_1.supportDir}/photo.jpg`);
        await Note_1.default.delete(note.id, { toTrash: true });
        await (0, test_utils_1.resourceService)().indexNoteResources();
        // Check that the association is made despite the note being deleted
        const noteResources = await NoteResource_1.default.all();
        expect(noteResources.length).toBe(1);
        expect(noteResources[0].note_id).toBe(note.id);
        // Also check that the resources are not deleted as orphan
        await (0, test_utils_1.resourceService)().deleteOrphanResources(0);
        expect((await NoteResource_1.default.all()).length).toBe(1);
    });
});
//# sourceMappingURL=ResourceService.test.js.map