"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const types_1 = require("./types");
const InteropService_Importer_Base_1 = require("./InteropService_Importer_Base");
const path_utils_1 = require("../../path-utils");
const InteropService_Importer_Md_1 = require("./InteropService_Importer_Md");
const path_1 = require("path");
const Logger_1 = require("@joplin/utils/Logger");
const uuid_1 = require("../../uuid");
const shim_1 = require("../../shim");
const logger = Logger_1.default.create('InteropService_Importer_OneNote');
// See onenote-converter README.md for more information
class InteropService_Importer_OneNote extends InteropService_Importer_Base_1.default {
    constructor() {
        super(...arguments);
        this.importedNotes = {};
        this.document = null;
        this.xmlSerializer = null;
    }
    async init(sourcePath, options) {
        await super.init(sourcePath, options);
        if (!options.document || !options.xmlSerializer) {
            throw new Error('OneNote importer requires document and XMLSerializer to be able to extract SVG from HTML.');
        }
        this.document = options.document;
        this.xmlSerializer = options.xmlSerializer;
    }
    getEntryDirectory(unzippedPath, entryName) {
        const withoutBasePath = entryName.replace(unzippedPath, '');
        return (0, path_1.normalize)(withoutBasePath).split(path_1.sep)[0];
    }
    async exec(result) {
        const sourcePath = (0, path_utils_1.rtrimSlashes)(this.sourcePath_);
        const unzipTempDirectory = await this.temporaryDirectory_(true);
        logger.info('Unzipping files...');
        const files = await shim_1.default.fsDriver().zipExtract({ source: sourcePath, extractTo: unzipTempDirectory });
        if (files.length === 0) {
            result.warnings.push('Zip file has no files.');
            return result;
        }
        const tempOutputDirectory = await this.temporaryDirectory_(true);
        const baseFolder = this.getEntryDirectory(unzipTempDirectory, files[0].entryName);
        const notebookBaseDir = (0, path_1.join)(unzipTempDirectory, baseFolder, path_1.sep);
        const outputDirectory2 = (0, path_1.join)(tempOutputDirectory, baseFolder);
        const notebookFiles = files.filter(e => e.name !== '.onetoc2' && e.name !== 'OneNote_RecycleBin.onetoc2');
        const { oneNoteConverter } = shim_1.default.requireDynamic('@joplin/onenote-converter');
        logger.info('Extracting OneNote to HTML');
        for (const notebookFile of notebookFiles) {
            const notebookFilePath = (0, path_1.join)(unzipTempDirectory, notebookFile.entryName);
            try {
                await oneNoteConverter(notebookFilePath, (0, path_1.resolve)(outputDirectory2), notebookBaseDir);
            }
            catch (error) {
                console.error(error);
            }
        }
        logger.info('Extracting SVGs into files');
        await this.moveSvgToLocalFile(tempOutputDirectory);
        logger.info('Importing HTML into Joplin');
        const importer = new InteropService_Importer_Md_1.default();
        importer.setMetadata({ fileExtensions: ['html'] });
        await importer.init(tempOutputDirectory, Object.assign(Object.assign({}, this.options_), { format: 'html', outputFormat: types_1.ImportModuleOutputFormat.Html }));
        logger.info('Finished');
        result = await importer.exec(result);
        return result;
    }
    async moveSvgToLocalFile(baseFolder) {
        const htmlFiles = await this.getValidHtmlFiles((0, path_1.resolve)(baseFolder));
        for (const file of htmlFiles) {
            const fileLocation = (0, path_1.join)(baseFolder, file.path);
            const originalHtml = await shim_1.default.fsDriver().readFile(fileLocation);
            const { svgs, html: updatedHtml } = this.extractSvgs(originalHtml, () => (0, uuid_1.uuidgen)(10));
            if (!svgs || !svgs.length)
                continue;
            await shim_1.default.fsDriver().writeFile(fileLocation, updatedHtml, 'utf8');
            await this.createSvgFiles(svgs, (0, path_1.join)(baseFolder, (0, path_1.dirname)(file.path)));
        }
    }
    async getValidHtmlFiles(baseFolder) {
        const files = await shim_1.default.fsDriver().readDirStats(baseFolder, { recursive: true });
        const htmlFiles = files.filter(f => !f.isDirectory() && f.path.endsWith('.html'));
        return htmlFiles;
    }
    async createSvgFiles(svgs, svgBaseFolder) {
        for (const svg of svgs) {
            await shim_1.default.fsDriver().writeFile((0, path_1.join)(svgBaseFolder, svg.title), svg.content, 'utf8');
        }
    }
    extractSvgs(html, titleGenerator) {
        const htmlDocument = this.document.implementation.createHTMLDocument('htmlDocument');
        const root = htmlDocument.createElement('html');
        const body = htmlDocument.createElement('body');
        root.appendChild(body);
        root.innerHTML = html;
        // get all "top-level" SVGS (ignore nested)
        const svgNodeList = root.querySelectorAll('svg');
        if (!svgNodeList || !svgNodeList.length) {
            return { svgs: [], html };
        }
        const svgs = [];
        for (const svgNode of svgNodeList) {
            const title = `${titleGenerator()}.svg`;
            const img = htmlDocument.createElement('img');
            img.setAttribute('style', svgNode.getAttribute('style'));
            img.setAttribute('src', `./${title}`);
            svgNode.removeAttribute('style');
            svgs.push({
                title,
                content: this.xmlSerializer.serializeToString(svgNode),
            });
            svgNode.parentElement.replaceChild(img, svgNode);
        }
        return {
            svgs,
            html: this.xmlSerializer.serializeToString(root),
        };
    }
}
exports.default = InteropService_Importer_OneNote;
//# sourceMappingURL=InteropService_Importer_OneNote.js.map