"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const Setting_1 = require("../../../models/Setting");
const test_utils_1 = require("../../../testing/test-utils");
const types_1 = require("../../database/types");
const OcrDriverTranscribe_1 = require("./OcrDriverTranscribe");
const registry_1 = require("../../../registry");
describe('OcrDriverTranscribe', () => {
    let mockApi;
    beforeEach(async () => {
        await (0, test_utils_1.setupDatabaseAndSynchronizer)(1);
        await (0, test_utils_1.switchClient)(1);
        mockApi = {
            exec: jest.fn(),
        };
        const mockApiMethod = jest.fn().mockResolvedValue(mockApi);
        const mockDriver = { api: mockApiMethod };
        const mockFileApi = { driver: jest.fn().mockReturnValue(mockDriver) };
        const mockSyncTarget = { fileApi: jest.fn().mockResolvedValue(mockFileApi) };
        registry_1.reg.syncTarget = jest.fn().mockReturnValue(mockSyncTarget);
    });
    it('should return an error if synchronization target is not set', async () => {
        const { resource } = await (0, test_utils_1.createNoteAndResource)();
        const htr = new OcrDriverTranscribe_1.default();
        const response = await htr.recognize('', 'mock-path', resource.id);
        expect(response.ocr_status).toBe(types_1.ResourceOcrStatus.Error);
    });
    it('should return correct response when successful', async () => {
        const { resource } = await (0, test_utils_1.createNoteAndResource)();
        mockApi.exec.mockResolvedValue(Promise.resolve({ jobId: 'not-a-real-job-id' }));
        mockApi.exec.mockResolvedValue(Promise.resolve({ state: 'pending', jobId: 'not-a-real-job-id' }));
        mockApi.exec.mockResolvedValue(Promise.resolve({ state: 'completed', jobId: 'not-a-real-job-id', output: { result: 'this is the final transcription' } }));
        const htr = new OcrDriverTranscribe_1.default([1]);
        Setting_1.default.setValue('sync.target', 9);
        const response = await htr.recognize('', resource.filename, resource.id);
        expect(response.ocr_status).toBe(types_1.ResourceOcrStatus.Done);
        expect(response.ocr_text).toBe('this is the final transcription');
    });
    it('should return error when unsuccessful', async () => {
        const { resource } = await (0, test_utils_1.createNoteAndResource)();
        mockApi.exec.mockResolvedValue(Promise.resolve({ jobId: 'not-a-real-job-id' }));
        mockApi.exec.mockResolvedValue(Promise.resolve({ state: 'failed', jobId: 'not-a-real-job-id', output: { stack: '', message: 'Something went wrong' } }));
        const htr = new OcrDriverTranscribe_1.default([1]);
        Setting_1.default.setValue('sync.target', 9);
        const response = await htr.recognize('', resource.filename, resource.id);
        expect(response.ocr_status).toBe(types_1.ResourceOcrStatus.Error);
        expect(response.ocr_error).toEqual({ stack: '', message: 'Something went wrong' });
    });
    it('should be able to retrieve jobId from database instead of creating a new job', async () => {
        const { resource } = await (0, test_utils_1.createNoteAndResource)();
        const jobId = 'jobIdThat should be reused latter';
        mockApi.exec.mockResolvedValue(Promise.resolve({ jobId }));
        mockApi.exec.mockImplementationOnce(() => { throw new Error('Network request failed'); });
        const htr = new OcrDriverTranscribe_1.default([1]);
        Setting_1.default.setValue('sync.target', 9);
        const response = await htr.recognize('', resource.filename, resource.id);
        await htr.dispose();
        expect(response.ocr_status).toBe(types_1.ResourceOcrStatus.Todo);
        expect(response.ocr_error).toBe('');
        // Simulating closing/opening application
        mockApi.exec.mockResolvedValue({ jobId, state: 'completed', output: { result: 'result' } });
        const htr2 = new OcrDriverTranscribe_1.default([1]);
        const response2 = await htr2.recognize('', resource.filename, resource.id);
        expect(response2.ocr_status).toBe(types_1.ResourceOcrStatus.Done);
        expect(response2.ocr_text).toBe('result');
    });
});
//# sourceMappingURL=OcrDriverTranscribe.test.js.map