"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const Setting_1 = require("../../models/Setting");
const PluginService_1 = require("../../services/plugins/PluginService");
const test_utils_1 = require("../../testing/test-utils");
const loadPlugins_1 = require("./loadPlugins");
const MockPluginRunner_1 = require("./testing/MockPluginRunner");
const reducer_1 = require("../../reducer");
const redux_1 = require("redux");
const MockPlatformImplementation_1 = require("./testing/MockPlatformImplementation");
const createTestPlugin_1 = require("../../testing/plugins/createTestPlugin");
const createMockReduxStore = () => {
    return (0, redux_1.createStore)((state = reducer_1.defaultState, action) => {
        return (0, reducer_1.default)(state, action);
    });
};
const defaultManifestProperties = {
    manifest_version: 1,
    version: '0.1.0',
    app_min_version: '2.3.4',
    platforms: ['desktop', 'mobile'],
};
const platformImplementation = new MockPlatformImplementation_1.default();
describe('loadPlugins', () => {
    beforeEach(async () => {
        await (0, test_utils_1.setupDatabaseAndSynchronizer)(1);
        await (0, test_utils_1.switchClient)(1);
    });
    afterEach(async () => {
        for (const id of PluginService_1.default.instance().pluginIds) {
            await PluginService_1.default.instance().unloadPlugin(id);
        }
        await PluginService_1.default.instance().destroy();
    });
    test('should load only enabled plugins', async () => {
        await (0, createTestPlugin_1.default)(Object.assign(Object.assign({}, defaultManifestProperties), { id: 'this.is.a.test.1', name: 'Disabled Plugin' }), { enabled: false });
        const enabledPluginId = 'this.is.a.test.2';
        await (0, createTestPlugin_1.default)(Object.assign(Object.assign({}, defaultManifestProperties), { id: enabledPluginId, name: 'Enabled Plugin' }));
        const pluginRunner = new MockPluginRunner_1.default();
        const store = createMockReduxStore();
        const loadPluginsOptions = {
            pluginRunner,
            pluginSettings: Setting_1.default.value('plugins.states'),
            platformImplementation,
            store,
            reloadAll: false,
            cancelEvent: { cancelled: false },
        };
        expect(Object.keys(PluginService_1.default.instance().plugins)).toHaveLength(0);
        await (0, loadPlugins_1.default)(loadPluginsOptions);
        await pluginRunner.waitForAllToBeRunning([enabledPluginId]);
        expect(pluginRunner.runningPluginIds).toMatchObject([enabledPluginId]);
        // No plugins were running before, so none should be stopped.
        expect(pluginRunner.stopCalledTimes).toBe(0);
        // Loading again should not re-run plugins
        await (0, loadPlugins_1.default)(loadPluginsOptions);
        // Should have tried to stop at most the disabled plugin (which is a no-op).
        expect(pluginRunner.stopCalledTimes).toBe(1);
        expect(pluginRunner.runningPluginIds).toMatchObject([enabledPluginId]);
    });
    test('should reload all plugins when reloadAll is true', async () => {
        const enabledCount = 3;
        for (let i = 0; i < enabledCount; i++) {
            await (0, createTestPlugin_1.default)(Object.assign(Object.assign({}, defaultManifestProperties), { id: `joplin.test.plugin.${i}`, name: `Enabled Plugin ${i}` }));
        }
        const disabledCount = 6;
        const disabledPlugins = [];
        for (let i = 0; i < disabledCount; i++) {
            disabledPlugins.push(await (0, createTestPlugin_1.default)(Object.assign(Object.assign({}, defaultManifestProperties), { id: `joplin.test.plugin.disabled.${i}`, name: `Disabled Plugin ${i}` }), { enabled: false }));
        }
        const pluginRunner = new MockPluginRunner_1.default();
        const store = createMockReduxStore();
        const loadPluginsOptions = {
            pluginRunner,
            pluginSettings: Setting_1.default.value('plugins.states'),
            platformImplementation,
            store,
            reloadAll: true,
            cancelEvent: { cancelled: false },
        };
        await (0, loadPlugins_1.default)(loadPluginsOptions);
        let expectedRunningIds = ['joplin.test.plugin.0', 'joplin.test.plugin.1', 'joplin.test.plugin.2'];
        await pluginRunner.waitForAllToBeRunning(expectedRunningIds);
        // No additional plugins should be running.
        expect([...pluginRunner.runningPluginIds].sort()).toMatchObject(expectedRunningIds);
        // No plugins were running before -- there were no plugins to stop
        expect(pluginRunner.stopCalledTimes).toBe(0);
        const testPlugin = disabledPlugins[2];
        expect(testPlugin.manifest.id).toBe('joplin.test.plugin.disabled.2');
        // Enabling a plugin and reloading it should cause all plugins to load.
        testPlugin.setEnabled(true);
        await (0, loadPlugins_1.default)(Object.assign(Object.assign({}, loadPluginsOptions), { pluginSettings: Setting_1.default.value('plugins.states') }));
        expectedRunningIds = ['joplin.test.plugin.0', 'joplin.test.plugin.1', 'joplin.test.plugin.2', 'joplin.test.plugin.disabled.2'];
        await pluginRunner.waitForAllToBeRunning(expectedRunningIds);
        // Reloading all should stop all plugins and rerun enabled plugins, even
        // if not enabled previously.
        expect(pluginRunner.stopCalledTimes).toBe(disabledCount + enabledCount);
        expect([...pluginRunner.runningPluginIds].sort()).toMatchObject(expectedRunningIds);
    });
});
//# sourceMappingURL=loadPlugins.test.js.map