"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const BaseModel_1 = require("../../../BaseModel");
const ItemChange_1 = require("../../../models/ItemChange");
const Note_1 = require("../../../models/Note");
const test_utils_1 = require("../../../testing/test-utils");
const Api_1 = require("../Api");
let api = null;
describe('routes/events', () => {
    beforeEach(async () => {
        api = new Api_1.default();
        await (0, test_utils_1.setupDatabaseAndSynchronizer)(1);
        await (0, test_utils_1.switchClient)(1);
    });
    it('should retrieve the latest events', async () => {
        let cursor = '0';
        {
            const response = await api.route(Api_1.RequestMethod.GET, 'events', { cursor });
            expect(response.cursor).toBe('0');
        }
        const note1 = await Note_1.default.save({ title: 'toto' });
        await Note_1.default.save({ id: note1.id, title: 'tutu' });
        const note2 = await Note_1.default.save({ title: 'tata' });
        await ItemChange_1.default.waitForAllSaved();
        {
            const response = await api.route(Api_1.RequestMethod.GET, 'events', { cursor });
            expect(response.cursor).toBe('3');
            expect(response.items.length).toBe(2);
            expect(response.has_more).toBe(false);
            expect(response.items.map((it) => it.item_id).sort()).toEqual([note1.id, note2.id].sort());
            cursor = response.cursor;
        }
        {
            const response = await api.route(Api_1.RequestMethod.GET, 'events', { cursor });
            expect(response.cursor).toBe(cursor);
            expect(response.items.length).toBe(0);
            expect(response.has_more).toBe(false);
        }
        await Note_1.default.save({ id: note2.id, title: 'titi' });
        await ItemChange_1.default.waitForAllSaved();
        {
            const response = await api.route(Api_1.RequestMethod.GET, 'events', { cursor });
            expect(response.cursor).toBe('4');
            expect(response.items.length).toBe(1);
            expect(response.items[0].item_id).toBe(note2.id);
        }
    });
    it('should limit the number of response items', async () => {
        const promises = [];
        for (let i = 0; i < 101; i++) {
            promises.push(Note_1.default.save({ title: 'toto' }));
        }
        await Promise.all(promises);
        await ItemChange_1.default.waitForAllSaved();
        const response1 = await api.route(Api_1.RequestMethod.GET, 'events', { cursor: '0' });
        expect(response1.items.length).toBe(100);
        expect(response1.has_more).toBe(true);
        const response2 = await api.route(Api_1.RequestMethod.GET, 'events', { cursor: response1.cursor });
        expect(response2.items.length).toBe(1);
        expect(response2.has_more).toBe(false);
    });
    it('should retrieve a single item', async () => {
        const beforeTime = Date.now();
        const note = await Note_1.default.save({ title: 'toto' });
        await ItemChange_1.default.waitForAllSaved();
        const response = await api.route(Api_1.RequestMethod.GET, 'events/1');
        expect(response.item_type).toBe(BaseModel_1.ModelType.Note);
        expect(response.type).toBe(1);
        expect(response.item_id).toBe(note.id);
        expect(response.created_time).toBeGreaterThanOrEqual(beforeTime);
        await (0, test_utils_1.expectThrow)(async () => api.route(Api_1.RequestMethod.GET, 'events/1234'));
    });
});
//# sourceMappingURL=events.test.js.map