"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const Note_1 = require("../../../models/Note");
const Api_1 = require("../Api");
const test_utils_1 = require("../../../testing/test-utils");
const Folder_1 = require("../../../models/Folder");
describe('routes/folders', () => {
    beforeEach(async () => {
        await (0, test_utils_1.setupDatabase)(1);
        await (0, test_utils_1.switchClient)(1);
    });
    test('should not include deleted folders in GET call', async () => {
        const api = new Api_1.default();
        const folder1 = await Folder_1.default.save({});
        const folder2 = await Folder_1.default.save({});
        await api.route(Api_1.RequestMethod.DELETE, `folders/${folder1.id}`);
        const tree = await api.route(Api_1.RequestMethod.GET, 'folders', { as_tree: 1 });
        expect(tree.length).toBe(1);
        expect(tree[0].id).toBe(folder2.id);
        const page = await api.route(Api_1.RequestMethod.GET, 'folders');
        expect(page.items.length).toBe(1);
        expect(page.items[0].id).toBe(folder2.id);
    });
    test('should not include deleted folders in GET folders/:id/notes call', async () => {
        const api = new Api_1.default();
        const folder = await Folder_1.default.save({});
        const note1 = await Note_1.default.save({ parent_id: folder.id });
        const note2 = await Note_1.default.save({ parent_id: folder.id });
        {
            const notes = await api.route(Api_1.RequestMethod.GET, `folders/${folder.id}/notes`);
            expect(notes.items.length).toBe(2);
        }
        await Note_1.default.delete(note1.id, { toTrash: true });
        {
            const notes = await api.route(Api_1.RequestMethod.GET, `folders/${folder.id}/notes`);
            expect(notes.items.length).toBe(1);
            expect(notes.items[0].id).toBe(note2.id);
        }
        // const tree = await api.route(RequestMethod.GET, 'folders', { as_tree: 1 });
        // expect(tree.length).toBe(1);
        // expect(tree[0].id).toBe(folder2.id);
        // const page = await api.route(RequestMethod.GET, 'folders');
        // expect(page.items.length).toBe(1);
        // expect(page.items[0].id).toBe(folder2.id);
    });
    test('should be able to delete to trash', async () => {
        const api = new Api_1.default();
        const folder1 = await Folder_1.default.save({});
        const folder2 = await Folder_1.default.save({});
        const note1 = await Note_1.default.save({ parent_id: folder1.id });
        const note2 = await Note_1.default.save({ parent_id: folder2.id });
        const beforeTime = Date.now();
        await api.route(Api_1.RequestMethod.DELETE, `folders/${folder1.id}`);
        await api.route(Api_1.RequestMethod.DELETE, `folders/${folder2.id}`, { permanent: '1' });
        expect((await Folder_1.default.load(folder1.id)).deleted_time).toBeGreaterThanOrEqual(beforeTime);
        expect(await Folder_1.default.load(folder2.id)).toBeFalsy();
        expect((await Note_1.default.load(note1.id)).deleted_time).toBeGreaterThanOrEqual(beforeTime);
        expect(await Note_1.default.load(note2.id)).toBeFalsy();
    });
});
//# sourceMappingURL=folders.test.js.map