"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const Logger_1 = require("@joplin/utils/Logger");
const shim_1 = require("../../../shim");
const notes_1 = require("./notes");
const Setting_1 = require("../../../models/Setting");
const fs_extra_1 = require("fs-extra");
const Resource_1 = require("../../../models/Resource");
const Api_1 = require("../Api");
const Note_1 = require("../../../models/Note");
const test_utils_1 = require("../../../testing/test-utils");
const md5 = require('md5');
const imagePath = `${__dirname}/../../../images/SideMenuHeader.png`;
const jpgBase64Content = '/9j/4AAQSkZJRgABAQAAAQABAAD/2wBDAAMCAgICAgMCAgIDAwMDBAYEBAQEBAgGBgUGCQgKCgkICQkKDA8MCgsOCwkJDRENDg8QEBEQCgwSExIQEw8QEBD/wAALCAAFAAUBAREA/8QAFAABAAAAAAAAAAAAAAAAAAAACf/EAB8QAAEEAQUBAAAAAAAAAAAAAAQBAgUGAwAREiExM//aAAgBAQAAPwBJarVpGHm7KWbapCSwyZ6FDjkLyYE1W/LHyV2zfOk2TrzX/9k=';
describe('routes/notes', () => {
    beforeEach(async () => {
        jest.resetAllMocks();
        await (0, test_utils_1.setupDatabase)(1);
        await (0, test_utils_1.switchClient)(1);
    });
    test.each([
        '/invalid/url',
        'htp/asdfasf.com',
        'https//joplinapp.org',
    ])('should not return a local file for invalid protocols', async (invalidUrl) => {
        expect(await (0, notes_1.downloadMediaFile)(invalidUrl)).toBe('');
    });
    test.each([
        'https://joplinapp.org/valid/image_url.png',
        'http://joplinapp.org/valid/image_url.png',
    ])('should try to download and return a local path to a valid URL', async (url) => {
        const fetchBlobSpy = jest.fn(async (_url, options) => {
            await (0, fs_extra_1.writeFile)(options.path, Buffer.from(jpgBase64Content, 'base64'));
        });
        const spy = jest.spyOn(shim_1.default, 'fetchBlob').mockImplementation(fetchBlobSpy);
        const response = await (0, notes_1.downloadMediaFile)(url);
        const files = await (0, fs_extra_1.readdir)(Setting_1.default.value('tempDir'));
        expect(files.length).toBe(1);
        expect(fetchBlobSpy).toHaveBeenCalledTimes(1);
        expect(response).toBe(`${Setting_1.default.value('tempDir')}/${files[0]}`);
        await (0, fs_extra_1.remove)(response);
        spy.mockRestore();
    });
    test('should get file from local drive if protocol allows it', async () => {
        const url = `file:///${imagePath}`;
        const originalFileContent = await (0, fs_extra_1.readFile)(imagePath);
        const response = await (0, notes_1.downloadMediaFile)(url, null, ['file:']);
        const files = await (0, fs_extra_1.readdir)(Setting_1.default.value('tempDir'));
        expect(files.length).toBe(1);
        expect(response).toBe(`${Setting_1.default.value('tempDir')}/${files[0]}`);
        const responseFileContent = await (0, fs_extra_1.readFile)(response);
        expect(md5(responseFileContent)).toBe(md5(originalFileContent));
        await (0, fs_extra_1.remove)(response);
    });
    test('should be able to handle URLs with data', async () => {
        const url = 'data:image/gif;base64,R0lGODlhEAAQAMQAAORHHOVSKudfOulrSOp3WOyDZu6QdvCchPGolfO0o/XBs/fNwfjZ0frl3/zy7////wAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAACH5BAkAABAALAAAAAAQABAAAAVVICSOZGlCQAosJ6mu7fiyZeKqNKToQGDsM8hBADgUXoGAiqhSvp5QAnQKGIgUhwFUYLCVDFCrKUE1lBavAViFIDlTImbKC5Gm2hB0SlBCBMQiB0UjIQA7';
        const originalFileContent = Buffer.from(url.split('data:image/gif;base64,')[1], 'base64');
        const response = await (0, notes_1.downloadMediaFile)(url);
        const files = await (0, fs_extra_1.readdir)(Setting_1.default.value('tempDir'));
        expect(files.length).toBe(1);
        expect(response).toBe(`${Setting_1.default.value('tempDir')}/${files[0]}`);
        const responseFileContent = await (0, fs_extra_1.readFile)(response);
        expect(md5(responseFileContent)).toBe(md5(originalFileContent));
        await (0, fs_extra_1.remove)(response);
    });
    test('should not process URLs with data that is not image type', async () => {
        const url = 'data:application/octet-stream;base64,dGhpcyBpcyBhIG1lc3NhZ2UK';
        Logger_1.default.globalLogger.enabled = false;
        const response = await (0, notes_1.downloadMediaFile)(url);
        Logger_1.default.globalLogger.enabled = true;
        const files = await (0, fs_extra_1.readdir)(Setting_1.default.value('tempDir'));
        expect(files.length).toBe(0);
        expect(response).toBe('');
    });
    test('should not process URLs from cid: protocol', async () => {
        const url = 'cid:ii_loq3d1100';
        const response = await (0, notes_1.downloadMediaFile)(url);
        const files = await (0, fs_extra_1.readdir)(Setting_1.default.value('tempDir'));
        expect(files.length).toBe(0);
        expect(response).toBe('');
    });
    test('should not copy content from invalid protocols', async () => {
        const url = 'file:///home/user/file.db';
        const allowedProtocols = [];
        const mediaFilePath = await (0, notes_1.downloadMediaFile)(url, null, allowedProtocols);
        const files = await (0, fs_extra_1.readdir)(Setting_1.default.value('tempDir'));
        expect(files.length).toBe(0);
        expect(mediaFilePath).toBe('');
    });
    test.each([
        'https://joplinapp.org/valid/image_url',
        'https://joplinapp.org/valid/image_url.invalid_url',
    ])('should correct the file extension in filename from files without or invalid ones', async (url) => {
        const spy = jest.spyOn(shim_1.default, 'fetchBlob').mockImplementation(async (_url, options) => {
            await (0, fs_extra_1.writeFile)(options.path, Buffer.from(jpgBase64Content, 'base64'));
            return {
                headers: {
                    'content-type': 'image/jpg',
                },
            };
        });
        const response = await (0, notes_1.downloadMediaFile)(url);
        const files = await (0, fs_extra_1.readdir)(Setting_1.default.value('tempDir'));
        expect(files.length).toBe(1);
        expect(response).toBe(`${Setting_1.default.value('tempDir')}/${files[0]}`);
        await (0, fs_extra_1.remove)(response);
        spy.mockRestore();
    });
    test('should be able to create resource from files in the filesystem', async () => {
        const result = await (0, notes_1.createResourcesFromPaths)([
            { originalUrl: 'asdf.png', path: `${__dirname}/../../../images/SideMenuHeader.png` },
        ]);
        const resources = await Resource_1.default.all();
        expect(result.length).toBe(1);
        expect(result[0].originalUrl).toBe('asdf.png');
        expect(result[0].path).toBe(`${__dirname}/../../../images/SideMenuHeader.png`);
        expect(result[0].resource.title).toBe('SideMenuHeader.png');
        expect(result[0].resource.file_extension).toBe('png');
        expect(resources.length).toBe(1);
        expect(result[0].resource).toEqual(resources[0]);
    });
    test('should not create resource from files that does not exist', async () => {
        Logger_1.default.globalLogger.enabled = false;
        const result = await (0, notes_1.createResourcesFromPaths)([
            { originalUrl: 'not-a-real-file', path: '/does/not/exist' },
        ]);
        Logger_1.default.globalLogger.enabled = true;
        expect(result[0].resource).toBe(null);
        const resources = await Resource_1.default.all();
        expect(resources.length).toBe(0);
    });
    test('should be able to delete to trash', async () => {
        const api = new Api_1.default();
        const note1 = await Note_1.default.save({});
        const note2 = await Note_1.default.save({});
        const beforeTime = Date.now();
        await api.route(Api_1.RequestMethod.DELETE, `notes/${note1.id}`);
        await api.route(Api_1.RequestMethod.DELETE, `notes/${note2.id}`, { permanent: '1' });
        expect((await Note_1.default.load(note1.id)).deleted_time).toBeGreaterThanOrEqual(beforeTime);
        expect(await Note_1.default.load(note2.id)).toBeFalsy();
    });
    test('should not stop execution if a file can not be processed', async () => {
        Logger_1.default.globalLogger.enabled = false;
        const result = await (0, notes_1.createResourcesFromPaths)([
            { originalUrl: 'asdf.png', path: `${__dirname}/bad-path-should-not-exist` },
            { originalUrl: 'asdf.png', path: `${__dirname}/../../../images/SideMenuHeader.png` },
        ]);
        Logger_1.default.globalLogger.enabled = true;
        expect(result.length).toBe(2);
    });
    test('should not return notes in the trash by default', async () => {
        const api = new Api_1.default();
        const note1 = await Note_1.default.save({});
        const note2 = await Note_1.default.save({});
        await Note_1.default.delete(note1.id, { toTrash: true });
        {
            const notes = await api.route(Api_1.RequestMethod.GET, 'notes');
            expect(notes.items.length).toBe(1);
            expect(notes.items[0].id).toBe(note2.id);
        }
        {
            const notes = await api.route(Api_1.RequestMethod.GET, 'notes', { include_deleted: '1' });
            expect(notes.items.length).toBe(2);
        }
    });
    test('should not return conflicts by default', async () => {
        const api = new Api_1.default();
        const note1 = await Note_1.default.save({});
        await Note_1.default.save({ is_conflict: 1 });
        {
            const notes = await api.route(Api_1.RequestMethod.GET, 'notes');
            expect(notes.items.length).toBe(1);
            expect(notes.items[0].id).toBe(note1.id);
        }
        {
            const notes = await api.route(Api_1.RequestMethod.GET, 'notes', { include_conflicts: '1' });
            expect(notes.items.length).toBe(2);
        }
    });
});
//# sourceMappingURL=notes.test.js.map