"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const Setting_1 = require("../../models/Setting");
const CommandService_1 = require("../CommandService");
const locale_1 = require("../../locale");
const KvStore_1 = require("../KvStore");
class SpellCheckerService {
    constructor() {
        this.latestSelectedLanguages_ = [];
    }
    static instance() {
        if (this.instance_)
            return this.instance_;
        this.instance_ = new SpellCheckerService();
        return this.instance_;
    }
    async initialize(driver) {
        this.driver_ = driver;
        this.latestSelectedLanguages_ = await this.loadLatestSelectedLanguages();
        this.setupDefaultLanguage();
        this.applyStateToDriver();
    }
    get defaultLanguage() {
        return 'en-US';
    }
    async loadLatestSelectedLanguages() {
        const result = await KvStore_1.default.instance().value('spellCheckerService.latestSelectedLanguages');
        if (!result)
            return [];
        return JSON.parse(result);
    }
    async addLatestSelectedLanguage(language) {
        // This function will add selected languages to the history. History size will be capped at languagesHistorySizeMax,
        // but it can be bigger. Enabled languages will always be in the history, even if it count greater then
        // languagesHistorySizeMax, in such case if one of the languages will be disabled it will disappear from history.
        const languagesHistorySizeMax = 5;
        const languages = this.latestSelectedLanguages_.slice();
        if (!languages.includes(language)) {
            languages.push(language);
        }
        if (languages.length > languagesHistorySizeMax) {
            // eslint-disable-next-line github/array-foreach -- Old code before rule was applied
            this.latestSelectedLanguages_.forEach(l => {
                if (!this.languages.includes(l) && languages.length > languagesHistorySizeMax)
                    languages.splice(languages.indexOf(l), 1);
            });
        }
        this.latestSelectedLanguages_ = languages;
        await KvStore_1.default.instance().setValue('spellCheckerService.latestSelectedLanguages', JSON.stringify(this.latestSelectedLanguages_));
    }
    setupDefaultLanguage() {
        if (Setting_1.default.value('spellChecker.languages').length === 0) {
            const l = this.driver_.language;
            if (this.availableLanguages.includes(l)) {
                this.setLanguage(l);
            }
            else {
                this.setLanguage(this.defaultLanguage);
            }
        }
    }
    get availableLanguages() {
        return this.driver_.availableLanguages;
    }
    applyStateToDriver() {
        this.driver_.setLanguages(this.enabled ? this.languages : []);
    }
    setLanguage(language) {
        let enabledLanguages = [...this.languages];
        if (enabledLanguages.includes(language)) {
            enabledLanguages = enabledLanguages.filter(obj => obj !== language);
        }
        else {
            enabledLanguages.push(language);
        }
        Setting_1.default.setValue('spellChecker.languages', enabledLanguages);
        this.applyStateToDriver();
        void this.addLatestSelectedLanguage(language);
    }
    get languages() {
        return Setting_1.default.value('spellChecker.languages');
    }
    get enabled() {
        return Setting_1.default.value('spellChecker.enabled');
    }
    toggleEnabled() {
        Setting_1.default.toggle('spellChecker.enabled');
        this.applyStateToDriver();
    }
    async addToDictionary(language, word) {
        this.driver_.addWordToSpellCheckerDictionary(language, word);
    }
    // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
    contextMenuItems(misspelledWord, dictionarySuggestions) {
        if (!misspelledWord)
            return [];
        const output = [];
        output.push({ type: 'separator' });
        if (dictionarySuggestions.length) {
            for (const suggestion of dictionarySuggestions) {
                output.push({
                    label: suggestion,
                    click: () => {
                        void CommandService_1.default.instance().execute('replaceMisspelling', suggestion);
                    },
                });
            }
        }
        else {
            output.push({
                label: `(${(0, locale_1._)('No suggestions')})`,
                enabled: false,
                click: () => { },
            });
        }
        output.push({ type: 'separator' });
        output.push({
            label: (0, locale_1._)('Add to dictionary'),
            click: () => {
                void this.addToDictionary(this.languages[0], misspelledWord);
            },
        });
        return output;
    }
    changeLanguageMenuItem(language, enabled, checked) {
        return {
            label: (0, locale_1.countryDisplayName)(language),
            type: 'checkbox',
            checked: checked,
            enabled: enabled,
            click: () => {
                this.setLanguage(language);
            },
        };
    }
    changeLanguageMenuItems(selectedLanguages, enabled) {
        const languageMenuItems = [];
        for (const locale of this.driver_.availableLanguages) {
            languageMenuItems.push(this.changeLanguageMenuItem(locale, enabled, selectedLanguages.includes(locale)));
        }
        // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
        languageMenuItems.sort((a, b) => {
            return a.label < b.label ? -1 : +1;
        });
        return languageMenuItems;
    }
    spellCheckerConfigMenuItems(selectedLanguages, useSpellChecker) {
        const latestLanguageItems = this.latestSelectedLanguages_.map((language) => {
            return this.changeLanguageMenuItem(language, true, selectedLanguages.includes(language));
        });
        // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
        if (latestLanguageItems.length)
            latestLanguageItems.splice(0, 0, { type: 'separator' });
        // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
        latestLanguageItems.sort((a, b) => {
            return a.label < b.label ? -1 : +1;
        });
        return [
            {
                label: (0, locale_1._)('Use spell checker'),
                type: 'checkbox',
                checked: useSpellChecker,
                click: () => {
                    this.toggleEnabled();
                },
            },
            ...latestLanguageItems,
            {
                type: 'separator',
            },
            // Can be removed once it does work
            // {
            // 	label: '⚠ Spell checker doesn\'t work in Markdown editor ⚠',
            // 	enabled: false,
            // },
            {
                type: 'separator',
            },
            {
                label: (0, locale_1._)('Change language'),
                submenu: this.changeLanguageMenuItems(selectedLanguages, useSpellChecker),
            },
        ];
    }
    spellCheckerConfigMenuItem(selectedLanguages, useSpellChecker) {
        return {
            label: (0, locale_1._)('Spell checker'),
            submenu: this.spellCheckerConfigMenuItems(selectedLanguages, useSpellChecker),
        };
    }
}
exports.default = SpellCheckerService;
//# sourceMappingURL=SpellCheckerService.js.map