"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const BaseItem_1 = require("../../models/BaseItem");
const Note_1 = require("../../models/Note");
const test_utils_1 = require("../../testing/test-utils");
const time_1 = require("../../time");
const ItemUploader_1 = require("./ItemUploader");
// eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
function clearArray(a) {
    a.splice(0, a.length);
}
function newFakeApi() {
    // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
    return { supportsMultiPut: true };
}
// eslint-disable-next-line @typescript-eslint/ban-types -- Old code before rule was applied
function newFakeApiCall(callRecorder, itemBodyCallback = null) {
    // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
    const apiCall = async (callName, ...args) => {
        callRecorder.push({ name: callName, args });
        if (callName === 'multiPut') {
            const [batch] = args;
            // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
            const output = { items: {} };
            for (const item of batch) {
                if (itemBodyCallback) {
                    output.items[item.name] = itemBodyCallback(item);
                }
                else {
                    output.items[item.name] = {
                        item: item.body,
                        error: null,
                    };
                }
            }
            return output;
        }
    };
    return apiCall;
}
describe('synchronizer/ItemUploader', () => {
    beforeEach(async () => {
        await (0, test_utils_1.setupDatabaseAndSynchronizer)(1);
        await (0, test_utils_1.setupDatabaseAndSynchronizer)(2);
        await (0, test_utils_1.switchClient)(1);
    });
    it('should batch uploads and use the cache afterwards', (async () => {
        const callRecorder = [];
        const itemUploader = new ItemUploader_1.default(newFakeApi(), newFakeApiCall(callRecorder));
        const notes = [
            await Note_1.default.save({ title: '1' }),
            await Note_1.default.save({ title: '2' }),
        ];
        await itemUploader.preUploadItems(notes);
        // There should be only one call to "multiPut" because the items have
        // been batched.
        expect(callRecorder.length).toBe(1);
        expect(callRecorder[0].name).toBe('multiPut');
        clearArray(callRecorder);
        // Now if we try to upload the item it shouldn't call the API because it
        // will use the cached item.
        await itemUploader.serializeAndUploadItem(Note_1.default, BaseItem_1.default.systemPath(notes[0]), notes[0]);
        expect(callRecorder.length).toBe(0);
        // Now try to process a note that hasn't been cached. In that case, it
        // should call "PUT" directly.
        const note3 = await Note_1.default.save({ title: '3' });
        await itemUploader.serializeAndUploadItem(Note_1.default, BaseItem_1.default.systemPath(note3), note3);
        expect(callRecorder.length).toBe(1);
        expect(callRecorder[0].name).toBe('put');
    }));
    it('should not batch upload if the items are over the batch size limit', (async () => {
        const callRecorder = [];
        const itemUploader = new ItemUploader_1.default(newFakeApi(), newFakeApiCall(callRecorder));
        itemUploader.maxBatchSize = 1;
        const notes = [
            await Note_1.default.save({ title: '1' }),
            await Note_1.default.save({ title: '2' }),
        ];
        await itemUploader.preUploadItems(notes);
        expect(callRecorder.length).toBe(0);
    }));
    it('should not use the cache if the note has changed since the pre-upload', (async () => {
        const callRecorder = [];
        const itemUploader = new ItemUploader_1.default(newFakeApi(), newFakeApiCall(callRecorder));
        const notes = [
            await Note_1.default.save({ title: '1' }),
            await Note_1.default.save({ title: '2' }),
        ];
        await itemUploader.preUploadItems(notes);
        clearArray(callRecorder);
        await itemUploader.serializeAndUploadItem(Note_1.default, BaseItem_1.default.systemPath(notes[0]), notes[0]);
        expect(callRecorder.length).toBe(0);
        await time_1.default.msleep(1);
        notes[1] = await Note_1.default.save({ title: '22' });
        await itemUploader.serializeAndUploadItem(Note_1.default, BaseItem_1.default.systemPath(notes[1]), notes[1]);
        expect(callRecorder.length).toBe(1);
    }));
    it('should respect the max batch size', (async () => {
        const callRecorder = [];
        const itemUploader = new ItemUploader_1.default(newFakeApi(), newFakeApiCall(callRecorder));
        const notes = [
            await Note_1.default.save({ title: '1' }),
            await Note_1.default.save({ title: '2' }),
            await Note_1.default.save({ title: '3' }),
        ];
        const noteSize = BaseItem_1.default.systemPath(notes[0]).length + (await Note_1.default.serializeForSync(notes[0])).length;
        itemUploader.maxBatchSize = noteSize * 2;
        // It should send two batches - one with two notes, and the second with
        // only one note.
        await itemUploader.preUploadItems(notes);
        expect(callRecorder.length).toBe(2);
        expect(callRecorder[0].args[0].length).toBe(2);
        expect(callRecorder[1].args[0].length).toBe(1);
    }));
    it('should rethrow error for items within the batch', (async () => {
        const callRecorder = [];
        const notes = [
            await Note_1.default.save({ title: '1' }),
            await Note_1.default.save({ title: '2' }),
            await Note_1.default.save({ title: '3' }),
        ];
        // Simulates throwing an error on note 2
        // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
        const itemBodyCallback = (item) => {
            if (item.name === BaseItem_1.default.systemPath(notes[1])) {
                return { error: new Error('Could not save item'), item: null };
            }
            else {
                return { error: null, item: item.body };
            }
        };
        const itemUploader = new ItemUploader_1.default(newFakeApi(), newFakeApiCall(callRecorder, itemBodyCallback));
        await itemUploader.preUploadItems(notes);
        await (0, test_utils_1.expectNotThrow)(async () => itemUploader.serializeAndUploadItem(Note_1.default, BaseItem_1.default.systemPath(notes[0]), notes[0]));
        await (0, test_utils_1.expectThrow)(async () => itemUploader.serializeAndUploadItem(Note_1.default, BaseItem_1.default.systemPath(notes[1]), notes[1]), null);
        await (0, test_utils_1.expectNotThrow)(async () => itemUploader.serializeAndUploadItem(Note_1.default, BaseItem_1.default.systemPath(notes[2]), notes[2]));
    }));
});
//# sourceMappingURL=ItemUploader.test.js.map