"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shim_1 = require("./shim");
const geolocation_node_1 = require("./geolocation-node");
const locale_1 = require("./locale");
const fs_driver_node_1 = require("./fs-driver-node");
const Note_1 = require("./models/Note");
const Resource_1 = require("./models/Resource");
const path_utils_1 = require("./path-utils");
const fs = require("fs-extra");
const promises_1 = require("fs/promises");
const replaceUnsupportedCharacters_1 = require("./utils/replaceUnsupportedCharacters");
const file_type_1 = require("file-type");
const crypto_1 = require("./services/e2ee/crypto");
const file_api_driver_local_1 = require("./file-api-driver-local");
const mimeUtils = require("./mime-utils");
const BaseItem_1 = require("./models/BaseItem");
const os_1 = require("os");
const { _ } = require('./locale');
const http = require('http');
const https = require('https');
const { HttpProxyAgent, HttpsProxyAgent } = require('hpagent');
const toRelative = require('relative');
const timers = require('timers');
const zlib = require('zlib');
const dgram = require('dgram');
// eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
const proxySettings = {};
function fileExists(filePath) {
    try {
        return fs.statSync(filePath).isFile();
    }
    catch (error) {
        return false;
    }
}
function isUrlHttps(url) {
    return url.startsWith('https');
}
function resolveProxyUrl(proxyUrl) {
    return (proxyUrl ||
        process.env['http_proxy'] ||
        process.env['https_proxy'] ||
        process.env['HTTP_PROXY'] ||
        process.env['HTTPS_PROXY']);
}
// https://github.com/sindresorhus/callsites/blob/main/index.js
function callsites() {
    const _prepareStackTrace = Error.prepareStackTrace;
    Error.prepareStackTrace = (_any, stack) => stack;
    const stack = new Error().stack.slice(1);
    Error.prepareStackTrace = _prepareStackTrace;
    return stack;
}
const gunzipFile = function (source, destination) {
    if (!fileExists(source)) {
        throw new Error(`No such file: ${source}`);
    }
    return new Promise((resolve, reject) => {
        // prepare streams
        const src = fs.createReadStream(source);
        const dest = fs.createWriteStream(destination);
        // extract the archive
        src.pipe(zlib.createGunzip()).pipe(dest);
        // callback on extract completion
        dest.on('close', () => {
            resolve(null);
        });
        src.on('error', () => {
            reject();
        });
        dest.on('error', () => {
            reject();
        });
    });
};
// eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
function setupProxySettings(options) {
    proxySettings.maxConcurrentConnections = options.maxConcurrentConnections;
    proxySettings.proxyTimeout = options.proxyTimeout;
    proxySettings.proxyEnabled = options.proxyEnabled;
    proxySettings.proxyUrl = options.proxyUrl;
}
function shimInit(options = null) {
    options = Object.assign({ sharp: null, keytar: null, React: null, appVersion: null, electronBridge: null, nodeSqlite: null, pdfJs: null, isAppleSilicon: () => false }, options);
    const sharp = options.sharp;
    const keytar = (shim_1.default.isWindows() || shim_1.default.isMac()) && !shim_1.default.isPortable() ? options.keytar : null;
    const appVersion = options.appVersion;
    const pdfJs = options.pdfJs;
    shim_1.default.setNodeSqlite(options.nodeSqlite);
    shim_1.default.fsDriver = () => {
        throw new Error('Not implemented');
    };
    shim_1.default.FileApiDriverLocal = file_api_driver_local_1.default;
    shim_1.default.Geolocation = geolocation_node_1.default;
    shim_1.default.FormData = require('form-data');
    shim_1.default.sjclModule = require('./vendor/sjcl.js');
    shim_1.default.crypto = crypto_1.default;
    shim_1.default.electronBridge_ = options.electronBridge;
    shim_1.default.fsDriver = () => {
        if (!shim_1.default.fsDriver_)
            shim_1.default.fsDriver_ = new fs_driver_node_1.default();
        return shim_1.default.fsDriver_;
    };
    shim_1.default.sharpEnabled = () => {
        return !!sharp;
    };
    shim_1.default.dgram = () => {
        return dgram;
    };
    if (options.React) {
        shim_1.default.react = () => {
            return options.React;
        };
    }
    shim_1.default.electronBridge = () => {
        return shim_1.default.electronBridge_;
    };
    shim_1.default.randomBytes = async (count) => {
        const buffer = require('crypto').randomBytes(count);
        return Array.from(buffer);
    };
    shim_1.default.isAppleSilicon = () => {
        return options.isAppleSilicon ? options.isAppleSilicon() : false;
    };
    shim_1.default.platformArch = () => {
        const c = (0, os_1.cpus)();
        if (!c.length)
            return '';
        return c[0].model;
    };
    // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
    shim_1.default.detectAndSetLocale = function (Setting) {
        let locale = shim_1.default.isElectron() ? shim_1.default.electronBridge().getLocale() : process.env.LANG;
        if (!locale)
            locale = (0, locale_1.defaultLocale)();
        locale = locale.split('.');
        locale = locale[0];
        locale = (0, locale_1.closestSupportedLocale)(locale);
        Setting.setValue('locale', locale);
        (0, locale_1.setLocale)(locale);
        return locale;
    };
    shim_1.default.writeImageToFile = async function (nativeImage, mime, targetPath) {
        if (shim_1.default.isElectron()) {
            // For Electron
            let buffer = null;
            mime = mime.toLowerCase();
            if (mime === 'image/png') {
                buffer = nativeImage.toPNG();
            }
            else if (mime === 'image/jpg' || mime === 'image/jpeg') {
                buffer = nativeImage.toJPEG(90);
            }
            if (!buffer)
                throw new Error(`Cannot resize image because mime type "${mime}" is not supported: ${targetPath}`);
            await shim_1.default.fsDriver().writeFile(targetPath, buffer, 'buffer');
        }
        else {
            throw new Error('Node support not implemented');
        }
    };
    shim_1.default.showMessageBox = async (message, options = null) => {
        if (shim_1.default.isElectron()) {
            return shim_1.default.electronBridge().showMessageBox(message, options);
        }
        else {
            throw new Error('Not implemented');
        }
    };
    const handleResizeImage_ = async function (filePath, targetPath, mime, resizeLargeImages) {
        const maxDim = Resource_1.default.IMAGE_MAX_DIMENSION;
        if (shim_1.default.isElectron()) {
            // For Electron/renderer process
            // Note that we avoid nativeImage because it loses rotation metadata.
            // See https://github.com/electron/electron/issues/41189
            //
            // After the upstream bug has been fixed, this should be reverted to using
            // nativeImage (see commit 99e8818ba093a931b1a0cbccbee0b94a4fd37a54 for the
            // original code).
            const image = new Image();
            image.src = filePath;
            await new Promise((resolve, reject) => {
                image.onload = () => resolve();
                image.onerror = () => reject(new Error(`Image at ${filePath} failed to load.`));
                image.onabort = () => reject(new Error(`Loading stopped for image at ${filePath}.`));
            });
            if (!image.complete || (image.width === 0 && image.height === 0)) {
                throw new Error(`Image is invalid or does not exist: ${filePath}`);
            }
            const saveOriginalImage = async () => {
                await shim_1.default.fsDriver().copy(filePath, targetPath);
                return true;
            };
            const saveResizedImage = async () => {
                let newWidth, newHeight;
                if (image.width > image.height) {
                    newWidth = maxDim;
                    newHeight = image.height * maxDim / image.width;
                }
                else {
                    newWidth = image.width * maxDim / image.height;
                    newHeight = maxDim;
                }
                const canvas = new OffscreenCanvas(newWidth, newHeight);
                const ctx = canvas.getContext('2d');
                ctx.drawImage(image, 0, 0, newWidth, newHeight);
                const resizedImage = await canvas.convertToBlob({ type: mime });
                await fs.writeFile(targetPath, Buffer.from(await resizedImage.arrayBuffer()));
                return true;
            };
            const canResize = image.width > maxDim || image.height > maxDim;
            if (canResize) {
                if (resizeLargeImages === 'alwaysAsk') {
                    const Yes = 0, No = 1, Cancel = 2;
                    const userAnswer = await shim_1.default.showMessageBox(`${_('You are about to attach a large image (%dx%d pixels). Would you like to resize it down to %d pixels before attaching it?', image.width, image.height, maxDim)}\n\n${_('(You may disable this prompt in the options)')}`, {
                        buttons: [_('Yes'), _('No'), _('Cancel')],
                    });
                    if (userAnswer === Yes)
                        return await saveResizedImage();
                    if (userAnswer === No)
                        return await saveOriginalImage();
                    if (userAnswer === Cancel)
                        return false;
                }
                else if (resizeLargeImages === 'alwaysResize') {
                    return await saveResizedImage();
                }
            }
            return await saveOriginalImage();
        }
        else {
            // For the CLI tool
            let md = null;
            // eslint-disable-next-line @typescript-eslint/no-explicit-any
            let image = null;
            if (sharp) {
                image = sharp(filePath);
                md = await image.metadata();
            }
            if (!md || (md.width <= maxDim && md.height <= maxDim)) {
                await shim_1.default.fsDriver().copy(filePath, targetPath);
                return true;
            }
            return new Promise((resolve, reject) => {
                image
                    .resize(Resource_1.default.IMAGE_MAX_DIMENSION, Resource_1.default.IMAGE_MAX_DIMENSION, {
                    fit: 'inside',
                    withoutEnlargement: true,
                })
                    // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
                    .toFile(targetPath, (error, info) => {
                    if (error) {
                        reject(error);
                    }
                    else {
                        resolve(info);
                    }
                });
            });
        }
    };
    // This is a bit of an ugly method that's used to both create a new resource
    // from a file, and update one. To update a resource, pass the
    // destinationResourceId option. This method is indirectly tested in
    // Api.test.ts.
    shim_1.default.createResourceFromPath = async function (filePath, defaultProps = null, options = null) {
        options = Object.assign({ resizeLargeImages: 'always', userSideValidation: false, destinationResourceId: '' }, options);
        const isUpdate = !!options.destinationResourceId;
        if (!(await fs.pathExists(filePath)))
            throw new Error(_('Cannot access %s', filePath));
        defaultProps = defaultProps ? defaultProps : {};
        let resourceId = defaultProps.id ? defaultProps.id : BaseItem_1.default.generateUuid();
        if (isUpdate)
            resourceId = options.destinationResourceId;
        let resource = isUpdate ? {} : Resource_1.default.new();
        resource.id = resourceId;
        // When this is an update we auto-update the mime type, in case the
        // content type has changed, but we keep the title. It is still possible
        // to modify the title on update using defaultProps.
        resource.mime = mimeUtils.fromFilename(filePath);
        if (!isUpdate)
            resource.title = (0, path_utils_1.basename)(filePath);
        let fileExt = (0, path_utils_1.safeFileExtension)((0, path_utils_1.fileExtension)(filePath));
        if (!resource.mime) {
            const detectedType = await (0, file_type_1.fromFile)(filePath);
            if (detectedType) {
                fileExt = fileExt ? fileExt : detectedType.ext;
                resource.mime = detectedType.mime;
            }
            else {
                resource.mime = 'application/octet-stream';
            }
        }
        resource.file_extension = fileExt;
        const targetPath = Resource_1.default.fullPath(resource);
        if (options.resizeLargeImages !== 'never' && ['image/jpeg', 'image/jpg', 'image/png'].includes(resource.mime)) {
            const ok = await handleResizeImage_(filePath, targetPath, resource.mime, options.resizeLargeImages);
            if (!ok)
                return null;
        }
        else {
            await fs.copy(filePath, targetPath, { overwrite: true });
        }
        // While a whole object can be passed as defaultProps, we only just
        // support the title and ID (used above). Any other prop should be
        // derived from the provided file.
        if ('title' in defaultProps)
            resource.title = defaultProps.title;
        const itDoes = await shim_1.default.fsDriver().waitTillExists(targetPath);
        if (!itDoes)
            throw new Error(`Resource file was not created: ${targetPath}`);
        const fileStat = await shim_1.default.fsDriver().stat(targetPath);
        resource.size = fileStat.size;
        // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
        const saveOptions = { isNew: true };
        if (options.userSideValidation)
            saveOptions.userSideValidation = true;
        if (isUpdate) {
            saveOptions.isNew = false;
            const tempPath = `${targetPath}.tmp`;
            await shim_1.default.fsDriver().move(targetPath, tempPath);
            resource = await Resource_1.default.save(resource, saveOptions);
            await Resource_1.default.updateResourceBlobContent(resource.id, tempPath);
            await shim_1.default.fsDriver().remove(tempPath);
            return resource;
        }
        else {
            return Resource_1.default.save(resource, saveOptions);
        }
    };
    shim_1.default.attachFileToNoteBody = async function (noteBody, filePath, position = null, options = null) {
        options = Object.assign({ createFileURL: false, markupLanguage: 1, resourcePrefix: '', resourceSuffix: '' }, options);
        const { basename } = require('path');
        const { escapeTitleText } = require('./markdownUtils').default;
        const { toFileProtocolPath } = require('./path-utils');
        let resource = null;
        if (!options.createFileURL) {
            resource = await shim_1.default.createResourceFromPath(filePath, null, options);
            if (!resource)
                return null;
        }
        const newBody = [];
        if (position === null) {
            position = noteBody ? noteBody.length : 0;
        }
        if (noteBody && position)
            newBody.push(noteBody.substr(0, position));
        if (!options.createFileURL) {
            newBody.push(options.resourcePrefix + Resource_1.default.markupTag(resource, options.markupLanguage) + options.resourceSuffix);
        }
        else {
            const filename = escapeTitleText(basename(filePath)); // to get same filename as standard drag and drop
            const fileURL = `[${filename}](${toFileProtocolPath(filePath)})`;
            newBody.push(options.resourcePrefix + fileURL + options.resourceSuffix);
        }
        if (noteBody)
            newBody.push(noteBody.substr(position));
        return newBody.join('');
    };
    shim_1.default.attachFileToNote = async function (note, filePath, options = {}) {
        var _a;
        if (!options)
            options = {};
        if (note.markup_language)
            options.markupLanguage = note.markup_language;
        const newBody = await shim_1.default.attachFileToNoteBody(note.body, filePath, (_a = options.position) !== null && _a !== void 0 ? _a : 0, options);
        if (!newBody)
            return null;
        const newNote = Object.assign(Object.assign({}, note), { body: newBody });
        return Note_1.default.save(newNote);
    };
    shim_1.default.imageToDataUrl = async (filePath, maxSize) => {
        if (shim_1.default.isElectron()) {
            const nativeImage = require('electron').nativeImage;
            let image = nativeImage.createFromPath(filePath);
            if (!image)
                throw new Error(`Could not load image: ${filePath}`);
            const ext = (0, path_utils_1.fileExtension)(filePath).toLowerCase();
            if (!['jpg', 'jpeg', 'png'].includes(ext))
                throw new Error(`Unsupported file format: ${ext}`);
            if (maxSize) {
                const size = image.getSize();
                if (size.width > maxSize || size.height > maxSize) {
                    console.warn(`Image is over ${maxSize}px - resizing it: ${filePath}`);
                    // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
                    const options = {};
                    if (size.width > size.height) {
                        options.width = maxSize;
                    }
                    else {
                        options.height = maxSize;
                    }
                    image = image.resize(options);
                }
            }
            return image.toDataURL();
        }
        else {
            throw new Error('Unsupported method');
        }
    };
    shim_1.default.imageFromDataUrl = async function (imageDataUrl, filePath, options = null) {
        if (options === null)
            options = {};
        if (shim_1.default.isElectron()) {
            const nativeImage = require('electron').nativeImage;
            let image = nativeImage.createFromDataURL(imageDataUrl);
            if (image.isEmpty())
                throw new Error('Could not convert data URL to image - perhaps the format is not supported (eg. image/gif)'); // Would throw for example if the image format is no supported (eg. image/gif)
            if (options.cropRect) {
                // Crop rectangle values need to be rounded or the crop() call will fail
                const c = options.cropRect;
                if ('x' in c)
                    c.x = Math.round(c.x);
                if ('y' in c)
                    c.y = Math.round(c.y);
                if ('width' in c)
                    c.width = Math.round(c.width);
                if ('height' in c)
                    c.height = Math.round(c.height);
                image = image.crop(c);
            }
            const mime = mimeUtils.fromDataUrl(imageDataUrl);
            await shim_1.default.writeImageToFile(image, mime, filePath);
        }
        else {
            if (options.cropRect)
                throw new Error('Crop rect not supported in Node');
            const imageDataURI = require('image-data-uri');
            const result = imageDataURI.decode(imageDataUrl);
            await shim_1.default.fsDriver().writeFile(filePath, result.dataBuffer, 'buffer');
        }
    };
    const nodeFetch = require('node-fetch');
    // Not used??
    shim_1.default.readLocalFileBase64 = path => {
        const data = fs.readFileSync(path);
        return new Buffer(data).toString('base64');
    };
    shim_1.default.fetch = async function (url, options = {}) {
        try { // Check if the url is valid
            new URL(url);
        }
        catch (error) { // If the url is not valid, a TypeError will be thrown
            throw new Error(`Not a valid URL: ${url}`);
        }
        const resolvedProxyUrl = resolveProxyUrl(proxySettings.proxyUrl);
        options.agent = (resolvedProxyUrl && proxySettings.proxyEnabled) ? shim_1.default.proxyAgent(url, resolvedProxyUrl) : null;
        return shim_1.default.fetchWithRetry(() => {
            return nodeFetch(url, options);
        }, options);
    };
    // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
    shim_1.default.fetchBlob = async function (url, options) {
        if (!options || !options.path)
            throw new Error('fetchBlob: target file path is missing');
        if (!options.method)
            options.method = 'GET';
        // if (!('maxRetry' in options)) options.maxRetry = 5;
        // 21 maxRedirects is the default amount from follow-redirects library
        // 20 seems to be the max amount that most popular browsers will allow
        if (!options.maxRedirects)
            options.maxRedirects = 21;
        if (!options.timeout)
            options.timeout = undefined;
        const urlParse = require('url').parse;
        url = urlParse(url.trim());
        const method = options.method ? options.method : 'GET';
        const http = url.protocol.toLowerCase() === 'http:' ? require('follow-redirects').http : require('follow-redirects').https;
        const headers = options.headers ? options.headers : {};
        const filePath = options.path;
        const downloadController = options.downloadController;
        // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
        function makeResponse(response) {
            return {
                ok: response.statusCode < 400,
                path: filePath,
                text: () => {
                    return response.statusMessage;
                },
                json: () => {
                    return { message: `${response.statusCode}: ${response.statusMessage}` };
                },
                status: response.statusCode,
                headers: response.headers,
            };
        }
        // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
        const requestOptions = {
            protocol: url.protocol,
            host: url.hostname,
            port: url.port,
            method: method,
            path: url.pathname + (url.query ? `?${url.query}` : ''),
            headers: headers,
            timeout: options.timeout,
            maxRedirects: options.maxRedirects,
        };
        const resolvedProxyUrl = resolveProxyUrl(proxySettings.proxyUrl);
        requestOptions.agent = (resolvedProxyUrl && proxySettings.proxyEnabled) ? shim_1.default.proxyAgent(url.href, resolvedProxyUrl) : null;
        const doFetchOperation = async () => {
            return new Promise((resolve, reject) => {
                // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
                let file = null;
                // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
                const cleanUpOnError = (error) => {
                    // We ignore any unlink error as we only want to report on the main error
                    void fs.unlink(filePath)
                        // eslint-disable-next-line promise/prefer-await-to-then -- Old code before rule was applied
                        .catch(() => { })
                        // eslint-disable-next-line promise/prefer-await-to-then -- Old code before rule was applied
                        .then(() => {
                        if (file) {
                            file.close(() => {
                                file = null;
                                reject(error);
                            });
                        }
                        else {
                            reject(error);
                        }
                    });
                };
                try {
                    // Note: relative paths aren't supported
                    file = fs.createWriteStream(filePath);
                    // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
                    file.on('error', (error) => {
                        cleanUpOnError(error);
                    });
                    const requestStart = new Date();
                    // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
                    const request = http.request(requestOptions, (response) => {
                        if (downloadController) {
                            response.on('data', downloadController.handleChunk(request));
                        }
                        response.pipe(file);
                        const isGzipped = response.headers['content-encoding'] === 'gzip';
                        file.on('finish', () => {
                            file.close(async () => {
                                if (isGzipped) {
                                    const gzipFilePath = `${filePath}.gzip`;
                                    await shim_1.default.fsDriver().move(filePath, gzipFilePath);
                                    try {
                                        await gunzipFile(gzipFilePath, filePath);
                                        // Calling request.destroy() within the downloadController can cause problems.
                                        // The response.pipe(file) will continue even after request.destroy() is called,
                                        // potentially causing the same promise to resolve while the cleanUpOnError
                                        // is removing the file that have been downloaded by this function.
                                        if (request.destroyed)
                                            return;
                                        resolve(makeResponse(response));
                                    }
                                    catch (error) {
                                        cleanUpOnError(error);
                                    }
                                    await shim_1.default.fsDriver().remove(gzipFilePath);
                                }
                                else {
                                    if (request.destroyed)
                                        return;
                                    resolve(makeResponse(response));
                                }
                            });
                        });
                    });
                    request.on('timeout', () => {
                        // We choose to not destroy the request when a timeout value is not specified to keep
                        // the behavior we had before the addition of this event handler.
                        if (!requestOptions.timeout)
                            return;
                        request.destroy(new Error(`Request timed out. Timeout value: ${requestOptions.timeout}ms. Actual connection time: ${new Date().getTime() - requestStart.getTime()}ms`));
                    });
                    // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
                    request.on('error', (error) => {
                        cleanUpOnError(error);
                    });
                    request.end();
                }
                catch (error) {
                    cleanUpOnError(error);
                }
            });
        };
        return shim_1.default.fetchWithRetry(doFetchOperation, options);
    };
    shim_1.default.uploadBlob = async function (url, options) {
        if (!options || !options.path)
            throw new Error('uploadBlob: source file path is missing');
        const content = await fs.readFile(options.path);
        options = Object.assign(Object.assign({}, options), { body: content });
        return shim_1.default.fetch(url, options);
    };
    shim_1.default.stringByteLength = function (string) {
        return Buffer.byteLength(string, 'utf-8');
    };
    shim_1.default.Buffer = Buffer;
    shim_1.default.openUrl = url => {
        // Returns true if it opens the file successfully; returns false if it could
        // not find the file.
        return shim_1.default.electronBridge().openExternal(url);
    };
    shim_1.default.httpAgent_ = null;
    shim_1.default.httpAgent = url => {
        if (!shim_1.default.httpAgent_) {
            const AgentSettings = {
                keepAlive: true,
                maxSockets: 1,
                keepAliveMsecs: 5000,
            };
            shim_1.default.httpAgent_ = {
                http: new http.Agent(AgentSettings),
                https: new https.Agent(AgentSettings),
            };
        }
        return url.startsWith('https') ? shim_1.default.httpAgent_.https : shim_1.default.httpAgent_.http;
    };
    shim_1.default.proxyAgent = (serverUrl, proxyUrl) => {
        const proxyAgentConfig = {
            keepAlive: true,
            maxSockets: proxySettings.maxConcurrentConnections,
            keepAliveMsecs: 5000,
            proxy: proxyUrl,
            timeout: proxySettings.proxyTimeout * 1000,
        };
        // Based on https://github.com/delvedor/hpagent#usage
        if (!isUrlHttps(proxyUrl) && !isUrlHttps(serverUrl)) {
            return new HttpProxyAgent(proxyAgentConfig);
        }
        else if (isUrlHttps(proxyUrl) && !isUrlHttps(serverUrl)) {
            return new HttpProxyAgent(proxyAgentConfig);
        }
        else if (!isUrlHttps(proxyUrl) && isUrlHttps(serverUrl)) {
            return new HttpsProxyAgent(proxyAgentConfig);
        }
        else {
            return new HttpsProxyAgent(proxyAgentConfig);
        }
    };
    shim_1.default.openOrCreateFile = (filepath, defaultContents) => {
        // If the file doesn't exist, create it
        if (!fs.existsSync(filepath)) {
            fs.writeFile(filepath, defaultContents, 'utf-8', (error) => {
                if (error) {
                    console.error(`error: ${error}`);
                }
            });
        }
        // Open the file
        // Don't use openUrl() there.
        // The underneath require('electron').shell.openExternal() has a bug
        // https://github.com/electron/electron/issues/31347
        return shim_1.default.electronBridge().openItem(filepath);
    };
    shim_1.default.waitForFrame = () => { };
    shim_1.default.appVersion = () => {
        if (appVersion)
            return appVersion();
        // Should not happen but don't throw an error because version number is
        // used in error messages.
        return 'unknown';
    };
    shim_1.default.pathRelativeToCwd = (path) => {
        return toRelative(process.cwd(), path);
    };
    shim_1.default.setTimeout = (fn, interval) => {
        return timers.setTimeout(fn, interval);
    };
    shim_1.default.setInterval = (fn, interval) => {
        return timers.setInterval(fn, interval);
    };
    shim_1.default.clearTimeout = (id) => {
        return timers.clearTimeout(id);
    };
    shim_1.default.clearInterval = (id) => {
        return timers.clearInterval(id);
    };
    shim_1.default.keytar = () => {
        return keytar;
    };
    shim_1.default.requireDynamic = (path) => {
        if (path.indexOf('.') === 0) {
            // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
            const sites = callsites();
            if (sites.length <= 1)
                throw new Error(`Cannot require file (1) ${path}`);
            const filename = sites[1].getFileName();
            if (!filename)
                throw new Error(`Cannot require file (2) ${path}`);
            const fileDirName = require('path').dirname(filename);
            return require(`${fileDirName}/${path}`);
        }
        else {
            return require(path);
        }
    };
    const loadPdf = async (path) => {
        const loadingTask = pdfJs.getDocument({
            url: path,
            // https://github.com/mozilla/pdf.js/issues/4244#issuecomment-1479534301
            useSystemFonts: true,
            // IMPORTANT: Set to false to mitigate CVE-2024-4367.
            isEvalSupported: false,
        });
        return await loadingTask.promise;
    };
    shim_1.default.pdfExtractEmbeddedText = async (pdfPath) => {
        const doc = await loadPdf(pdfPath);
        const textByPage = [];
        try {
            for (let pageNum = 1; pageNum <= doc.numPages; pageNum++) {
                const page = await doc.getPage(pageNum);
                const textContent = await page.getTextContent();
                const strings = textContent.items.map(item => {
                    var _a;
                    const text = (_a = item.str) !== null && _a !== void 0 ? _a : '';
                    return text;
                }).join('\n');
                // Some PDFs contain unsupported characters that can lead to hard-to-debug issues.
                // We remove them here.
                textByPage.push((0, replaceUnsupportedCharacters_1.default)(strings));
            }
        }
        finally {
            await doc.destroy();
        }
        return textByPage;
    };
    shim_1.default.pdfToImages = async (pdfPath, outputDirectoryPath, options) => {
        var _a, _b, _c;
        if (typeof HTMLCanvasElement === 'undefined') {
            throw new Error('Unsupported -- the Canvas element is required.');
        }
        const createCanvas = () => {
            return document.createElement('canvas');
        };
        const canvasToBuffer = async (canvas) => {
            const quality = 0.8;
            const canvasToBlob = async (canvas) => {
                return new Promise(resolve => {
                    canvas.toBlob(blob => resolve(blob), 'image/jpg', quality);
                });
            };
            const blob = await canvasToBlob(canvas);
            return Buffer.from(await blob.arrayBuffer());
        };
        const filePrefix = `page_${Date.now()}`;
        const output = [];
        const doc = await loadPdf(pdfPath);
        try {
            const startPage = (_a = options === null || options === void 0 ? void 0 : options.minPage) !== null && _a !== void 0 ? _a : 1;
            const endPage = Math.min(doc.numPages, (_b = options === null || options === void 0 ? void 0 : options.maxPage) !== null && _b !== void 0 ? _b : doc.numPages);
            for (let pageNum = startPage; pageNum <= endPage; pageNum++) {
                const page = await doc.getPage(pageNum);
                const viewport = page.getViewport({ scale: (_c = options === null || options === void 0 ? void 0 : options.scaleFactor) !== null && _c !== void 0 ? _c : 2 });
                const canvas = createCanvas();
                const ctx = canvas.getContext('2d');
                if (!ctx) {
                    throw new Error('Unable to get 2D rendering context from canvas.');
                }
                canvas.height = viewport.height;
                canvas.width = viewport.width;
                const renderTask = page.render({ canvasContext: ctx, viewport: viewport });
                await renderTask.promise;
                const buffer = await canvasToBuffer(canvas);
                const filePath = `${outputDirectoryPath}/${filePrefix}_${pageNum.toString().padStart(4, '0')}.jpg`;
                output.push(filePath);
                await (0, promises_1.writeFile)(filePath, buffer, 'binary');
                if (!(await shim_1.default.fsDriver().exists(filePath)))
                    throw new Error(`Could not write to file: ${filePath}`);
            }
        }
        finally {
            await doc.destroy();
        }
        return output;
    };
    shim_1.default.pdfInfo = async (pdfPath) => {
        const doc = await loadPdf(pdfPath);
        return { pageCount: doc.numPages };
    };
}
module.exports = { shimInit, setupProxySettings };
//# sourceMappingURL=shim-init-node.js.map