"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.default = versionInfo;
const Logger_1 = require("@joplin/utils/Logger");
const locale_1 = require("./locale");
const Setting_1 = require("./models/Setting");
const registry_1 = require("./registry");
const KeychainService_1 = require("./services/keychain/KeychainService");
const shim_1 = require("./shim");
const logger = Logger_1.default.create('versionInfo');
function getPluginLists(plugins) {
    const pluginList = [];
    if (Object.keys(plugins).length > 0) {
        for (const pluginId in plugins) {
            pluginList.push(`${plugins[pluginId].manifest.name}: ${plugins[pluginId].manifest.version}`);
        }
    }
    pluginList.sort(Intl.Collator().compare);
    let completeList = '';
    let summary = '';
    if (pluginList.length > 0) {
        completeList = ['\n', ...pluginList].join('\n');
        if (pluginList.length > 20) {
            summary = [
                '\n',
                ...[...pluginList].filter((_, index) => index < 20),
                '...',
            ].join('\n');
        }
        else {
            summary = completeList;
        }
    }
    return {
        completeList,
        summary,
    };
}
function versionInfo(packageInfo, plugins) {
    const p = packageInfo;
    let gitInfo = '';
    if ('git' in p) {
        gitInfo = (0, locale_1._)('Revision: %s (%s)', p.git.hash, p.git.branch);
        if (p.git.branch === 'HEAD')
            gitInfo = gitInfo.slice(0, -7);
    }
    const copyrightText = 'Copyright © 2016-YYYY Laurent Cozic';
    const now = new Date();
    const header = [
        p.description,
        '',
        copyrightText.replace('YYYY', `${now.getFullYear()}`),
    ];
    let keychainSupported = false;
    try {
        // To allow old keys to be read, certain apps allow read-only keychain access:
        keychainSupported = Setting_1.default.value('keychain.supported') >= 1 && !KeychainService_1.default.instance().readOnly;
    }
    catch (error) {
        logger.error('Failed to determine if keychain is supported', error);
    }
    const body = [
        (0, locale_1._)('%s %s (%s, %s)', p.name, p.version, Setting_1.default.value('env'), shim_1.default.platformName()),
        '',
        (0, locale_1._)('Device: %s', shim_1.default.deviceString()),
        (0, locale_1._)('Client ID: %s', Setting_1.default.value('clientId')),
        (0, locale_1._)('Sync Version: %s', Setting_1.default.value('syncVersion')),
        (0, locale_1._)('Profile Version: %s', registry_1.reg.db().version()),
        (0, locale_1._)('Keychain Supported: %s', keychainSupported ? (0, locale_1._)('Yes') : (0, locale_1._)('No')),
        (0, locale_1._)('Alternative instance ID: %s', Setting_1.default.value('altInstanceId') || '-'),
    ];
    if (gitInfo) {
        body.push(`\n${gitInfo}`);
    }
    const pluginList = getPluginLists(plugins);
    return {
        header: header.join('\n'),
        body: body.join('\n').concat(pluginList.completeList),
        message: header.concat(body).join('\n').concat(pluginList.summary),
    };
}
//# sourceMappingURL=versionInfo.js.map