"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const fs = require("fs-extra");
const Logger_1 = require("@joplin/utils/Logger");
const path_utils_1 = require("@joplin/lib/path-utils");
const { DatabaseDriverNode } = require('@joplin/lib/database-driver-node.js');
const JoplinDatabase_1 = require("@joplin/lib/JoplinDatabase");
const BaseModel_1 = require("@joplin/lib/BaseModel");
const Folder_1 = require("@joplin/lib/models/Folder");
const Note_1 = require("@joplin/lib/models/Note");
const Setting_1 = require("@joplin/lib/models/Setting");
const execa_1 = require("execa");
const utils_1 = require("@joplin/utils");
const nodeSqlite = require('sqlite3');
const { loadKeychainServiceAndSettings } = require('@joplin/lib/services/SettingUtils');
const { default: shimInitCli } = require('./utils/shimInitCli');
const baseDir = `${(0, path_utils_1.dirname)(__dirname)}/tests/cli-integration`;
const joplinAppPath = `${__dirname}/main.js`;
shimInitCli({ nodeSqlite, appVersion: () => require('../package.json').version, keytar: null });
require('@joplin/lib/testing/test-utils');
function createClient(id) {
    return {
        id: id,
        profileDir: `${baseDir}/client${id}`,
    };
}
async function execCommand(client, command) {
    const result = await (0, execa_1.node)(joplinAppPath, ['--update-geolocation-disabled', '--env', 'dev', '--profile', client.profileDir, ...(0, utils_1.splitCommandString)(command)]);
    if (result.exitCode !== 0) {
        throw new Error(`Command failed: ${command}:\nstderr: ${result.stderr}\nstdout: ${result.stdout}`);
    }
    return result.stdout;
}
async function clearDatabase(db) {
    await db.transactionExecBatch(['DELETE FROM folders', 'DELETE FROM notes', 'DELETE FROM tags', 'DELETE FROM note_tags', 'DELETE FROM resources', 'DELETE FROM deleted_items']);
}
describe('cli-integration-tests', () => {
    let client;
    let db;
    beforeAll(async () => {
        await fs.remove(baseDir);
        await fs.mkdir(baseDir);
        client = createClient(1);
        // Initialize the database by running a client command and exiting.
        await execCommand(client, 'version');
        const dbLogger = new Logger_1.default();
        dbLogger.addTarget(Logger_1.TargetType.Console);
        dbLogger.setLevel(Logger_1.default.LEVEL_WARN);
        db = new JoplinDatabase_1.default(new DatabaseDriverNode());
        db.setLogger(dbLogger);
        await db.open({ name: `${client.profileDir}/database.sqlite` });
        BaseModel_1.default.setDb(db);
        Setting_1.default.setConstant('rootProfileDir', client.profileDir);
        Setting_1.default.setConstant('profileDir', client.profileDir);
        await loadKeychainServiceAndSettings([]);
    });
    beforeEach(async () => {
        await clearDatabase(db);
    });
    it.each([
        'version',
        'help',
    ])('should run command %j without crashing', async (command) => {
        await execCommand(client, command);
    });
    it('should support the \'ls\' command', async () => {
        await execCommand(client, 'mkbook nb1');
        await execCommand(client, 'mknote note1');
        await execCommand(client, 'mknote note2');
        const r = await execCommand(client, 'ls');
        expect(r.indexOf('note1') >= 0).toBe(true);
        expect(r.indexOf('note2') >= 0).toBe(true);
    });
    it('should support the \'mv\' command', async () => {
        await execCommand(client, 'mkbook nb2');
        await execCommand(client, 'mkbook nb1');
        await execCommand(client, 'mknote n1');
        await execCommand(client, 'mv n1 nb2');
        const f1 = await Folder_1.default.loadByTitle('nb1');
        const f2 = await Folder_1.default.loadByTitle('nb2');
        let notes1 = await Note_1.default.previews(f1.id);
        let notes2 = await Note_1.default.previews(f2.id);
        expect(notes1.length).toBe(0);
        expect(notes2.length).toBe(1);
        await execCommand(client, 'mknote note1');
        await execCommand(client, 'mknote note2');
        await execCommand(client, 'mknote note3');
        await execCommand(client, 'mknote blabla');
        notes1 = await Note_1.default.previews(f1.id);
        notes2 = await Note_1.default.previews(f2.id);
        expect(notes1.length).toBe(4);
        expect(notes2.length).toBe(1);
        await execCommand(client, 'mv \'note*\' nb2');
        notes2 = await Note_1.default.previews(f2.id);
        notes1 = await Note_1.default.previews(f1.id);
        expect(notes1.length).toBe(1);
        expect(notes2.length).toBe(4);
    });
    it('should support the \'use\' command', async () => {
        await execCommand(client, 'mkbook nb1');
        await execCommand(client, 'mkbook nb2');
        await execCommand(client, 'mknote n1');
        await execCommand(client, 'mknote n2');
        const f1 = await Folder_1.default.loadByTitle('nb1');
        const f2 = await Folder_1.default.loadByTitle('nb2');
        let notes1 = await Note_1.default.previews(f1.id);
        let notes2 = await Note_1.default.previews(f2.id);
        expect(notes1.length).toBe(0);
        expect(notes2.length).toBe(2);
        await execCommand(client, 'use nb1');
        await execCommand(client, 'mknote note2');
        await execCommand(client, 'mknote note3');
        notes1 = await Note_1.default.previews(f1.id);
        notes2 = await Note_1.default.previews(f2.id);
        expect(notes1.length).toBe(2);
    });
    it('should support creating and removing folders', async () => {
        await execCommand(client, 'mkbook nb1');
        let folders = await Folder_1.default.all();
        expect(folders.length).toBe(1);
        expect(folders[0].title).toBe('nb1');
        await execCommand(client, 'mkbook nb1');
        folders = await Folder_1.default.all();
        expect(folders.length).toBe(2);
        expect(folders[0].title).toBe('nb1');
        expect(folders[1].title).toBe('nb1');
        await execCommand(client, 'rmbook -p -f nb1');
        folders = await Folder_1.default.all();
        expect(folders.length).toBe(1);
        await execCommand(client, 'rmbook -p -f nb1');
        folders = await Folder_1.default.all();
        expect(folders.length).toBe(0);
    });
    it('should support creating and removing notes', async () => {
        await execCommand(client, 'mkbook nb1');
        await execCommand(client, 'mknote n1');
        let notes = await Note_1.default.all();
        expect(notes.length).toBe(1);
        expect(notes[0].title).toBe('n1');
        await execCommand(client, 'rmnote -p -f n1');
        notes = await Note_1.default.all();
        expect(notes.length).toBe(0);
        await execCommand(client, 'mknote n1');
        await execCommand(client, 'mknote n2');
        notes = await Note_1.default.all();
        expect(notes.length).toBe(2);
        // Should fail to delete a non-existent note
        let failed = false;
        try {
            await execCommand(client, 'rmnote -f \'blabla*\'');
        }
        catch (error) {
            failed = true;
        }
        expect(failed).toBe(true);
        notes = await Note_1.default.all();
        expect(notes.length).toBe(2);
        await execCommand(client, 'rmnote -f -p \'n*\'');
        notes = await Note_1.default.all();
        expect(notes.length).toBe(0);
    });
    it('should support listing the contents of notes', async () => {
        await execCommand(client, 'mkbook nb1');
        await execCommand(client, 'mknote mynote');
        const folder = await Folder_1.default.loadByTitle('nb1');
        const note = await Note_1.default.loadFolderNoteByField(folder.id, 'title', 'mynote');
        let r = await execCommand(client, 'cat mynote');
        expect(r).toContain('mynote');
        expect(r).not.toContain(note.id);
        r = await execCommand(client, 'cat -v mynote');
        expect(r).toContain(note.id);
    });
    it('should support changing settings with config', async () => {
        await execCommand(client, 'config editor vim');
        await Setting_1.default.reset();
        await Setting_1.default.load();
        expect(Setting_1.default.value('editor')).toBe('vim');
        await execCommand(client, 'config editor subl');
        await Setting_1.default.reset();
        await Setting_1.default.load();
        expect(Setting_1.default.value('editor')).toBe('subl');
        const r = await execCommand(client, 'config');
        expect(r.indexOf('editor') >= 0).toBe(true);
        expect(r.indexOf('subl') >= 0).toBe(true);
    });
    it('should support copying folders with cp', async () => {
        await execCommand(client, 'mkbook nb2');
        await execCommand(client, 'mkbook nb1');
        await execCommand(client, 'mknote n1');
        await execCommand(client, 'cp n1');
        const f1 = await Folder_1.default.loadByTitle('nb1');
        const f2 = await Folder_1.default.loadByTitle('nb2');
        let notes = await Note_1.default.previews(f1.id);
        expect(notes.length).toBe(2);
        await execCommand(client, 'cp n1 nb2');
        const notesF1 = await Note_1.default.previews(f1.id);
        expect(notesF1.length).toBe(2);
        notes = await Note_1.default.previews(f2.id);
        expect(notes.length).toBe(1);
        expect(notes[0].title).toBe(notesF1[0].title);
    });
});
//# sourceMappingURL=cli-integration-tests.test.js.map