"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const base_command_1 = require("./base-command");
const InteropService_1 = require("@joplin/lib/services/interop/InteropService");
const BaseModel_1 = require("@joplin/lib/BaseModel");
const { cliUtils } = require('./cli-utils.js');
const app_1 = require("./app");
const locale_1 = require("@joplin/lib/locale");
const array_1 = require("@joplin/lib/array");
const Folder_1 = require("@joplin/lib/models/Folder");
class Command extends base_command_1.default {
    usage() {
        return 'import <path> [notebook]';
    }
    description() {
        return (0, locale_1._)('Imports data into Joplin.');
    }
    options() {
        const service = InteropService_1.default.instance();
        const formats = service
            .modules()
            .filter(m => m.type === 'importer')
            .map(m => m.format);
        return [
            ['--format <format>', (0, locale_1._)('Source format: %s', ['auto'].concat((0, array_1.unique)(formats)).join(', '))],
            ['-f, --force', (0, locale_1._)('Do not ask for confirmation.')],
            ['--output-format <output-format>', (0, locale_1._)('Output format: %s', 'md, html')],
        ];
    }
    // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
    async action(args) {
        let destinationFolder = await (0, app_1.default)().loadItem(BaseModel_1.default.TYPE_FOLDER, args.notebook);
        if (args.notebook && !destinationFolder)
            throw new Error((0, locale_1._)('Cannot find "%s".', args.notebook));
        if (!destinationFolder)
            destinationFolder = await Folder_1.default.defaultFolder();
        const importOptions = {};
        importOptions.path = args.path;
        importOptions.format = args.options.format ? args.options.format : 'auto';
        importOptions.destinationFolderId = destinationFolder ? destinationFolder.id : null;
        let lastProgress = '';
        // onProgress/onError supported by Enex import only
        importOptions.onProgress = progressState => {
            const line = [];
            line.push((0, locale_1._)('Found: %d.', progressState.loaded));
            line.push((0, locale_1._)('Created: %d.', progressState.created));
            if (progressState.updated)
                line.push((0, locale_1._)('Updated: %d.', progressState.updated));
            if (progressState.skipped)
                line.push((0, locale_1._)('Skipped: %d.', progressState.skipped));
            if (progressState.resourcesCreated)
                line.push((0, locale_1._)('Resources: %d.', progressState.resourcesCreated));
            if (progressState.notesTagged)
                line.push((0, locale_1._)('Tagged: %d.', progressState.notesTagged));
            lastProgress = line.join(' ');
            cliUtils.redraw(lastProgress);
        };
        importOptions.onError = error => {
            const s = error.stack ? error.stack : error.toString();
            this.stdout(s);
        };
        if (args.options.outputFormat)
            importOptions.outputFormat = args.options.outputFormat;
        (0, app_1.default)().gui().showConsole();
        this.stdout((0, locale_1._)('Importing notes...'));
        const service = InteropService_1.default.instance();
        const result = await service.import(importOptions);
        result.warnings.map(w => this.stdout(w));
        cliUtils.redrawDone();
        if (lastProgress)
            this.stdout((0, locale_1._)('The notes have been imported: %s', lastProgress));
    }
}
module.exports = Command;
//# sourceMappingURL=command-import.js.map