"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.runtime = exports.declaration = void 0;
const locale_1 = require("../locale");
const Note_1 = require("../models/Note");
const renderer_1 = require("@joplin/renderer");
const convertHtmlToMarkdown_1 = require("./convertHtmlToMarkdown");
const shim_1 = require("../shim");
const readOnly_1 = require("../models/utils/readOnly");
const BaseModel_1 = require("../BaseModel");
const ItemChange_1 = require("../models/ItemChange");
const Setting_1 = require("../models/Setting");
const Logger_1 = require("@joplin/utils/Logger");
const logger = Logger_1.default.create('convertNoteToMarkdown');
exports.declaration = {
    name: 'convertNoteToMarkdown',
    label: () => (0, locale_1._)('Convert to Markdown'),
};
const runtime = () => {
    return {
        execute: async (context, noteIds = []) => {
            if (typeof noteIds === 'string') {
                noteIds = [noteIds];
            }
            if (noteIds.length === 0) {
                noteIds = context.state.selectedNoteIds;
            }
            const notes = await Note_1.default.loadItemsByIdsOrFail(noteIds);
            try {
                let isFirst = true;
                let processedCount = 0;
                for (const note of notes) {
                    if (note.markup_language === renderer_1.MarkupLanguage.Markdown) {
                        logger.warn('Skipping item: Already Markdown.');
                        continue;
                    }
                    if (await (0, readOnly_1.itemIsReadOnly)(Note_1.default, BaseModel_1.ModelType.Note, ItemChange_1.default.SOURCE_UNSPECIFIED, note.id, Setting_1.default.value('sync.userId'), context.state.shareService)) {
                        throw new Error((0, locale_1._)('Cannot convert read-only item: "%s"', note.title));
                    }
                    const markdownBody = await (0, convertHtmlToMarkdown_1.runtime)().execute(context, note.body);
                    const newNote = await Note_1.default.duplicate(note.id);
                    newNote.body = markdownBody;
                    newNote.markup_language = renderer_1.MarkupLanguage.Markdown;
                    await Note_1.default.save(newNote);
                    await Note_1.default.delete(note.id, { toTrash: true });
                    processedCount++;
                    if (isFirst) {
                        context.dispatch({
                            type: 'NOTE_SELECT',
                            id: newNote.id,
                        });
                        isFirst = false;
                    }
                }
                void shim_1.default.showToast((0, locale_1._n)('The note has been converted to Markdown and the original note has been moved to the trash', 'The notes have been converted to Markdown and the original notes have been moved to the trash', processedCount), { type: shim_1.ToastType.Success });
            }
            catch (error) {
                await shim_1.default.showErrorDialog((0, locale_1._)('Could not convert notes to Markdown: %s', error.message));
            }
        },
        enabledCondition: 'selectionIncludesHtmlNotes && (multipleNotesSelected || !noteIsReadOnly)',
    };
};
exports.runtime = runtime;
//# sourceMappingURL=convertNoteToMarkdown.js.map