"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.settingsToComponents2 = exports.settingsSections = exports.settingsToComponents = exports.saveSettings = exports.scheduleSaveSettings = exports.updateSettingValue = exports.checkSyncConfigMessages = exports.checkSyncConfig = exports.advancedSettingsButton_click = exports.init = exports.defaultScreenState = void 0;
const Setting_1 = require("../../../models/Setting");
const SyncTargetRegistry_1 = require("../../../SyncTargetRegistry");
const { _ } = require('../../../locale');
const reselect_1 = require("reselect");
const Logger_1 = require("@joplin/utils/Logger");
const settingValidations_1 = require("../../../models/settings/settingValidations");
const ObjectUtils_1 = require("../../../ObjectUtils");
const logger = Logger_1.default.create('config-shared');
exports.defaultScreenState = {
    checkSyncConfigResult: null,
    settings: {},
    changedSettingKeys: [],
    showAdvancedSettings: false,
};
let onSettingsSaved = () => { };
const init = (reg) => {
    onSettingsSaved = (event) => {
        const savedSettingKeys = event.savedSettingKeys;
        // After changing the sync settings we immediately trigger a sync
        // operation. This will ensure that the client gets the sync info as
        // early as possible, in particular the encryption state (encryption
        // keys, whether it's enabled, etc.). This should prevent situations
        // where the user tried to setup E2EE on the client even though it's
        // already been done on another client.
        if (savedSettingKeys.find(s => s.startsWith('sync.'))) {
            logger.info('Sync settings have been changed - scheduling a sync');
            void reg.scheduleSync();
        }
    };
};
exports.init = init;
const advancedSettingsButton_click = (comp) => {
    comp.setState((state) => {
        return { showAdvancedSettings: !state.showAdvancedSettings };
    });
};
exports.advancedSettingsButton_click = advancedSettingsButton_click;
// eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
const checkSyncConfig = async (comp, settings) => {
    const syncTargetId = settings['sync.target'];
    const SyncTargetClass = SyncTargetRegistry_1.default.classById(syncTargetId);
    const options = Object.assign(Object.assign({}, Setting_1.default.subValues(`sync.${syncTargetId}`, settings, { includeConstants: true })), Setting_1.default.subValues('net', settings, { includeConstants: true }));
    comp.setState({ checkSyncConfigResult: 'checking' });
    const result = await SyncTargetClass.checkConfig((0, ObjectUtils_1.convertValuesToFunctions)(options));
    comp.setState({ checkSyncConfigResult: result });
    if (result.ok) {
        // Users often expect config to be auto-saved at this point, if the config check was successful
        await (0, exports.saveSettings)(comp);
    }
    return result;
};
exports.checkSyncConfig = checkSyncConfig;
const checkSyncConfigMessages = (comp) => {
    const result = comp.state.checkSyncConfigResult;
    const output = [];
    if (result === 'checking') {
        output.push(_('Checking... Please wait.'));
    }
    else if (result && result.ok) {
        output.push(_('Success! Synchronisation configuration appears to be correct.'));
    }
    else if (result && !result.ok) {
        output.push(_('Error. Please check that URL, username, password, etc. are correct and that the sync target is accessible. The reported error was:'));
        output.push(result.errorMessage);
    }
    return output;
};
exports.checkSyncConfigMessages = checkSyncConfigMessages;
// eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
const updateSettingValue = (comp, key, value, callback) => {
    if (!callback)
        callback = () => { };
    comp.setState((state) => {
        // @react-native-community/slider (4.4.0) will emit a valueChanged event
        // when the component is mounted, even though the value hasn't changed.
        // We should ignore this, otherwise it will mark the settings as
        // unsaved.
        //
        // Upstream: https://github.com/callstack/react-native-slider/issues/395
        //
        // https://github.com/laurent22/joplin/issues/7503
        if (state.settings[key] === value) {
            logger.info('Trying to update a setting that has not changed - skipping it.', key, value);
            return {};
        }
        const settings = Object.assign({}, state.settings);
        const changedSettingKeys = state.changedSettingKeys.slice();
        settings[key] = Setting_1.default.formatValue(key, value);
        if (changedSettingKeys.indexOf(key) < 0)
            changedSettingKeys.push(key);
        return {
            settings: settings,
            changedSettingKeys: changedSettingKeys,
        };
    }, callback);
    const metadata = Setting_1.default.settingMetadata(key);
    if (metadata.autoSave) {
        (0, exports.scheduleSaveSettings)(comp);
    }
};
exports.updateSettingValue = updateSettingValue;
let scheduleSaveSettingsIID = null;
const scheduleSaveSettings = (comp) => {
    if (scheduleSaveSettingsIID)
        clearTimeout(scheduleSaveSettingsIID);
    scheduleSaveSettingsIID = setTimeout(async () => {
        scheduleSaveSettingsIID = null;
        await (0, exports.saveSettings)(comp);
    }, 100);
};
exports.scheduleSaveSettings = scheduleSaveSettings;
const saveSettings = async (comp) => {
    const savedSettingKeys = comp.state.changedSettingKeys.slice();
    const validationMessage = await (0, settingValidations_1.default)(savedSettingKeys, comp.state.settings);
    if (validationMessage) {
        alert(validationMessage);
        return false;
    }
    for (const key in comp.state.settings) {
        if (!comp.state.settings.hasOwnProperty(key))
            continue;
        if (comp.state.changedSettingKeys.indexOf(key) < 0)
            continue;
        Setting_1.default.setValue(key, comp.state.settings[key]);
    }
    comp.setState({ changedSettingKeys: [] });
    onSettingsSaved({ savedSettingKeys });
    return true;
};
exports.saveSettings = saveSettings;
// eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
const settingsToComponents = (comp, device, settings) => {
    const keys = Setting_1.default.keys(true, device);
    const settingComps = [];
    for (let i = 0; i < keys.length; i++) {
        const key = keys[i];
        if (!Setting_1.default.isPublic(key))
            continue;
        const md = Setting_1.default.settingMetadata(key);
        if (md.show && !md.show(settings))
            continue;
        const settingComp = comp.settingToComponent(key, settings[key]);
        if (!settingComp)
            continue;
        settingComps.push(settingComp);
    }
    return settingComps;
};
exports.settingsToComponents = settingsToComponents;
const deviceSelector = (state) => state.device;
const settingsSelector = (state) => state.settings;
exports.settingsSections = (0, reselect_1.createSelector)(deviceSelector, settingsSelector, (device, settings) => {
    const keys = Setting_1.default.keys(true, device);
    const metadatas = [];
    for (let i = 0; i < keys.length; i++) {
        const key = keys[i];
        if (!Setting_1.default.isPublic(key))
            continue;
        const md = Setting_1.default.settingMetadata(key);
        if (md.show && !md.show(settings))
            continue;
        metadatas.push(md);
    }
    const output = Setting_1.default.groupMetadatasBySections(metadatas);
    if (device === Setting_1.AppType.Desktop || device === Setting_1.AppType.Cli) {
        output.push({
            name: 'encryption',
            metadatas: [],
            isScreen: true,
        });
        output.push({
            name: 'server',
            metadatas: [],
            isScreen: true,
        });
        output.push({
            name: 'keymap',
            metadatas: [],
            isScreen: true,
        });
    }
    else {
        output.push(...([
            'tools', 'importOrExport', 'moreInfo',
        ].map((name) => {
            return {
                name,
                metadatas: [],
            };
        })));
    }
    // Ideally we would also check if the user was able to synchronize
    // but we don't have a way of doing that besides making a request to Joplin Cloud
    const syncTargetIsJoplinCloud = settings['sync.target'] === SyncTargetRegistry_1.default.nameToId('joplinCloud');
    if (syncTargetIsJoplinCloud) {
        output.push({
            name: 'joplinCloud',
            metadatas: [],
            isScreen: true,
        });
    }
    const order = Setting_1.default.sectionOrder();
    const sortOrderFor = (section) => {
        if (section.source === Setting_1.SettingSectionSource.Plugin) {
            // Plugins should go after all other sections
            return order.length + 1;
        }
        return order.indexOf(section.name);
    };
    output.sort((a, b) => {
        const o1 = sortOrderFor(a);
        const o2 = sortOrderFor(b);
        if (o1 === o2) {
            const l1 = Setting_1.default.sectionNameToLabel(a.name);
            const l2 = Setting_1.default.sectionNameToLabel(b.name);
            return l1.toLowerCase() < l2.toLowerCase() ? -1 : +1;
        }
        return o1 - o2;
    });
    return output;
});
const settingsToComponents2 = (
// eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
comp, device, settings, selectedSectionName = '') => {
    const sectionComps = [];
    const sections = (0, exports.settingsSections)({ device, settings });
    for (let i = 0; i < sections.length; i++) {
        const section = sections[i];
        const sectionComp = comp.sectionToComponent(section.name, section, settings, selectedSectionName === section.name);
        if (!sectionComp)
            continue;
        sectionComps.push(sectionComp);
    }
    return sectionComps;
};
exports.settingsToComponents2 = settingsToComponents2;
//# sourceMappingURL=config-shared.js.map