"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const locale_1 = require("../../../../locale");
const PluginService_1 = require("../../../../services/plugins/PluginService");
const shim_1 = require("../../../../shim");
const immer_1 = require("immer");
const useOnDeleteHandler = (pluginSettingsRef, onSettingsChange, deleteNow) => {
    const React = shim_1.default.react();
    return React.useCallback(async (event) => {
        const item = event.item;
        const confirmed = await shim_1.default.showConfirmationDialog((0, locale_1._)('Delete plugin "%s"?', item.manifest.name));
        if (!confirmed)
            return;
        if (deleteNow) {
            const pluginService = PluginService_1.default.instance();
            // We first unload the plugin. This is done here rather than in pluginService.uninstallPlugins
            // because unloadPlugin may not work on desktop.
            const plugin = pluginService.plugins[item.manifest.id];
            if (plugin) {
                await pluginService.unloadPlugin(item.manifest.id);
            }
        }
        // Important: To avoid race conditions, don't run any async code between fetching plugin
        // settings from the ref and calling onSettingsChange.
        let newSettings = pluginSettingsRef.current;
        if (deleteNow) {
            newSettings = (0, immer_1.produce)(newSettings, (draft) => {
                delete draft[item.manifest.id];
            });
            onSettingsChange({ value: newSettings });
            await PluginService_1.default.instance().uninstallPlugin(item.manifest.id);
        }
        else {
            // Setting .deleted causes the app to delete this plugin on startup.
            newSettings = (0, immer_1.produce)(newSettings, (draft) => {
                if (!draft[item.manifest.id])
                    draft[item.manifest.id] = (0, PluginService_1.defaultPluginSetting)();
                draft[item.manifest.id].deleted = true;
            });
            onSettingsChange({ value: newSettings });
        }
    }, [pluginSettingsRef, onSettingsChange, deleteNow]);
};
exports.default = useOnDeleteHandler;
//# sourceMappingURL=useOnDeleteHandler.js.map