"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const test_utils_1 = require("../testing/test-utils");
const Folder_1 = require("../models/Folder");
const Note_1 = require("../models/Note");
const Resource_1 = require("../models/Resource");
const shim_1 = require("../shim");
const errors_1 = require("../errors");
const fs_extra_1 = require("fs-extra");
const types_1 = require("../services/database/types");
const testImagePath = `${test_utils_1.supportDir}/photo.jpg`;
const setupFolderNoteResourceReadOnly = async (shareId) => {
    const cleanup = (0, test_utils_1.simulateReadOnlyShareEnv)(shareId);
    let folder = await Folder_1.default.save({});
    let note = await Note_1.default.save({ parent_id: folder.id });
    await shim_1.default.attachFileToNote(note, testImagePath);
    let resource = (await Resource_1.default.all())[0];
    folder = await Folder_1.default.save({ id: folder.id, share_id: shareId });
    note = await Note_1.default.save({ id: note.id, share_id: shareId });
    resource = await Resource_1.default.save({ id: resource.id, share_id: shareId });
    resource = await Resource_1.default.load(resource.id); // reload to get all properties
    return { cleanup, folder, note, resource };
};
describe('models/Resource', () => {
    beforeEach(async () => {
        await (0, test_utils_1.setupDatabaseAndSynchronizer)(1);
        await (0, test_utils_1.switchClient)(1);
    });
    it('should have a "done" fetch_status when created locally', (async () => {
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        const note1 = await Note_1.default.save({ title: 'ma note', parent_id: folder1.id });
        await shim_1.default.attachFileToNote(note1, testImagePath);
        const resource1 = (await Resource_1.default.all())[0];
        const ls = await Resource_1.default.localState(resource1);
        expect(ls.fetch_status).toBe(Resource_1.default.FETCH_STATUS_DONE);
    }));
    it('should have a default local state', (async () => {
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        const note1 = await Note_1.default.save({ title: 'ma note', parent_id: folder1.id });
        await shim_1.default.attachFileToNote(note1, testImagePath);
        const resource1 = (await Resource_1.default.all())[0];
        const ls = await Resource_1.default.localState(resource1);
        expect(!ls.id).toBe(true);
        expect(ls.resource_id).toBe(resource1.id);
        expect(ls.fetch_status).toBe(Resource_1.default.FETCH_STATUS_DONE);
    }));
    it('should save and delete local state', (async () => {
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        const note1 = await Note_1.default.save({ title: 'ma note', parent_id: folder1.id });
        await shim_1.default.attachFileToNote(note1, testImagePath);
        const resource1 = (await Resource_1.default.all())[0];
        await Resource_1.default.setLocalState(resource1, { fetch_status: Resource_1.default.FETCH_STATUS_IDLE });
        let ls = await Resource_1.default.localState(resource1);
        expect(!!ls.id).toBe(true);
        expect(ls.fetch_status).toBe(Resource_1.default.FETCH_STATUS_IDLE);
        await Resource_1.default.delete(resource1.id);
        ls = await Resource_1.default.localState(resource1);
        expect(!ls.id).toBe(true);
    }));
    it('should resize the resource if the image is below the required dimensions', (async () => {
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        const note1 = await Note_1.default.save({ title: 'ma note', parent_id: folder1.id });
        const previousMax = Resource_1.default.IMAGE_MAX_DIMENSION;
        Resource_1.default.IMAGE_MAX_DIMENSION = 5;
        await shim_1.default.attachFileToNote(note1, testImagePath);
        Resource_1.default.IMAGE_MAX_DIMENSION = previousMax;
        const resource1 = (await Resource_1.default.all())[0];
        const originalStat = await shim_1.default.fsDriver().stat(testImagePath);
        const newStat = await shim_1.default.fsDriver().stat(Resource_1.default.fullPath(resource1));
        expect(newStat.size < originalStat.size).toBe(true);
    }));
    it('should not resize the resource if the image is below the required dimensions', (async () => {
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        const note1 = await Note_1.default.save({ title: 'ma note', parent_id: folder1.id });
        await shim_1.default.attachFileToNote(note1, testImagePath);
        const resource1 = (await Resource_1.default.all())[0];
        const originalStat = await shim_1.default.fsDriver().stat(testImagePath);
        const newStat = await shim_1.default.fsDriver().stat(Resource_1.default.fullPath(resource1));
        expect(originalStat.size).toBe(newStat.size);
    }));
    it('should set the blob_updated_time property if the blob is updated', (async () => {
        const note = await Note_1.default.save({});
        await shim_1.default.attachFileToNote(note, testImagePath);
        const resourceA = (await Resource_1.default.all())[0];
        expect(resourceA.updated_time).toBe(resourceA.blob_updated_time);
        await (0, test_utils_1.msleep)(1);
        await Resource_1.default.updateResourceBlobContent(resourceA.id, testImagePath);
        const resourceB = (await Resource_1.default.all())[0];
        expect(resourceB.updated_time).toBeGreaterThan(resourceA.updated_time);
        expect(resourceB.blob_updated_time).toBeGreaterThan(resourceA.blob_updated_time);
    }));
    it('should NOT set the blob_updated_time property if the blob is NOT updated', (async () => {
        const note = await Note_1.default.save({});
        await shim_1.default.attachFileToNote(note, testImagePath);
        const resourceA = (await Resource_1.default.all())[0];
        await (0, test_utils_1.msleep)(1);
        // We only update the resource metadata - so the blob timestamp should
        // not change
        await Resource_1.default.save({ id: resourceA.id, title: 'new title' });
        const resourceB = (await Resource_1.default.all())[0];
        expect(resourceB.updated_time).toBeGreaterThan(resourceA.updated_time);
        expect(resourceB.blob_updated_time).toBe(resourceA.blob_updated_time);
    }));
    it('should not allow modifying a read-only resource', async () => {
        const { cleanup, resource } = await setupFolderNoteResourceReadOnly('123456789');
        await (0, test_utils_1.expectThrow)(async () => Resource_1.default.save({ id: resource.id, share_id: '123456789', title: 'cannot do this!' }), errors_1.ErrorCode.IsReadOnly);
        cleanup();
    });
    it('should not allow modifying a read-only resource content', async () => {
        const { cleanup, resource } = await setupFolderNoteResourceReadOnly('123456789');
        const tempFilePath = await (0, test_utils_1.createTempFile)('something');
        await (0, test_utils_1.expectThrow)(async () => Resource_1.default.updateResourceBlobContent(resource.id, tempFilePath), errors_1.ErrorCode.IsReadOnly);
        await (0, fs_extra_1.remove)(tempFilePath);
        cleanup();
    });
    it('should not allow deleting a read-only resource', async () => {
        const { cleanup, resource } = await setupFolderNoteResourceReadOnly('123456789');
        expect(await (0, fs_extra_1.pathExists)(Resource_1.default.fullPath(resource))).toBe(true);
        await (0, test_utils_1.expectThrow)(async () => Resource_1.default.delete(resource.id), errors_1.ErrorCode.IsReadOnly);
        // Also check that the resource blob has not been deleted
        expect(await (0, fs_extra_1.pathExists)(Resource_1.default.fullPath(resource))).toBe(true);
        cleanup();
    });
    it('should return resources since a certain time and ID', async () => {
        expect((await Resource_1.default.allForNormalization(0, '')).length).toBe(0);
        const testData = [
            ['00000000000000000000000000000001', 1536700000000],
            ['ddddddddddddddddddddddddddddddd1', 1536700000001],
            ['ddddddddddddddddddddddddddddddd3', 1536700000001],
            ['ddddddddddddddddddddddddddddddd2', 1536700000001],
            ['bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbb1', 1536700000002],
        ];
        for (const [id, updatedTime] of testData) {
            await Resource_1.default.save({
                id,
                created_time: updatedTime,
                updated_time: updatedTime,
                user_updated_time: updatedTime,
                user_created_time: updatedTime,
                mime: 'application/octet-stream',
                ocr_text: 'test',
                ocr_status: types_1.ResourceOcrStatus.Done,
            }, { isNew: true, autoTimestamp: false });
        }
        expect((await Resource_1.default.allForNormalization(0, '')).length).toBe(testData.length);
        {
            const resources = await Resource_1.default.allForNormalization(1536700000001, 'ddddddddddddddddddddddddddddddd2');
            expect(resources.length).toBe(2);
            expect(resources.map(r => r.id)).toEqual(['ddddddddddddddddddddddddddddddd3', 'bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbb1']);
        }
        {
            const resources = await Resource_1.default.allForNormalization(1536700000000, '00000000000000000000000000000001');
            expect(resources.length).toBe(4);
            expect(resources.map(r => r.id)).toEqual(['ddddddddddddddddddddddddddddddd1', 'ddddddddddddddddddddddddddddddd2', 'ddddddddddddddddddddddddddddddd3', 'bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbb1']);
        }
    });
});
//# sourceMappingURL=Resource.test.js.map