"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const BaseModel_1 = require("../../BaseModel");
const Folder_1 = require("../Folder");
const ItemChange_1 = require("../ItemChange");
const Note_1 = require("../Note");
const reducer_1 = require("../../services/share/reducer");
const readOnly_1 = require("./readOnly");
const Resource_1 = require("../Resource");
const shim_1 = require("../../shim");
const test_utils_1 = require("../../testing/test-utils");
const BaseItem_1 = require("../BaseItem");
const checkReadOnly = (itemType, item, shareData = reducer_1.defaultState, syncUserId = '') => {
    return (0, readOnly_1.itemIsReadOnlySync)(itemType, ItemChange_1.default.SOURCE_UNSPECIFIED, item, syncUserId, shareData);
};
const createTestResource = async () => {
    const tempFile = (0, test_utils_1.tempFilePath)('txt');
    await shim_1.default.fsDriver().writeFile(tempFile, 'Test', 'utf8');
    const note1 = await Note_1.default.save({ title: 'note' });
    await shim_1.default.attachFileToNote(note1, tempFile);
};
const makeOwnerOfShare = (shareId, syncUserId) => {
    BaseItem_1.default.syncShareCache.shares = BaseItem_1.default.syncShareCache.shares.map(share => {
        if (share.id === shareId) {
            return Object.assign(Object.assign({}, share), { user: {
                    id: syncUserId,
                    email: 'test@example.com',
                } });
        }
        return share;
    });
};
describe('readOnly', () => {
    beforeEach(async () => {
        await (0, test_utils_1.setupDatabaseAndSynchronizer)(1);
        await (0, test_utils_1.switchClient)(1);
    });
    test('trashed items should be marked as read-only', async () => {
        let folder = await Folder_1.default.save({ title: 'Test' });
        let note = await Note_1.default.save({ parent_id: folder.id, title: 'Test note' });
        expect(checkReadOnly(BaseModel_1.ModelType.Note, note)).toBe(false);
        expect(checkReadOnly(BaseModel_1.ModelType.Folder, folder)).toBe(false);
        await Folder_1.default.delete(folder.id, { toTrash: true });
        // Should be deleted
        note = await Note_1.default.load(note.id);
        expect(note.deleted_time).not.toBe(0);
        folder = await Folder_1.default.load(folder.id);
        expect(folder.deleted_time).not.toBe(0);
        expect(checkReadOnly(BaseModel_1.ModelType.Note, note)).toBe(true);
        expect(checkReadOnly(BaseModel_1.ModelType.Folder, folder)).toBe(true);
    });
    test('should support checking if resources are not read-only', async () => {
        await createTestResource();
        const resource = (await Resource_1.default.all())[0];
        expect(checkReadOnly(BaseModel_1.ModelType.Resource, resource)).toBe(false);
    });
    test('should support checking that resources are read-only due to a share', async () => {
        await createTestResource();
        const share_id = '123456';
        let resource = (await Resource_1.default.all())[0];
        resource = await Resource_1.default.save(Object.assign(Object.assign({}, resource), { share_id }));
        const cleanup = (0, test_utils_1.simulateReadOnlyShareEnv)(share_id);
        expect(checkReadOnly(BaseModel_1.ModelType.Resource, resource, BaseItem_1.default.syncShareCache)).toBe(true);
        cleanup();
    });
    test('should support checking that items are read-only when there are multiple shares', async () => {
        const shareId1 = '123456';
        const shareId2 = '234567';
        const note = await Note_1.default.save({ body: 'test', share_id: shareId1 });
        const syncUserId = 'test-user-id';
        const cleanup = (0, test_utils_1.simulateReadOnlyShareEnv)([shareId1, shareId2]);
        // If the owner of a different share, should be read-only
        makeOwnerOfShare(shareId2, syncUserId);
        expect(checkReadOnly(BaseModel_1.ModelType.Note, note, BaseItem_1.default.syncShareCache, syncUserId)).toBe(true);
        // If also the owner of the same share, it should not be read-only
        makeOwnerOfShare(shareId1, syncUserId);
        expect(checkReadOnly(BaseModel_1.ModelType.Note, note, BaseItem_1.default.syncShareCache, syncUserId)).toBe(false);
        cleanup();
    });
});
//# sourceMappingURL=readOnly.test.js.map