"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const Resource_1 = require("../models/Resource");
const Setting_1 = require("../models/Setting");
const BaseService_1 = require("./BaseService");
const ResourceService_1 = require("./ResourceService");
const Logger_1 = require("@joplin/utils/Logger");
const shim_1 = require("../shim");
const checkDisabledSyncItemsNotification_1 = require("./synchronizer/utils/checkDisabledSyncItemsNotification");
const { Dirnames } = require('./synchronizer/utils/types');
const EventEmitter = require('events');
class ResourceFetcher extends BaseService_1.default {
    // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
    constructor(fileApi = null) {
        super();
        // eslint-disable-next-line @typescript-eslint/ban-types, @typescript-eslint/no-explicit-any -- Old code before rule was applied, Old code before rule was applied
        this.dispatch = (_o) => { };
        this.logger_ = new Logger_1.default();
        // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
        this.queue_ = [];
        // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
        this.fetchingItems_ = {};
        this.maxDownloads_ = 3;
        this.addingResources_ = false;
        this.eventEmitter_ = new EventEmitter();
        // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
        this.autoAddResourcesCalls_ = [];
        this.setFileApi(fileApi);
    }
    static instance() {
        if (ResourceFetcher.instance_)
            return ResourceFetcher.instance_;
        ResourceFetcher.instance_ = new ResourceFetcher();
        return ResourceFetcher.instance_;
    }
    // eslint-disable-next-line @typescript-eslint/ban-types -- Old code before rule was applied
    on(eventName, callback) {
        return this.eventEmitter_.on(eventName, callback);
    }
    // eslint-disable-next-line @typescript-eslint/ban-types -- Old code before rule was applied
    off(eventName, callback) {
        return this.eventEmitter_.removeListener(eventName, callback);
    }
    setLogger(logger) {
        this.logger_ = logger;
    }
    logger() {
        return this.logger_;
    }
    // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
    setFileApi(v) {
        if (v !== null && typeof v !== 'function')
            throw new Error(`fileApi must be a function that returns the API. Type is ${typeof v}`);
        this.fileApi_ = v;
    }
    async fileApi() {
        return this.fileApi_();
    }
    queuedItemIndex_(resourceId) {
        for (let i = 0; i < this.fetchingItems_.length; i++) {
            const item = this.fetchingItems_[i];
            if (item.id === resourceId)
                return i;
        }
        return -1;
    }
    updateReport() {
        const fetchingCount = Object.keys(this.fetchingItems_).length;
        this.dispatch({
            type: 'RESOURCE_FETCHER_SET',
            fetchingCount: fetchingCount,
            toFetchCount: fetchingCount + this.queue_.length,
        });
    }
    async markForDownload(resourceIds) {
        if (!Array.isArray(resourceIds))
            resourceIds = [resourceIds];
        const fetchStatuses = await Resource_1.default.fetchStatuses(resourceIds);
        const idsToKeep = [];
        for (const status of fetchStatuses) {
            if (status.fetch_status !== Resource_1.default.FETCH_STATUS_IDLE)
                continue;
            idsToKeep.push(status.resource_id);
        }
        for (const id of idsToKeep) {
            await Resource_1.default.markForDownload(id);
        }
        for (const id of idsToKeep) {
            this.queueDownload_(id, 'high');
        }
    }
    queueDownload_(resourceId, priority = null) {
        if (priority === null)
            priority = 'normal';
        const index = this.queuedItemIndex_(resourceId);
        if (index >= 0)
            return false;
        if (this.fetchingItems_[resourceId])
            return false;
        const item = { id: resourceId };
        if (priority === 'high') {
            this.queue_.splice(0, 0, item);
        }
        else {
            this.queue_.push(item);
        }
        this.updateReport();
        this.scheduleQueueProcess();
        return true;
    }
    async startDownload_(resourceId) {
        if (this.fetchingItems_[resourceId])
            return;
        this.fetchingItems_[resourceId] = true;
        this.updateReport();
        const resource = await Resource_1.default.load(resourceId);
        const localState = await Resource_1.default.localState(resource);
        const completeDownload = async (emitDownloadComplete = true, localResourceContentPath = '') => {
            // 2019-05-12: This is only necessary to set the file size of the resources that come via
            // sync. The other ones have been done using migrations/20.js. This code can be removed
            // after a few months.
            if (resource && resource.size < 0 && localResourceContentPath && !resource.encryption_blob_encrypted) {
                await shim_1.default.fsDriver().waitTillExists(localResourceContentPath);
                await ResourceService_1.default.autoSetFileSizes();
            }
            delete this.fetchingItems_[resource.id];
            this.logger().debug(`ResourceFetcher: Removed from fetchingItems: ${resource.id}. New: ${JSON.stringify(this.fetchingItems_)}`);
            this.scheduleQueueProcess();
            // Note: This downloadComplete event is not really right or useful because the resource
            // might still be encrypted and the caller usually can't do much with this. In particular
            // the note being displayed will refresh the resource images but since they are still
            // encrypted it's not useful. Probably, the views should listen to DecryptionWorker events instead.
            if (resource && emitDownloadComplete)
                this.eventEmitter_.emit('downloadComplete', { id: resource.id, encrypted: !!resource.encryption_blob_encrypted });
            this.updateReport();
        };
        if (!resource) {
            this.logger().info(`ResourceFetcher: Attempting to download a resource that does not exist (has been deleted?): ${resourceId}`);
            await completeDownload(false);
            return;
        }
        // Shouldn't happen, but just to be safe don't re-download the
        // resource if it's already been downloaded.
        if (localState.fetch_status === Resource_1.default.FETCH_STATUS_DONE) {
            await completeDownload(false);
            return;
        }
        const fileApi = await this.fileApi();
        if (!fileApi) {
            this.logger().debug('ResourceFetcher: Disabled because fileApi is not set');
            return;
        }
        this.fetchingItems_[resourceId] = resource;
        const localResourceContentPath = Resource_1.default.fullPath(resource, !!resource.encryption_blob_encrypted);
        const remoteResourceContentPath = `${Dirnames.Resources}/${resource.id}`;
        await Resource_1.default.setLocalState(resource, { fetch_status: Resource_1.default.FETCH_STATUS_STARTED });
        this.logger().debug(`ResourceFetcher: Downloading resource: ${resource.id}`);
        this.eventEmitter_.emit('downloadStarted', { id: resource.id });
        try {
            await fileApi.get(remoteResourceContentPath, { path: localResourceContentPath, target: 'file' });
            if (!(await shim_1.default.fsDriver().exists(localResourceContentPath)))
                throw new Error(`Resource not found: ${resource.id}`);
            await Resource_1.default.setLocalState(resource, { fetch_status: Resource_1.default.FETCH_STATUS_DONE });
            this.logger().debug(`ResourceFetcher: Resource downloaded: ${resource.id}`);
            await completeDownload(true, localResourceContentPath);
        }
        catch (error) {
            this.logger().error(`ResourceFetcher: Could not download resource: ${resource.id}`, error);
            await Resource_1.default.setLocalState(resource, { fetch_status: Resource_1.default.FETCH_STATUS_ERROR, fetch_error: error.message });
            await completeDownload();
        }
    }
    processQueue_() {
        while (Object.getOwnPropertyNames(this.fetchingItems_).length < this.maxDownloads_) {
            if (!this.queue_.length)
                break;
            const item = this.queue_.splice(0, 1)[0];
            void this.startDownload_(item.id);
        }
        if (!this.queue_.length) {
            void this.autoAddResources(10);
        }
    }
    async waitForAllFinished() {
        return new Promise((resolve) => {
            const iid = shim_1.default.setInterval(() => {
                if (!this.updateReportIID_ &&
                    !this.scheduleQueueProcessIID_ &&
                    !this.queue_.length &&
                    !this.autoAddResourcesCalls_.length &&
                    !Object.getOwnPropertyNames(this.fetchingItems_).length) {
                    shim_1.default.clearInterval(iid);
                    resolve(null);
                }
            }, 100);
        });
    }
    async autoAddResources(limit = null) {
        this.autoAddResourcesCalls_.push(true);
        try {
            if (limit === null)
                limit = 10;
            if (this.addingResources_)
                return;
            this.addingResources_ = true;
            this.logger().info(`ResourceFetcher: Auto-add resources: Mode: ${Setting_1.default.value('sync.resourceDownloadMode')}`);
            let count = 0;
            const resources = await Resource_1.default.needToBeFetched(Setting_1.default.value('sync.resourceDownloadMode'), limit);
            for (let i = 0; i < resources.length; i++) {
                const added = this.queueDownload_(resources[i].id);
                if (added)
                    count++;
            }
            this.logger().info(`ResourceFetcher: Auto-added resources: ${count}`);
            // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
            await (0, checkDisabledSyncItemsNotification_1.default)((action) => this.dispatch(action));
        }
        finally {
            this.addingResources_ = false;
            this.autoAddResourcesCalls_.pop();
        }
    }
    async start() {
        await Resource_1.default.resetStartedFetchStatus();
        void this.autoAddResources(10);
    }
    async startAndWait() {
        await this.start();
        await this.waitForAllFinished();
    }
    scheduleQueueProcess() {
        if (this.scheduleQueueProcessIID_) {
            shim_1.default.clearTimeout(this.scheduleQueueProcessIID_);
            this.scheduleQueueProcessIID_ = null;
        }
        this.scheduleQueueProcessIID_ = shim_1.default.setTimeout(() => {
            this.processQueue_();
            this.scheduleQueueProcessIID_ = null;
        }, 100);
    }
    scheduleAutoAddResources() {
        if (this.scheduleAutoAddResourcesIID_)
            return;
        this.scheduleAutoAddResourcesIID_ = shim_1.default.setTimeout(() => {
            this.scheduleAutoAddResourcesIID_ = null;
            void ResourceFetcher.instance().autoAddResources();
        }, 1000);
    }
    async fetchAll() {
        await Resource_1.default.resetStartedFetchStatus();
        void this.autoAddResources(null);
    }
    async destroy() {
        this.eventEmitter_.removeAllListeners();
        if (this.scheduleQueueProcessIID_) {
            shim_1.default.clearTimeout(this.scheduleQueueProcessIID_);
            this.scheduleQueueProcessIID_ = null;
        }
        if (this.scheduleAutoAddResourcesIID_) {
            shim_1.default.clearTimeout(this.scheduleAutoAddResourcesIID_);
            this.scheduleAutoAddResourcesIID_ = null;
        }
        await this.waitForAllFinished();
        this.eventEmitter_ = null;
        ResourceFetcher.instance_ = null;
    }
}
exports.default = ResourceFetcher;
//# sourceMappingURL=ResourceFetcher.js.map