"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const InteropService_1 = require("../../services/interop/InteropService");
const test_utils_1 = require("../../testing/test-utils");
const Folder_1 = require("../../models/Folder");
const Note_1 = require("../../models/Note");
const Tag_1 = require("../../models/Tag");
const time_1 = require("../../time");
const fs = require("fs-extra");
const types_1 = require("./types");
const frontMatter_1 = require("../../utils/frontMatter");
async function recreateExportDir() {
    const dir = (0, test_utils_1.exportDir)();
    await fs.remove(dir);
    await fs.mkdirp(dir);
}
describe('interop/InteropService_Exporter_Md_frontmatter', () => {
    async function exportAndLoad(path) {
        const service = InteropService_1.default.instance();
        await service.export({
            path: (0, test_utils_1.exportDir)(),
            format: types_1.ExportModuleOutputFormat.MarkdownFrontMatter,
        });
        return await fs.readFile(path, 'utf8');
    }
    beforeEach(async () => {
        await (0, test_utils_1.setupDatabaseAndSynchronizer)(1);
        await (0, test_utils_1.switchClient)(1);
        await recreateExportDir();
    });
    test('should export MD file with YAML header', (async () => {
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        await Note_1.default.save({ title: 'ma', latitude: 58.2222, user_updated_time: 1, user_created_time: 1, body: '**ma note**', parent_id: folder1.id });
        const content = await exportAndLoad(`${(0, test_utils_1.exportDir)()}/folder1/ma.md`);
        expect(content.startsWith('---')).toBe(true);
        expect(content).toContain('title: ma');
        expect(content).toContain('updated:'); // Will be current time of test run
        expect(content).toContain(`created: ${time_1.default.unixMsToRfc3339Sec(1)}`);
        expect(content).toContain('latitude: 58.22220000');
        expect(content).toContain('longitude: 0.00000000');
        expect(content).toContain('altitude: 0.0000');
        expect(content).toContain('**ma note**');
        expect(content).not.toContain('completed?');
        expect(content).not.toContain('author');
        expect(content).not.toContain('source');
        expect(content).not.toContain('due');
    }));
    test('should export without additional quotes', (async () => {
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        await Note_1.default.save({ title: '-60', body: '**ma note**', parent_id: folder1.id });
        const content = await exportAndLoad(`${(0, test_utils_1.exportDir)()}/folder1/-60.md`);
        expect(content).toContain('title: -60');
    }));
    test('should export tags', (async () => {
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        const note = await Note_1.default.save({ title: 'Title', body: '**ma note**', parent_id: folder1.id });
        await Tag_1.default.addNoteTagByTitle(note.id, 'lamp');
        await Tag_1.default.addNoteTagByTitle(note.id, 'moth');
        await Tag_1.default.addNoteTagByTitle(note.id, 'godzilla');
        const content = await exportAndLoad(`${(0, test_utils_1.exportDir)()}/folder1/Title.md`);
        expect(content).toContain('tags:\n  - godzilla\n  - lamp\n  - moth');
    }));
    test('should export todo', (async () => {
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        await Note_1.default.save({ title: 'Todo', is_todo: 1, todo_due: 1, body: '**ma note**', parent_id: folder1.id });
        const content = await exportAndLoad(`${(0, test_utils_1.exportDir)()}/folder1/Todo.md`);
        expect(content).toContain(`due: ${time_1.default.unixMsToRfc3339Sec(1)}`);
        expect(content).toContain('completed?: no');
    }));
    test('should export author', (async () => {
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        await Note_1.default.save({ title: 'Author', author: 'Scott Joplin', body: '**ma note**', parent_id: folder1.id });
        const content = await exportAndLoad(`${(0, test_utils_1.exportDir)()}/folder1/Author.md`);
        expect(content).toContain('author: Scott Joplin');
    }));
    test('should export source', (async () => {
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        await Note_1.default.save({ title: 'Source', source_url: 'https://joplinapp.org', body: '**ma note**', parent_id: folder1.id });
        const content = await exportAndLoad(`${(0, test_utils_1.exportDir)()}/folder1/Source.md`);
        expect(content).toContain('source: https://joplinapp.org');
    }));
    test('should export fields in the correct order', (async () => {
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        const note = await Note_1.default.save({
            title: 'Fields',
            is_todo: 1,
            todo_due: 1,
            author: 'Scott Joplin',
            source_url: 'https://joplinapp.org',
            body: '**ma note**',
            parent_id: folder1.id,
        });
        await Tag_1.default.addNoteTagByTitle(note.id, 'piano');
        await Tag_1.default.addNoteTagByTitle(note.id, 'greatness');
        const content = await exportAndLoad(`${(0, test_utils_1.exportDir)()}/folder1/Fields.md`);
        const fieldIndices = frontMatter_1.fieldOrder.map(field => content.indexOf(field));
        expect(fieldIndices).toBe(fieldIndices.sort());
    }));
    test('should export title with a newline encoded', (async () => {
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        await Note_1.default.save({ title: 'Source\ntitle', body: '**ma note**', parent_id: folder1.id });
        const content = await exportAndLoad(`${(0, test_utils_1.exportDir)()}/folder1/Source_title.md`);
        expect(content).toContain('title: |-\n  Source\n  title');
    }));
    test('should not export coordinates if they\'re not available', (async () => {
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        await Note_1.default.save({ title: 'Coordinates', body: '**ma note**', parent_id: folder1.id });
        const content = await exportAndLoad(`${(0, test_utils_1.exportDir)()}/folder1/Coordinates.md`);
        expect(content).not.toContain('latitude');
        expect(content).not.toContain('longitude');
        expect(content).not.toContain('altitude');
    }));
    test('should export note without tag keyword if the tag has been deleted', (async () => {
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        const note = await Note_1.default.save({ title: 'NoTag', body: '**ma note**', parent_id: folder1.id });
        const tag = await Tag_1.default.save({ title: 'tag' });
        await Tag_1.default.setNoteTagsByIds(note.id, [tag.id]);
        await Tag_1.default.delete(tag.id);
        const content = await exportAndLoad(`${(0, test_utils_1.exportDir)()}/folder1/NoTag.md`);
        expect(content).not.toContain('tag');
    }));
    test('should export a valid file when the title starts with a dash', (async () => {
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        await Note_1.default.save({ title: '- title with dash', body: '**ma note**', parent_id: folder1.id });
        const content = await exportAndLoad(`${(0, test_utils_1.exportDir)()}/folder1/- title with dash.md`);
        expect(content).toContain('title: \'- title with dash\'');
    }));
});
//# sourceMappingURL=InteropService_Exporter_Md_frontmatter.test.js.map