"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const Setting_1 = require("../../models/Setting");
const shim_1 = require("../../shim");
const test_utils_1 = require("../../testing/test-utils");
const KeychainService_1 = require("./KeychainService");
const KeychainServiceDriver_dummy_1 = require("./KeychainServiceDriver.dummy");
const KeychainServiceDriver_electron_1 = require("./KeychainServiceDriver.electron");
const KeychainServiceDriver_node_1 = require("./KeychainServiceDriver.node");
const mockSafeStorage = ({ // Safe storage
isEncryptionAvailable = () => true, encryptString = async (s) => (`e:${s}`), decryptString = async (s) => s.substring(2), }) => {
    const mock = {
        isEncryptionAvailable: jest.fn(isEncryptionAvailable),
        encryptString: jest.fn(encryptString),
        decryptString: jest.fn(decryptString),
        getSelectedStorageBackend: jest.fn(() => 'mock'),
    };
    shim_1.default.electronBridge = () => ({
        safeStorage: mock,
    });
    return mock;
};
const mockKeytar = () => {
    const storage = new Map();
    const keytarMock = {
        getPassword: jest.fn(async (key, client) => {
            return storage.get(`${client}--${key}`);
        }),
        setPassword: jest.fn(async (key, client, password) => {
            if (!password)
                throw new Error('Keytar doesn\'t support empty passwords.');
            storage.set(`${client}--${key}`, password);
        }),
        deletePassword: jest.fn(async (key, client) => {
            storage.delete(`${client}--${key}`);
        }),
    };
    shim_1.default.keytar = () => keytarMock;
    return keytarMock;
};
const makeDrivers = () => [
    new KeychainServiceDriver_electron_1.default(Setting_1.default.value('appId'), Setting_1.default.value('clientId')),
    new KeychainServiceDriver_node_1.default(Setting_1.default.value('appId'), Setting_1.default.value('clientId')),
];
const testSaveLoadSecureSetting = async (expectedPassword) => {
    Setting_1.default.setValue('encryption.masterPassword', expectedPassword);
    await Setting_1.default.saveAll();
    await Setting_1.default.load();
    expect(Setting_1.default.value('encryption.masterPassword')).toBe(expectedPassword);
};
describe('KeychainService', () => {
    beforeEach(async () => {
        await (0, test_utils_1.setupDatabaseAndSynchronizer)(0);
        await (0, test_utils_1.switchClient)(0);
        KeychainService_1.default.instance().readOnly = false;
        Setting_1.default.setValue('keychain.supported', 1);
        shim_1.default.electronBridge = null;
        shim_1.default.keytar = null;
    });
    test('should copy keys from keytar to safeStorage', async () => {
        const keytarMock = mockKeytar();
        await KeychainService_1.default.instance().initialize(makeDrivers());
        // Set a secure setting
        Setting_1.default.setValue('encryption.masterPassword', 'testing');
        await Setting_1.default.saveAll();
        mockSafeStorage({});
        await KeychainService_1.default.instance().initialize(makeDrivers());
        await Setting_1.default.load();
        expect(Setting_1.default.value('encryption.masterPassword')).toBe('testing');
        await Setting_1.default.saveAll();
        // For now, passwords should not be removed from old backends -- this allows
        // users to revert to an earlier version of Joplin without data loss.
        expect(keytarMock.deletePassword).not.toHaveBeenCalled();
        expect(shim_1.default.electronBridge().safeStorage.encryptString).toHaveBeenCalled();
        expect(shim_1.default.electronBridge().safeStorage.encryptString).toHaveBeenCalledWith('testing');
        await Setting_1.default.load();
        expect(Setting_1.default.value('encryption.masterPassword')).toBe('testing');
    });
    test('should use keytar when safeStorage is unavailable', async () => {
        const keytarMock = mockKeytar();
        await KeychainService_1.default.instance().initialize(makeDrivers());
        await testSaveLoadSecureSetting('test-password');
        expect(keytarMock.setPassword).toHaveBeenCalledWith(`${Setting_1.default.value('appId')}.setting.encryption.masterPassword`, `${Setting_1.default.value('clientId')}@joplin`, 'test-password');
    });
    test('should re-check for keychain support when a new driver is added', async () => {
        mockKeytar();
        mockSafeStorage({});
        Setting_1.default.setValue('keychain.supported', -1);
        await KeychainService_1.default.instance().initialize([
            new KeychainServiceDriver_dummy_1.default(Setting_1.default.value('appId'), Setting_1.default.value('clientId')),
        ]);
        await KeychainService_1.default.instance().detectIfKeychainSupported();
        expect(Setting_1.default.value('keychain.supported')).toBe(0);
        // Should re-run the check after keytar and safeStorage are available.
        await KeychainService_1.default.instance().initialize(makeDrivers());
        await KeychainService_1.default.instance().detectIfKeychainSupported();
        expect(Setting_1.default.value('keychain.supported')).toBe(1);
        // Should re-run the check if safeStorage and keytar are both no longer available.
        await KeychainService_1.default.instance().initialize([]);
        await KeychainService_1.default.instance().detectIfKeychainSupported();
        expect(Setting_1.default.value('keychain.supported')).toBe(0);
    });
    test('should handle the case where safeStorage.encryptString throws', async () => {
        mockSafeStorage({
            encryptString: () => {
                throw new Error('Failed!');
            },
        });
        Setting_1.default.setValue('keychain.supported', -1);
        await KeychainService_1.default.instance().initialize(makeDrivers());
        await (0, test_utils_1.withWarningSilenced)(/Encrypting a setting failed/, async () => {
            await KeychainService_1.default.instance().detectIfKeychainSupported();
        });
        expect(Setting_1.default.value('keychain.supported')).toBe(0);
    });
    test('should load settings from a read-only KeychainService if not present in the database', async () => {
        mockSafeStorage({});
        const service = KeychainService_1.default.instance();
        await service.initialize(makeDrivers());
        expect(await service.setPassword('setting.encryption.masterPassword', 'keychain password')).toBe(true);
        service.readOnly = true;
        await service.initialize(makeDrivers());
        await Setting_1.default.load();
        expect(Setting_1.default.value('encryption.masterPassword')).toBe('keychain password');
    });
    test('settings should be saved to database with a read-only keychain', async () => {
        const safeStorage = mockSafeStorage({});
        const service = KeychainService_1.default.instance();
        service.readOnly = true;
        await service.initialize(makeDrivers());
        await service.detectIfKeychainSupported();
        expect(Setting_1.default.value('keychain.supported')).toBe(1);
        await testSaveLoadSecureSetting('testing...');
        expect(safeStorage.encryptString).not.toHaveBeenCalledWith('testing...');
    });
    test('loading settings with a read-only keychain should prefer the database', async () => {
        const safeStorage = mockSafeStorage({});
        const service = KeychainService_1.default.instance();
        await service.initialize(makeDrivers());
        // Set an initial value
        expect(await service.setPassword('setting.encryption.masterPassword', 'test keychain password')).toBe(true);
        service.readOnly = true;
        await service.initialize(makeDrivers());
        safeStorage.encryptString.mockClear();
        Setting_1.default.setValue('encryption.masterPassword', 'test database password');
        await Setting_1.default.saveAll();
        await Setting_1.default.load();
        expect(Setting_1.default.value('encryption.masterPassword')).toBe('test database password');
        expect(await service.password('setting.encryption.masterPassword')).toBe('test keychain password');
        // Should not have attempted to encrypt settings in read-only mode.
        expect(safeStorage.encryptString).not.toHaveBeenCalled();
    });
});
//# sourceMappingURL=KeychainService.test.js.map