"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const BaseModel_1 = require("../../BaseModel");
const Note_1 = require("../../models/Note");
const Resource_1 = require("../../models/Resource");
const shim_1 = require("../../shim");
const test_utils_1 = require("../../testing/test-utils");
const types_1 = require("../database/types");
const SearchEngine_1 = require("./SearchEngine");
const newSearchEngine = () => {
    const engine = new SearchEngine_1.default();
    engine.setDb((0, test_utils_1.db)());
    return engine;
};
const createNoteAndResource = async () => {
    const note = await Note_1.default.save({});
    await Note_1.default.save({});
    await shim_1.default.attachFileToNote(note, `${test_utils_1.ocrSampleDir}/testocr.png`);
    const resource = (await Resource_1.default.all())[0];
    await (0, test_utils_1.resourceService)().indexNoteResources();
    return { note, resource };
};
describe('SearchEngine.resources', () => {
    beforeEach(async () => {
        global.console = require('console');
        await (0, test_utils_1.setupDatabaseAndSynchronizer)(1);
        await (0, test_utils_1.switchClient)(1);
    });
    it('should index resources', async () => {
        const engine = newSearchEngine();
        await Resource_1.default.save({
            id: '00000000000000000000000000000001',
            mime: 'image/jpeg',
            title: 'Bonjour ça va ?',
            ocr_status: types_1.ResourceOcrStatus.Done,
            ocr_text: 'héllô, hôw äre yoù ?',
        }, { isNew: true });
        await engine.syncTables();
        const normalized = await (0, test_utils_1.db)().selectAll('select * from items_normalized');
        expect(normalized[0].title).toBe('bonjour ca va ?');
        expect(normalized[0].body).toBe('hello, how are you ?');
    });
    it('should return notes associated with indexed resources', (async () => {
        const { note, resource } = await createNoteAndResource();
        const ocrService = (0, test_utils_1.newOcrService)();
        await ocrService.processResources();
        const searchEngine = newSearchEngine();
        await searchEngine.syncTables();
        const results = await searchEngine.search('lazy fox');
        expect(results.length).toBe(1);
        expect(results[0].id).toBe(note.id);
        expect(results[0].item_id).toBe(resource.id);
        expect(results[0].item_type).toBe(BaseModel_1.ModelType.Resource);
        await ocrService.dispose();
    }));
    it('should not return resources associated with deleted notes', (async () => {
        const { note } = await createNoteAndResource();
        const note2 = await Note_1.default.save({ body: 'lazy fox' });
        await Note_1.default.delete(note.id, { toTrash: true });
        const ocrService = (0, test_utils_1.newOcrService)();
        await ocrService.processResources();
        const searchEngine = newSearchEngine();
        await searchEngine.syncTables();
        const results = await searchEngine.search('lazy fox');
        expect(results.length).toBe(1);
        expect(results[0].id).toBe(note2.id);
    }));
    it('should delete normalized data when a resource is deleted', async () => {
        const engine = newSearchEngine();
        const resource = await Resource_1.default.save({
            id: '00000000000000000000000000000001',
            mime: 'image/jpeg',
            title: 'hello',
            ocr_status: types_1.ResourceOcrStatus.Done,
            ocr_text: 'hi',
        }, { isNew: true });
        await engine.syncTables();
        expect((await (0, test_utils_1.db)().selectAll('select * from items_normalized')).length).toBe(1);
        await Resource_1.default.delete(resource.id);
        expect((await (0, test_utils_1.db)().selectAll('select * from items_normalized')).length).toBe(0);
    });
    it('should sort resources', async () => {
        const engine = newSearchEngine();
        const resourceData = [
            ['abcd abcd abcd', 'efgh'],
            ['abcd', 'ijkl'],
            ['ijkl', 'mnop'],
        ];
        for (const [title, body] of resourceData) {
            await Resource_1.default.save({
                mime: 'image/jpeg',
                title,
                ocr_status: types_1.ResourceOcrStatus.Done,
                ocr_text: body,
            });
            await (0, test_utils_1.msleep)(1);
        }
        await engine.syncTables();
        const results = await engine.search('abcd', { includeOrphanedResources: true });
        expect(results[0].title).toBe('abcd abcd abcd');
        expect(results[1].title).toBe('abcd');
    });
});
//# sourceMappingURL=SearchEngine.resources.test.js.map