"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const Note_1 = require("../../models/Note");
const test_utils_1 = require("../../testing/test-utils");
const Folder_1 = require("../../models/Folder");
const syncInfoUtils_1 = require("../synchronizer/syncInfoUtils");
const ppk_1 = require("../e2ee/ppk/ppk");
const MasterKey_1 = require("../../models/MasterKey");
const utils_1 = require("../e2ee/utils");
const Logger_1 = require("@joplin/utils/Logger");
const shim_1 = require("../../shim");
const Resource_1 = require("../../models/Resource");
const fs_extra_1 = require("fs-extra");
const BaseItem_1 = require("../../models/BaseItem");
const ResourceService_1 = require("../ResourceService");
const Setting_1 = require("../../models/Setting");
const BaseModel_1 = require("../../BaseModel");
const test_utils_synchronizer_1 = require("../../testing/test-utils-synchronizer");
const mockShareService_1 = require("../../testing/share/mockShareService");
const testImagePath = `${test_utils_1.supportDir}/photo.jpg`;
const mockServiceForNoteSharing = () => {
    return (0, mockShareService_1.default)({
        getShares: async () => {
            return { items: [] };
        },
        postShares: async () => null,
        getShareInvitations: async () => null,
    });
};
describe('ShareService', () => {
    beforeEach(async () => {
        await (0, test_utils_1.setupDatabaseAndSynchronizer)(1);
        await (0, test_utils_1.switchClient)(1);
    });
    it('should not change the note user timestamps when sharing or unsharing', async () => {
        let note = await Note_1.default.save({});
        const service = mockServiceForNoteSharing();
        await (0, test_utils_1.msleep)(1);
        await service.shareNote(note.id, false);
        function checkTimestamps(previousNote, newNote) {
            // After sharing or unsharing, only the updated_time property should
            // be updated, for sync purposes. All other timestamps shouldn't
            // change.
            expect(previousNote.user_created_time).toBe(newNote.user_created_time);
            expect(previousNote.user_updated_time).toBe(newNote.user_updated_time);
            expect(previousNote.updated_time < newNote.updated_time).toBe(true);
            expect(previousNote.created_time).toBe(newNote.created_time);
        }
        {
            const noteReloaded = await Note_1.default.load(note.id);
            checkTimestamps(note, noteReloaded);
            note = noteReloaded;
        }
        await (0, test_utils_1.msleep)(1);
        await service.unshareNote(note.id);
        {
            const noteReloaded = await Note_1.default.load(note.id);
            checkTimestamps(note, noteReloaded);
        }
    });
    it('should not encrypt items that are shared', async () => {
        const folder = await Folder_1.default.save({});
        const note = await Note_1.default.save({ parent_id: folder.id });
        await shim_1.default.attachFileToNote(note, testImagePath);
        const service = mockServiceForNoteSharing();
        (0, syncInfoUtils_1.setEncryptionEnabled)(true);
        await (0, test_utils_1.loadEncryptionMasterKey)();
        await (0, test_utils_1.synchronizerStart)();
        let previousBlobUpdatedTime = Infinity;
        {
            const allItems = await (0, test_utils_synchronizer_1.remoteNotesFoldersResources)();
            expect(allItems.map(it => it.encryption_applied)).toEqual([1, 1, 1]);
            previousBlobUpdatedTime = allItems.find(it => it.type_ === BaseModel_1.ModelType.Resource).blob_updated_time;
        }
        await service.shareNote(note.id, false);
        await (0, test_utils_1.msleep)(1);
        await Folder_1.default.updateAllShareIds((0, test_utils_1.resourceService)(), []);
        await (0, test_utils_1.synchronizerStart)();
        {
            const allItems = await (0, test_utils_synchronizer_1.remoteNotesFoldersResources)();
            expect(allItems.find(it => it.type_ === BaseModel_1.ModelType.Note).encryption_applied).toBe(0);
            expect(allItems.find(it => it.type_ === BaseModel_1.ModelType.Folder).encryption_applied).toBe(1);
            const resource = allItems.find(it => it.type_ === BaseModel_1.ModelType.Resource);
            expect(resource.encryption_applied).toBe(0);
            // Indicates that both the metadata and blob have been decrypted on
            // the sync target.
            expect(resource.blob_updated_time).toBe(resource.updated_time);
            expect(resource.blob_updated_time).toBeGreaterThan(previousBlobUpdatedTime);
        }
    });
    // eslint-disable-next-line @typescript-eslint/ban-types -- Old code before rule was applied
    function testShareFolderService(extraExecHandlers = {}) {
        let nextShareId = 1;
        let shares = [];
        const shareByFolderId = (folderId) => {
            return shares.find(share => share.folder_id === folderId);
        };
        return (0, mockShareService_1.default)({
            // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
            onExec: async (method, path, query, body) => {
                if (extraExecHandlers[`${method} ${path}`])
                    return extraExecHandlers[`${method} ${path}`](query, body);
                if (method === 'GET' && path === 'api/shares') {
                    return {
                        items: [...shares],
                    };
                }
                if (method === 'POST' && path === 'api/shares') {
                    // Return the existing share, if it exists. This is to match the behavior
                    // of Joplin Server.
                    const existingShare = shareByFolderId(body.folder_id);
                    if (existingShare) {
                        return existingShare;
                    }
                    // Use a predictable ID:
                    const id = `share_${nextShareId++}`;
                    const share = {
                        id,
                        master_key_id: body.master_key_id,
                        folder_id: body.folder_id,
                    };
                    shares.push(share);
                    return share;
                }
                if (method === 'DELETE' && path.startsWith('api/shares/')) {
                    const id = path.replace(/^api\/shares\//, '');
                    shares = shares.filter(share => share.id !== id);
                    return;
                }
                throw new Error(`Unhandled: ${method} ${path}`);
            },
        });
    }
    const prepareNoteFolderResource = async () => {
        const folder = await Folder_1.default.save({});
        let note = await Note_1.default.save({ parent_id: folder.id });
        note = await shim_1.default.attachFileToNote(note, testImagePath);
        const resourceId = (await Note_1.default.linkedResourceIds(note.body))[0];
        const resource = await Resource_1.default.load(resourceId);
        await (0, test_utils_1.resourceService)().indexNoteResources();
        return { folder, note, resource };
    };
    async function testShareFolder(service) {
        const { folder, note, resource } = await prepareNoteFolderResource();
        const share = await service.shareFolder(folder.id);
        expect(share.id).toBe('share_1');
        expect((await Folder_1.default.load(folder.id)).share_id).toBe('share_1');
        expect((await Note_1.default.load(note.id)).share_id).toBe('share_1');
        expect((await Resource_1.default.load(resource.id)).share_id).toBe('share_1');
        return { share, folder, note, resource };
    }
    it('should share a folder', async () => {
        await testShareFolder(testShareFolderService());
    });
    it('should share a folder - E2EE', async () => {
        const masterKey = await (0, test_utils_1.loadEncryptionMasterKey)();
        await (0, utils_1.setupAndEnableEncryption)((0, test_utils_1.encryptionService)(), masterKey, '111111');
        const ppk = await (0, ppk_1.generateKeyPair)((0, test_utils_1.encryptionService)(), '111111');
        (0, syncInfoUtils_1.setPpk)(ppk);
        const shareService = testShareFolderService();
        expect(await MasterKey_1.default.count()).toBe(1);
        let { folder, note, resource } = await prepareNoteFolderResource();
        BaseItem_1.default.shareService_ = shareService;
        Resource_1.default.shareService_ = shareService;
        await shareService.shareFolder(folder.id);
        await Folder_1.default.updateAllShareIds((0, test_utils_1.resourceService)(), []);
        // The share service should automatically create a new encryption key
        // specifically for that shared folder
        expect(await MasterKey_1.default.count()).toBe(2);
        folder = await Folder_1.default.load(folder.id);
        note = await Note_1.default.load(note.id);
        resource = await Resource_1.default.load(resource.id);
        // The key that is not the master key is the folder key
        const folderKey = (await MasterKey_1.default.all()).find(mk => mk.id !== masterKey.id);
        // Double-check that it's going to encrypt the folder using the shared
        // key (and not the user's own master key)
        expect(folderKey.id).not.toBe(masterKey.id);
        expect(folder.master_key_id).toBe(folderKey.id);
        await (0, utils_1.loadMasterKeysFromSettings)((0, test_utils_1.encryptionService)());
        try {
            const serializedNote = await Note_1.default.serializeForSync(note);
            expect(serializedNote).toContain(folderKey.id);
            // The resource should be encrypted using the above key (if it is,
            // the key ID will be in the header).
            const result = await Resource_1.default.fullPathForSyncUpload(resource);
            const content = await (0, fs_extra_1.readFile)(result.path, 'utf8');
            expect(content).toContain(folderKey.id);
            {
                await (0, test_utils_1.synchronizerStart)();
                const remoteItems = await (0, test_utils_synchronizer_1.remoteNotesFoldersResources)();
                expect(remoteItems.map(it => it.encryption_applied)).toEqual([1, 1, 1]);
            }
        }
        finally {
            BaseItem_1.default.shareService_ = shareService;
            Resource_1.default.shareService_ = null;
        }
    });
    it('should add a recipient', async () => {
        (0, syncInfoUtils_1.setEncryptionEnabled)(true);
        await (0, utils_1.updateMasterPassword)('', '111111');
        const ppk = await (0, ppk_1.generateKeyPair)((0, test_utils_1.encryptionService)(), '111111');
        (0, syncInfoUtils_1.setPpk)(ppk);
        const recipientPpk = await (0, ppk_1.generateKeyPair)((0, test_utils_1.encryptionService)(), '222222');
        expect(ppk.id).not.toBe(recipientPpk.id);
        let uploadedEmail = '';
        let uploadedMasterKey = null;
        const service = testShareFolderService({
            // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
            'GET api/users/toto%40example.com/public_key': async (_query, _body) => {
                return recipientPpk;
            },
            // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
            'POST api/shares/share_1/users': async (_query, body) => {
                uploadedEmail = body.email;
                uploadedMasterKey = JSON.parse(body.master_key);
            },
        });
        const { share } = await testShareFolder(service);
        await service.addShareRecipient(share.id, share.master_key_id, 'toto@example.com', { can_read: 1, can_write: 1 });
        expect(uploadedEmail).toBe('toto@example.com');
        const content = JSON.parse(uploadedMasterKey.content);
        expect(recipientPpk.id).toBe(content.ppkId);
    });
    it('should not update the folder\'s master_key_id when shareFolder is called multiple times', async () => {
        await (0, utils_1.generateMasterKeyAndEnableEncryption)((0, test_utils_1.encryptionService)(), 'testing!');
        const ppk = await (0, ppk_1.generateKeyPair)((0, test_utils_1.encryptionService)(), '111111');
        (0, syncInfoUtils_1.setPpk)(ppk);
        expect((0, syncInfoUtils_1.localSyncInfo)().e2ee).toBe(true);
        expect((0, syncInfoUtils_1.localSyncInfo)().masterKeys).toHaveLength(1);
        const service = testShareFolderService();
        const { share, folder } = await testShareFolder(service);
        // Should have created a new master key for the share
        expect((0, syncInfoUtils_1.localSyncInfo)().masterKeys).toHaveLength(2);
        expect((0, syncInfoUtils_1.localSyncInfo)().masterKeys.some(key => key.id === share.master_key_id)).toBe(true);
        // Sharing an already-shared folder should keep the existing key:
        const share2 = await service.shareFolder(folder.id);
        expect(share2.master_key_id).toBe(share.master_key_id);
        expect(await Folder_1.default.load(folder.id)).toHaveProperty('master_key_id', share.master_key_id);
        // Should not have added a new master key
        expect((0, syncInfoUtils_1.localSyncInfo)().masterKeys).toHaveLength(2);
    });
    it('should use a different master key when folders are unshared, then shared again', async () => {
        await (0, utils_1.generateMasterKeyAndEnableEncryption)((0, test_utils_1.encryptionService)(), 'testing!');
        const ppk = await (0, ppk_1.generateKeyPair)((0, test_utils_1.encryptionService)(), '111111');
        (0, syncInfoUtils_1.setPpk)(ppk);
        const service = testShareFolderService();
        const { share, folder } = await testShareFolder(service);
        await service.refreshShares();
        await service.unshareFolder(folder.id);
        const share2 = await service.shareFolder(folder.id);
        expect(share2.master_key_id).toBeTruthy();
        expect(share2.folder_id).toBe(folder.id);
        expect(share.master_key_id).not.toBe(share2.master_key_id);
    });
    it('should leave folders that are no longer with the user', async () => {
        // `checkShareConsistency` will emit a warning so we need to silent it
        // in tests.
        const previousLogLevel = Logger_1.default.globalLogger.setLevel(Logger_1.LogLevel.Error);
        const service = testShareFolderService({
            // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
            'GET api/shares': async (_query, _body) => {
                return {
                    items: [],
                    has_more: false,
                };
            },
        });
        const folder = await Folder_1.default.save({ share_id: 'nolongershared' });
        await service.checkShareConsistency();
        expect(await Folder_1.default.load(folder.id)).toBeFalsy();
        Logger_1.default.globalLogger.setLevel(previousLogLevel);
    });
    it('should leave a shared folder', async () => {
        const folder1 = await (0, test_utils_1.createFolderTree)('', [
            {
                title: 'folder 1',
                children: [
                    {
                        title: 'note 1',
                    },
                    {
                        title: 'note 2',
                    },
                ],
            },
        ]);
        const resourceService = new ResourceService_1.default();
        await Folder_1.default.save({ id: folder1.id, share_id: '123456789' });
        await Folder_1.default.updateAllShareIds(resourceService, []);
        const cleanup = (0, test_utils_1.simulateReadOnlyShareEnv)('123456789');
        const shareService = testShareFolderService();
        await shareService.leaveSharedFolder(folder1.id, 'somethingrandom');
        expect(await Folder_1.default.count()).toBe(0);
        expect(await Note_1.default.count()).toBe(0);
        const deletedItems = await BaseItem_1.default.deletedItems(Setting_1.default.value('sync.target'));
        expect(deletedItems.length).toBe(1);
        expect(deletedItems[0].item_type).toBe(BaseModel_1.ModelType.Folder);
        expect(deletedItems[0].item_id).toBe(folder1.id);
        cleanup();
    });
});
//# sourceMappingURL=ShareService.test.js.map