"use strict";
// To create a sync target snapshot for the current syncVersion:
// - In test-utils, set syncTargetName_ to "filesystem"
// - Then run:
// node tests/support/createSyncTargetSnapshot.js normal && node tests/support/createSyncTargetSnapshot.js e2ee
//
// These tests work by a taking a sync target snapshot at a version n and upgrading it to n+1.
Object.defineProperty(exports, "__esModule", { value: true });
const LockHandler_1 = require("./LockHandler");
const MigrationHandler_1 = require("./MigrationHandler");
const types_1 = require("./utils/types");
const test_utils_1 = require("../../testing/test-utils");
const syncTargetUtils_1 = require("../../testing/syncTargetUtils");
const Setting_1 = require("../../models/Setting");
const MasterKey_1 = require("../../models/MasterKey");
const utils_1 = require("../e2ee/utils");
const syncInfoUtils_1 = require("./syncInfoUtils");
const specTimeout = 60000 * 10; // Nextcloud tests can be slow
let lockHandler_ = null;
let migrationHandler_ = null;
function lockHandler() {
    if (lockHandler_)
        return lockHandler_;
    lockHandler_ = new LockHandler_1.default((0, test_utils_1.fileApi)());
    return lockHandler_;
}
function migrationHandler(clientId = 'abcd') {
    if (migrationHandler_)
        return migrationHandler_;
    migrationHandler_ = new MigrationHandler_1.default((0, test_utils_1.fileApi)(), (0, test_utils_1.db)(), lockHandler(), LockHandler_1.LockClientType.Desktop, clientId);
    return migrationHandler_;
}
const migrationTests = {
    2: async function () {
        const items = (await (0, test_utils_1.fileApi)().list('', { includeHidden: true })).items;
        // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
        expect(items.filter((i) => i.path === '.resource' && i.isDir).length).toBe(1);
        // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
        expect(items.filter((i) => i.path === 'locks' && i.isDir).length).toBe(1);
        // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
        expect(items.filter((i) => i.path === 'temp' && i.isDir).length).toBe(1);
        // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
        expect(items.filter((i) => i.path === 'info.json' && !i.isDir).length).toBe(1);
        const versionForOldClients = await (0, test_utils_1.fileApi)().get('.sync/version.txt');
        expect(versionForOldClients).toBe('2');
    },
    3: async function () {
        const items = (await (0, test_utils_1.fileApi)().list('', { includeHidden: true })).items;
        // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
        expect(items.filter((i) => i.path === '.resource' && i.isDir).length).toBe(1);
        // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
        expect(items.filter((i) => i.path === 'locks' && i.isDir).length).toBe(1);
        // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
        expect(items.filter((i) => i.path === 'temp' && i.isDir).length).toBe(1);
        // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
        expect(items.filter((i) => i.path === 'info.json' && !i.isDir).length).toBe(1);
        const versionForOldClients = await (0, test_utils_1.fileApi)().get('.sync/version.txt');
        expect(versionForOldClients).toBe('2');
    },
};
const maxSyncVersion = Number(Object.keys(migrationTests).sort().pop());
async function testMigration(migrationVersion, maxSyncVersion) {
    await (0, syncTargetUtils_1.deploySyncTargetSnapshot)('normal', migrationVersion - 1);
    const info = await (0, syncInfoUtils_1.fetchSyncInfo)((0, test_utils_1.fileApi)());
    expect(info.version).toBe(migrationVersion - 1);
    // Now, migrate to the new version
    Setting_1.default.setConstant('syncVersion', migrationVersion);
    await migrationHandler().upgrade(migrationVersion);
    // Verify that it has been upgraded
    const newInfo = await (0, syncInfoUtils_1.fetchSyncInfo)((0, test_utils_1.fileApi)());
    expect(newInfo.version).toBe(migrationVersion);
    await migrationTests[migrationVersion]();
    // If we're not on the latest version, we exit here, because although the
    // synchronizer can run the migration from one version to another, it cannot
    // sync the data on an older version (since the code has been changed to
    // work with the latest version).
    if (migrationVersion !== maxSyncVersion)
        return;
    // Now sync with that upgraded target
    await (0, test_utils_1.synchronizer)().start();
    // Check that the data has not been altered
    await (0, test_utils_1.expectNotThrow)(async () => await (0, syncTargetUtils_1.checkTestData)(syncTargetUtils_1.testData));
    // Check what happens if we switch to a different client and sync
    await (0, test_utils_1.switchClient)(2);
    Setting_1.default.setConstant('syncVersion', migrationVersion);
    await (0, test_utils_1.synchronizer)().start();
    await (0, test_utils_1.expectNotThrow)(async () => await (0, syncTargetUtils_1.checkTestData)(syncTargetUtils_1.testData));
}
async function testMigrationE2EE(migrationVersion, maxSyncVersion) {
    // First create some test data that will be used to validate
    // that the migration didn't alter any data.
    await (0, syncTargetUtils_1.deploySyncTargetSnapshot)('e2ee', migrationVersion - 1);
    // Now, migrate to the new version
    Setting_1.default.setConstant('syncVersion', migrationVersion);
    await migrationHandler().upgrade(migrationVersion);
    // Verify that it has been upgraded
    const newInfo = await (0, syncInfoUtils_1.fetchSyncInfo)((0, test_utils_1.fileApi)());
    expect(newInfo.version).toBe(migrationVersion);
    await migrationTests[migrationVersion]();
    if (migrationVersion !== maxSyncVersion)
        return;
    // Now sync with that upgraded target
    await (0, test_utils_1.synchronizer)().start();
    // Decrypt the data
    const masterKey = (await MasterKey_1.default.all())[0];
    Setting_1.default.setObjectValue('encryption.passwordCache', masterKey.id, '123456');
    await (0, utils_1.loadMasterKeysFromSettings)((0, test_utils_1.encryptionService)());
    await (0, test_utils_1.decryptionWorker)().start();
    // Check that the data has not been altered
    await (0, test_utils_1.expectNotThrow)(async () => await (0, syncTargetUtils_1.checkTestData)(syncTargetUtils_1.testData));
    // Check what happens if we switch to a different client and sync
    await (0, test_utils_1.switchClient)(2);
    Setting_1.default.setConstant('syncVersion', migrationVersion);
    await (0, test_utils_1.synchronizer)().start();
    // Should throw because data hasn't been decrypted yet
    await (0, test_utils_1.expectThrow)(async () => await (0, syncTargetUtils_1.checkTestData)(syncTargetUtils_1.testData));
    // Enable E2EE and decrypt
    Setting_1.default.setObjectValue('encryption.passwordCache', masterKey.id, '123456');
    await (0, utils_1.loadMasterKeysFromSettings)((0, test_utils_1.encryptionService)());
    await (0, test_utils_1.decryptionWorker)().start();
    // Should not throw because data is decrypted
    await (0, test_utils_1.expectNotThrow)(async () => await (0, syncTargetUtils_1.checkTestData)(syncTargetUtils_1.testData));
}
let previousSyncTargetName = '';
describe('MigrationHandler', () => {
    beforeEach(async () => {
        // Note that, for undocumented reasons, the timeout argument passed
        // to `test()` (or `it()`) is ignored if it is higher than the
        // global Jest timeout. So we need to set it globally.
        //
        // https://github.com/facebook/jest/issues/5055#issuecomment-513585906
        jest.setTimeout(specTimeout);
        // To test the migrations, we have to use the filesystem sync target
        // because the sync target snapshots are plain files. Eventually
        // it should be possible to copy a filesystem target to memory
        // but for now that will do.
        previousSyncTargetName = (0, test_utils_1.setSyncTargetName)('filesystem');
        lockHandler_ = null;
        migrationHandler_ = null;
        await (0, test_utils_1.setupDatabaseAndSynchronizer)(1);
        await (0, test_utils_1.setupDatabaseAndSynchronizer)(2);
        await (0, test_utils_1.switchClient)(1);
    });
    afterEach(async () => {
        (0, test_utils_1.setSyncTargetName)(previousSyncTargetName);
    });
    it('should init a new sync target', (async () => {
        // Check that basic folders "locks" and "temp" are created for new sync targets.
        await migrationHandler().upgrade(1);
        const result = await (0, test_utils_1.fileApi)().list();
        // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
        expect(result.items.filter((i) => i.path === types_1.Dirnames.Locks).length).toBe(1);
        // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
        expect(result.items.filter((i) => i.path === types_1.Dirnames.Temp).length).toBe(1);
    }), specTimeout);
    it('should not allow syncing if the sync target is out-dated', (async () => {
        await (0, test_utils_1.synchronizer)().start();
        await (0, test_utils_1.fileApi)().put('info.json', `{"version":${Setting_1.default.value('syncVersion') - 1}}`);
        await (0, test_utils_1.expectThrow)(async () => await migrationHandler().checkCanSync(), 'outdatedSyncTarget');
    }), specTimeout);
    it('should not allow syncing if the client is out-dated', (async () => {
        await (0, test_utils_1.synchronizer)().start();
        await (0, test_utils_1.fileApi)().put('info.json', `{"version":${Setting_1.default.value('syncVersion') + 1}}`);
        await (0, test_utils_1.expectThrow)(async () => await migrationHandler().checkCanSync(), 'outdatedClient');
    }), specTimeout);
    it('should apply migration 2 normal', async () => {
        await testMigration(2, maxSyncVersion);
    }, specTimeout);
    it('should apply migration 2 E2EE', async () => {
        await testMigrationE2EE(2, maxSyncVersion);
    }, specTimeout);
    it('should apply migration 3 normal', async () => {
        await testMigration(3, maxSyncVersion);
    }, specTimeout);
    it('should apply migration 3 E2EE', async () => {
        await testMigrationE2EE(3, maxSyncVersion);
    }, specTimeout);
});
//# sourceMappingURL=synchronizer_MigrationHandler.test.js.map