use crate::page::Renderer;
use color_eyre::Result;
use color_eyre::eyre::ContextCompat;
use parser::contents::EmbeddedFile;
use parser::property::embedded_file::FileType;
use parser_utils::{fs_driver, log};
use std::path::PathBuf;

impl<'a> Renderer<'a> {
    pub(crate) fn render_embedded_file(&mut self, file: &EmbeddedFile) -> Result<String> {
        let content;

        let filename = self.determine_filename(file.filename())?;
        let path = fs_driver().join(self.output.as_str(), filename.as_str());
        log!("Rendering embedded file: {:?}", path);
        fs_driver().write_file(&path, file.data())?;

        let file_type = Self::guess_type(file);

        match file_type {
            // TODO: we still don't have support for the audio tag on html notes https://github.com/laurent22/joplin/issues/11939
            // FileType::Audio => content = format!("<audio class=\"media-player media-audio\"controls><source src=\"{}\" type=\"audio/x-wav\"></source></audio>", filename),
            FileType::Video => content = format!("<video controls src=\"{}\"></video>", filename),
            FileType::Unknown | FileType::Audio => {
                content = format!(
                    "<p style=\"font-size: 11pt; line-height: 17px;\"><a href=\"{}\">{}</a></p>",
                    filename, filename
                )
            }
        };

        Ok(self.render_with_note_tags(file.note_tags(), content))
    }

    fn guess_type(file: &EmbeddedFile) -> FileType {
        match file.file_type() {
            FileType::Audio => return FileType::Audio,
            FileType::Video => return FileType::Video,
            _ => {}
        };

        let filename = file.filename();

        if let Some(mime) = mime_guess::from_path(filename).first() {
            if mime.type_() == "audio" {
                return FileType::Audio;
            }

            if mime.type_() == "video" {
                return FileType::Video;
            }
        }
        FileType::Unknown
    }

    pub(crate) fn determine_filename(&mut self, filename: &str) -> Result<String> {
        let mut i = 0;
        let mut current_filename = filename.to_string();

        loop {
            if !self.section.files.contains(&current_filename) {
                self.section.files.insert(current_filename.clone());

                return Ok(current_filename);
            }

            let path = PathBuf::from(filename);
            let ext = path.extension().unwrap_or_default();
            let base = path
                .as_os_str()
                .to_str()
                .wrap_err("Embedded file name is non utf-8")?
                .strip_suffix(ext.to_string_lossy().as_ref())
                .wrap_err("Failed to strip extension from file name")?
                .trim_matches('.');

            current_filename = format!("{}-{}.{}", base, i, ext.to_string_lossy());

            i += 1;
        }
    }
}
